﻿using System.Collections.Generic;
using System.Linq;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.ConsentSchemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using DataSchemas = HIPS.PcehrDataStore.Schemas;
using ServiceSchemas = HIPS.PatientSchemas;

namespace HIPS.PatientBusinessLogic
{
    public class PatientListLogic
    {
        /// <summary>
        /// Lists admitted or recently discharged patients in hospital with or without a PCEHR.
        /// </summary>
        /// <param name="user">The user who is requesting the information.</param>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="withPcehr">True to list patients with a PCEHR, false to list patients without a PCEHR. Null to list all patients.</param>
        /// <param name="daysDischarged">Number of days after discharge that patients are included in the list. Set to 0 to list only currently admitted patients.</param>
        /// <returns>A patient list response with status and a list of patients in hospital.</returns>
        public ServiceSchemas.PatientListResponse ListPatientsInHospital(UserDetails user, string hospitalCodeSystem, string hospitalCode, bool? withPcehr, int daysDischarged)
        {
            ServiceSchemas.PatientListResponse response = new ServiceSchemas.PatientListResponse();
            PatientListDl dataAccess = new PatientListDl(user);
            int? hospitalId = GetHospitalId(hospitalCodeSystem, hospitalCode, response);
            if (response.HipsResponse.Status == HipsResponseIndicator.InvalidHospital)
            {
                return response;
            }
            List<DataSchemas.PatientInHospital> data;
            response.HipsResponse = dataAccess.ListPatientsInHospital(hospitalCodeSystem, hospitalId, withPcehr, daysDischarged, out data);
            response.PatientInHospitalList = data.Select(a => MapDataToService(a)).ToList();
            return response;
        }

        /// <summary>
        /// Resolves the hospital code (if supplied) to a hospital ID. If no hospital code is supplied, returns null.
        /// If an invalid hospital code is supplied, returns null and sets the response status to InvalidHospital.
        /// </summary>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="response">The HIPS response.</param>
        /// <returns>The hospital ID.</returns>
        private static int? GetHospitalId(string hospitalCodeSystem, string hospitalCode, ServiceSchemas.PatientListResponse response)
        {
            int? hospitalId;
            if (string.IsNullOrEmpty(hospitalCode))
            {
                hospitalId = null;
            }
            else
            {
                DataSchemas.Hospital hospital = HospitalSingleton.Value.Find(hospitalCode, hospitalCodeSystem);
                if (hospital == null)
                {
                    hospitalId = null;
                    response.HipsResponse.Status = HipsResponseIndicator.InvalidHospital;
                    response.HipsResponse.HipsErrorMessage = ResponseStrings.HospitalNotFound;
                }
                else
                {
                    hospitalId = hospital.HospitalId.Value;
                }
            }
            return hospitalId;
        }

        /// <summary>
        /// Maps the database schema to the web service schema for a patient in hospital.
        /// </summary>
        /// <param name="data">The database schema object.</param>
        /// <returns>A web service schema object.</returns>
        private ServiceSchemas.PatientInHospital MapDataToService(DataSchemas.PatientInHospital data)
        {
            ServiceSchemas.PatientInHospital info = new ServiceSchemas.PatientInHospital();
            info.Ihi = data.Ihi;
            info.IhiStatus = (IhiStatus)data.IhiStatusId;
            info.IhiRecordStatus = (IhiRecordStatus)data.IhiRecordStatusId;
            info.IhiLastValidated = data.IhiLastValidated;
            info.DateOfBirth = data.DateOfBirth;
            info.MedicareNumber = data.MedicareNumber;
            info.MedicareIrn = data.MedicareIrn;
            info.DvaNumber = data.DvaNumber;
            info.CurrentSexId = data.CurrentSexId;
            info.CurrentSex = (SexEnumerator)data.CurrentSexId;
            info.FamilyName = data.FamilyName;
            info.GivenNames = data.GivenNames;
            info.RegisteredSex = (SexEnumerator)data.RegisteredSexId;
            info.RegisteredFamilyName = data.RegisteredFamilyName;
            info.RegisteredGivenName = data.RegisteredGivenName;
            info.Suffix = data.Suffix;
            info.Title = data.Title;
            info.HospitalId = data.HospitalId;
            info.HospitalCode = data.HospitalCode;
            info.HospitalName = data.HospitalName;
            info.Mrn = data.Mrn;
            info.Ward = data.Ward;
            info.Room = data.Room;
            info.Bed = data.Bed;
            info.AdmissionDate = data.AdmissionDate;
            info.DischargeDate = data.DischargeDate;
            info.ResponsibleProviderFamilyName = data.ResponsibleProviderFamilyName;
            info.ResponsibleProviderGivenNames = data.ResponsibleProviderGivenNames;
            info.ResponsibleProviderTitle = data.ResponsibleProviderTitle;
            info.ResponsibleProviderSuffix = data.ResponsibleProviderSuffix;
            if (!string.IsNullOrEmpty(data.Ihi) && info.IhiStatus == IhiStatus.Active)
            {
                if (data.PcehrAdvertised.HasValue && data.PcehrAdvertised.Value)
                {
                    info.ParticipationStatus = ParticipationStatus.PcehrAdvertised;
                }
                else if (data.PcehrDisclosed.HasValue && data.PcehrDisclosed.Value)
                {
                    info.ParticipationStatus = ParticipationStatus.RequestedUpload;
                }
                else
                {
                    info.ParticipationStatus = ParticipationStatus.PcehrNotAdvertised;
                }
            }
            else
            {
                info.ParticipationStatus = ParticipationStatus.NoValidIhi;
            }
            return info;
        }
    }
}
