﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the ClinicalDocument table
    /// </summary>
    public class ClinicalDocumentDl : BaseDl
    {
        #region Constructors

        public ClinicalDocumentDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Gets the clinical documents for the specified episode and optionally filter by document type.
        /// </summary>
        /// <param name="episodeId">The episode id.</param>
        /// <param name="documentTypeId">The document type</param>
        /// <returns>List of clinical documents</returns>
        public List<ClinicalDocument> GetAll(int episodeId, int? documentTypeId)
        {
            List<ClinicalDocument> result = new List<ClinicalDocument>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentGet"))
                {
                    AddIntParameter("EpisodeId", episodeId, command);
                    AddIntParameter("DocumentTypeId", documentTypeId, command);
                    result = GetPopulatedBusinessList<ClinicalDocument>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageClinicalDocumentGetAll, episodeId, documentTypeId), ex, User, LogMessage.HIPS_MESSAGE_036);
            }
            return result;
        }

        /// <summary>
        /// Saves the specified clinical document.
        /// </summary>
        /// <param name="ClinicalDocument">The ClinicalDocument.</param>
        /// <returns></returns>
        public bool Save(ClinicalDocument ClinicalDocument, SqlTransaction transaction = null)
        {
            if (ClinicalDocument.ClinicalDocumentId.HasValue && ClinicalDocument.ClinicalDocumentId > -1)
            {
                return Update(ClinicalDocument, transaction);
            }
            return Insert(ClinicalDocument, transaction);
        }

        /// <summary>
        /// Inserts the specified ClinicalDocument.
        /// </summary>
        /// <param name="ClinicalDocument">The ClinicalDocument.</param>
        /// <returns></returns>
        private bool Insert(ClinicalDocument ClinicalDocument, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentInsert", transaction))
            {
                return base.Insert<ClinicalDocument>(ClinicalDocument, command);
            }
        }

        /// <summary>
        /// Updates the specified ClinicalDocument.
        /// </summary>
        /// <param name="ClinicalDocument">The ClinicalDocument.</param>
        /// <returns></returns>
        private bool Update(ClinicalDocument ClinicalDocument, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentUpdate", transaction))
            {
                return base.Update<ClinicalDocument>(ClinicalDocument, command);
            }
        }

        /// <summary>
        /// Gets the clinical document with the specified episode ID and source system set ID.
        /// </summary>
        /// <param name="episodeId">The episode ID.</param>
        /// <param name="sourceSystemSetId">The source system set ID</param>
        /// <returns>The clinical document</returns>
        public ClinicalDocument Get(int episodeId, string sourceSystemSetId)
        {
            ClinicalDocument result = new ClinicalDocument();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentGet"))
                {
                    command.Parameters.Add(new SqlParameter("@EpisodeId", episodeId));
                    command.Parameters.Add(new SqlParameter("@SourceSystemSetId", sourceSystemSetId));
                    PopulateBusinessObject<ClinicalDocument>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageClinicalDocumentGet, sourceSystemSetId, episodeId), ex, User, LogMessage.HIPS_MESSAGE_037);
            }
            return result;
        }

        /// <summary>
        /// Gets the clinical document with the specified source system set ID.
        /// Reminder: ensure the document is attached to the right episode / patient.
        /// </summary>
        /// <param name="sourceSystemSetId">The source system set ID</param>
        /// <returns>The clinical document</returns>
        public ClinicalDocument GetBySetId(string sourceSystemSetId)
        {
            ClinicalDocument result = new ClinicalDocument();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.ClinicalDocumentGet"))
                {
                    command.Parameters.Add(new SqlParameter("@SourceSystemSetId", sourceSystemSetId));
                    PopulateBusinessObject<ClinicalDocument>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageClinicalDocumentGetBySetId, sourceSystemSetId), ex, User, LogMessage.HIPS_MESSAGE_037);
            }
            return result;
        }

        #endregion Methods
    }
}