﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the episode table
    /// </summary>
    public class EpisodeDl : BaseDl
    {
        #region Constructors

        public EpisodeDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Gets all.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="lastUpdate">The last update.</param>
        /// <param name="blockCancelledEpisodeLifeCycles">If set to <c>true</c> [block cancelled episode life cycles] Set tpo false by default.</param>
        /// <returns></returns>
        public List<Episode> GetAll(int? patientId, DateTime? lastUpdate, bool blockCancelledEpisodeLifeCycles = false)
        {
            List<Episode> episodes = new List<Episode>();
            List<HealthProviderIndividual> providers = new List<HealthProviderIndividual>();

            try
            {
                using (SqlCommand command = GetSqlCommand("hips.EpisodeGet"))
                {
                    AddIntParameter("@PatientId", patientId, command);
                    AddDateParameter("@DateLastUpdated", lastUpdate, command);
                    AddBoolParameter("@BlockCancelledEpisodeLifeCycles", blockCancelledEpisodeLifeCycles, command);

                    episodes = GetPopulatedBusinessList<Episode>(command.ExecuteReader());
                    command.Connection.Close();
                }

                using (SqlCommand command = GetSqlCommand("hips.EpisodeProviderGet"))
                {
                    AddIntParameter("@PatientId", patientId, command);
                    AddDateParameter("@DateLastUpdated", lastUpdate, command);

                    providers = GetPopulatedBusinessList<HealthProviderIndividual>(command.ExecuteReader());
                    command.Connection.Close();
                }

                // Assign the proper responsible providers to each episode
                foreach (Episode episode in episodes)
                {
                    episode.ResponsibleProvider = providers.Where(a => a.HealthProviderIndividualId.Value == episode.ResponsibleProviderId).FirstOrDefault();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(string.Format(ConstantsResource.ErrorMessageEpisodeGet, patientId), ex, User, LogMessage.HIPS_MESSAGE_051);
            }
            return episodes;
        }

        /// <summary>
        /// Updates the specified episode.
        /// </summary>
        /// <param name="episode">The episode.</param>
        /// <returns></returns>
        public bool Update(Episode episode, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.EpisodeUpdate", transaction))
            {
                return base.Update<Episode>(episode, command);
            }
        }

        /// <summary>
        /// Inserts the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Insert(Episode item, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.EpisodeInsert", transaction))
            {
                return base.Insert<Episode>(item, command);
            }
        }

        /// <summary>
        /// Deletes the specified item.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public bool Delete(Episode item, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.EpisodeDelete", transaction))
            {
                return base.Delete<Episode>(item, command);
            }
        }

        #endregion Methods
    }
}