﻿using System;
using System.Runtime.Serialization;
using HIPS.Base.Schemas;
using BaseEnumerators = HIPS.Base.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.Schemas
{
    /// <summary>
    /// This class represents a Patient Master IHI link
    /// </summary>
    [KnownType(typeof(PatientMasterIhi))]
    [Serializable]
    [DataContract]
    public class PatientMasterIhi : BaseSchema
    {
        #region Private Members

        private string request;
        private string response;

        #endregion Private Members

        /// <summary>
        /// Gets or sets the patient master ID.
        /// </summary>
        /// <value>
        /// The patient master ID.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ApplicationGeneratedKey)]
        public int? PatientMasterId
        {
            get
            {
                return base.Id;
            }
            set
            {
                base.Id = value;
            }
        }

        /// <summary>
        /// Gets or sets the IHI.
        /// </summary>
        /// <value>
        /// The ihi.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string Ihi { get; set; }

        /// <summary>
        /// Gets or sets the Usable IHI.
        /// As the system may need to store an IHI that cannot be used. The data access layer will perform
        /// the logic to populate this.
        /// </summary>
        /// <value>
        /// The returned ihi.
        /// </value>
        [DataMember]
        public string UsableIhi { get; set; }

        /// <summary>
        /// Gets or sets the ihi status id.
        /// </summary>
        /// <value>
        /// The ihi status id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int IhiStatusId { get; set; }

        /// <summary>
        /// Gets or sets the ihi record status id.
        /// </summary>
        /// <value>
        /// The ihi record status id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int IhiRecordStatusId { get; set; }

        /// <summary>
        /// Gets or sets the name of the registered family.
        /// </summary>
        /// <value>
        /// The name of the registered family.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string RegisteredFamilyName { get; set; }

        /// <summary>
        /// Gets or sets the registered first names.
        /// </summary>
        /// <value>
        /// The registered first names.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string RegisteredGivenName { get; set; }

        /// <summary>
        /// Gets or sets the registered sex ID.
        /// </summary>
        /// <value>
        /// The registered sex ID.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int RegisteredSexId { get; set; }

        /// <summary>
        /// Gets or sets the date last validated.
        /// </summary>
        /// <value>
        /// The date last validated.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public DateTime DateLastValidated { get; set; }

        #region Properties for that the data access layer needs for errors and auditing

        [DataMember]
        public string Request
        {
            get
            {
                return this.request;
            }
            set
            {
                if (!string.IsNullOrEmpty(value))
                {
                    this.request = value.Replace(@"encoding=""UTF-8""", @"encoding=""UTF-16""");
                }
                else
                {
                    this.request = value;
                }
            }
        }

        /// <summary>
        /// Gets or sets the response.
        /// </summary>
        /// <value>
        /// The response.
        /// </value>
        [DataMember]
        public string Response
        {
            get
            {
                return this.response;
            }
            set
            {
                if (!string.IsNullOrEmpty(value))
                {
                    this.response = value.Replace(@"encoding=""UTF-8""", @"encoding=""UTF-16""");
                }
                else
                {
                    this.response = value;
                }
            }
        }

        #region Properties Used to check that the Patient does not exist within the system already

        /// <summary>
        /// Gets or sets the hospital ID. If the hospital ID is specified then
        /// the duplicate IHI and duplicate patient alerts are created only
        /// where duplicates exist within the same hospital.
        /// </summary>
        /// <value>
        /// The hospital ID.
        /// </value>
        [DataMember]
        public int? HospitalId { get; set; }

        /// <summary>
        /// Gets or sets the medicare number.
        /// </summary>
        /// <value>
        /// The medicare number.
        /// </value>
        [DataMember]
        public string MedicareNumber { get; set; }

        /// <summary>
        /// Gets or sets the medicare number sequence.
        /// </summary>
        /// <value>
        /// The medicare number sequence.
        /// </value>
        [DataMember]
        public string MedicareNumberSequence { get; set; }

        /// <summary>
        /// Gets or sets the dva number.
        /// </summary>
        /// <value>
        /// The dva number.
        /// </value>
        [DataMember]
        public string DvaNumber { get; set; }

        /// <summary>
        /// Gets or sets the date of birth.
        /// </summary>
        /// <value>
        /// The date of birth.
        /// </value>
        [DataMember]
        public DateTime DateOfBirth { get; set; }

        #endregion Properties Used to check that the Patient does not exist within the system already

        #endregion Properties for that the data access layer needs for errors and auditing
    }
}