﻿-- =============================================
-- This script upgrades Release 1.1 to Release 1.1a with the
-- Queue Status Pending/Success/Failure in HL7MessageLog.
-- =============================================

-- Create the QueueStatus table
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
SET ANSI_PADDING ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[hips].[QueueStatus]') AND type in (N'U'))
BEGIN
CREATE TABLE [hips].[QueueStatus](
	[QueueStatusId] [int] NOT NULL,
	[Description] [varchar](20) NOT NULL,
	[DateCreated] [datetime] NOT NULL,
	[UserCreated] [varchar](256) NOT NULL,
	[DateModified] [datetime] NOT NULL,
	[UserModified] [varchar](256) NOT NULL,
 CONSTRAINT [PK_QueueStatus] PRIMARY KEY CLUSTERED 
(
	[QueueStatusId] ASC
)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
) ON [PRIMARY]
END
GO

-- Populate the values in the QueueStatus table

IF NOT EXISTS (SELECT QueueStatusId FROM hips.QueueStatus WHERE QueueStatusId = 1)
BEGIN
	INSERT hips.QueueStatus (QueueStatusId, Description, DateCreated, UserCreated, DateModified, UserModified)
	VALUES (1, 'Pending', GETDATE(), 'HIPS', GETDATE(), 'HIPS')
END
IF NOT EXISTS (SELECT QueueStatusId FROM hips.QueueStatus WHERE QueueStatusId = 2)
BEGIN
	INSERT hips.QueueStatus (QueueStatusId, Description, DateCreated, UserCreated, DateModified, UserModified)
	VALUES (2, 'Success', GETDATE(), 'HIPS', GETDATE(), 'HIPS')
END
IF NOT EXISTS (SELECT QueueStatusId FROM hips.QueueStatus WHERE QueueStatusId = 3)
BEGIN
	INSERT hips.QueueStatus (QueueStatusId, Description, DateCreated, UserCreated, DateModified, UserModified)
	VALUES (3, 'Failure', GETDATE(), 'HIPS', GETDATE(), 'HIPS')
END
GO

-- Add the new column in the HL7MessageLog table

IF NOT EXISTS (SELECT * from sys.columns WHERE name = 'QueueStatusId' and object_id = OBJECT_ID(N'[hips].[HL7MessageLog]'))
BEGIN
	ALTER TABLE hips.HL7MessageLog ADD [QueueStatusId] INT NULL
END
GO
IF NOT EXISTS (SELECT * FROM sys.foreign_keys WHERE object_id = OBJECT_ID(N'[hips].[FK_HL7MessageLog_QueueStatus]') AND parent_object_id = OBJECT_ID(N'[hips].[HL7MessageLog]'))
ALTER TABLE [hips].[HL7MessageLog]  WITH CHECK ADD  CONSTRAINT [FK_HL7MessageLog_QueueStatus] FOREIGN KEY([QueueStatusId])
REFERENCES [hips].[QueueStatus] ([QueueStatusId])
GO
IF  EXISTS (SELECT * FROM sys.foreign_keys WHERE object_id = OBJECT_ID(N'[hips].[FK_HL7MessageLog_QueueStatus]') AND parent_object_id = OBJECT_ID(N'[hips].[HL7MessageLog]'))
ALTER TABLE [hips].[HL7MessageLog] CHECK CONSTRAINT [FK_HL7MessageLog_QueueStatus]
GO

-- Populate the values and change to not null before removing the old column

IF EXISTS (SELECT * from sys.columns WHERE name = 'IsSuccessful' and object_id = OBJECT_ID(N'[hips].[HL7MessageLog]'))
BEGIN
	DECLARE @Command NVARCHAR(256);
	SET @Command = N'UPDATE hips.HL7MessageLog SET [QueueStatusId] = CASE [IsSuccessful] WHEN 1 THEN 2 WHEN 0 THEN 3 END';
	EXECUTE (@Command);

	SET @Command = N'ALTER TABLE hips.HL7MessageLog ALTER COLUMN [QueueStatusId] INT NOT NULL';
	EXECUTE (@Command);

	SET @Command = N'ALTER TABLE hips.HL7MessageLog DROP COLUMN [IsSuccessful]';
	EXECUTE (@Command);
END
GO

-- Drop and Create the Stored Procedures that reference this column

DROP PROCEDURE [hips].[HL7MessageLogGet]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Simon Biber
-- Create date: 25 July 2012
-- Description:   Gets a specified record from the HL7Message Table
-- Modified:
-- 16/04/2013 - sbiber - Get only the most recently created record with the criteria
--                       Column IsSuccessful bit changed to QueueStatusId int
-- =============================================
CREATE PROCEDURE [hips].[HL7MessageLogGet] 
(
		 @HL7MessageLogId INT = NULL
		,@SendingApplication		 varchar(180) =NULL
		,@SendingFacility		     varchar(180) = NULL
		,@MessageControlId		     varchar(25) = NULL
		,@MessageTime	DATETIME = NULL      
) AS
BEGIN

SELECT TOP 1
		[HL7MessageLogId]
       ,[HospitalId]
       ,[HospitalPatientIdentifier]
       ,[SendingApplication]
       ,[SendingFacility]
       ,[MessageControlId]
       ,[DateTimeOfMessage]
       ,[QueueStatusId]
       ,[ShouldRetry]
       ,[RetryCount]
       ,[FailureReason]
       ,[FullMessage]
       ,[DateCreated]
       ,[UserCreated]
       ,[DateModified]
       ,[UserModified]
FROM [hips].[HL7MessageLog] 
WHERE (@HL7MessageLogId IS NULL OR HL7MessageLogId = @HL7MessageLogId)
AND (@SendingApplication IS NULL OR SendingApplication = @SendingApplication)
AND (@SendingFacility IS NULL OR SendingFacility = @SendingFacility)
AND (@MessageControlId IS NULL OR MessageControlId = @MessageControlId)
AND (@MessageTime IS NULL OR DateTimeOfMessage = @MessageTime)
ORDER BY DateCreated DESC

END

GO

DROP PROCEDURE [hips].[HL7MessageLogUpdate]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Simon Biber
-- Create date: 22/07/2012
-- Description:   Updates an existing record in the HL7 Message Log Table
-- 16/04/2013 - sbiber - Column IsSuccessful bit changed to QueueStatusId int
-- ============================================
CREATE PROCEDURE [hips].[HL7MessageLogUpdate] 
(
	 @HL7MessageLogId int
	,@HospitalId                 int = NULL
	,@HospitalPatientIdentifier  varchar(20)=NULL
	,@SendingApplication		 varchar(180) =NULL
	,@SendingFacility		     varchar(180) = NULL
	,@MessageControlId		     varchar(25) = NULL
	,@DateTimeOfMessage		     datetime = NULL
	,@QueueStatusId			     int
	,@ShouldRetry			     bit
	,@RetryCount        	     int 
	,@FailureReason			     varchar(256) = NULL
	,@FullMessage			     varchar(max) 
	,@UserModified               varchar(256)                          
    ,@DateModified               datetime
)
AS
BEGIN
    SET NOCOUNT ON
 DECLARE @intErrorCode int
    SET @intErrorCode = 0
    IF NOT EXISTS (SELECT TOP 1 [DateModified]
                FROM [hips].[HL7MessageLog]
        WHERE [HL7MessageLogId] = @HL7MessageLogId
        AND [DateModified] = @DateModified)
    BEGIN
        RAISERROR 50001 'Record has already been updated!'
        SET @intErrorCode = @@ERROR
    END
 IF @intErrorCode = 0
    BEGIN
        UPDATE [hips].[HL7MessageLog]
           SET  [HospitalId]        = @HospitalId               
       ,[HospitalPatientIdentifier]	= @HospitalPatientIdentifier
       ,[SendingApplication]		= @SendingApplication		  
       ,[SendingFacility]			= @SendingFacility		  
       ,[MessageControlId]			= @MessageControlId		  
       ,[DateTimeOfMessage]			= @DateTimeOfMessage		  
       ,[QueueStatusId]				= @QueueStatusId			  
       ,[ShouldRetry]				= @ShouldRetry			  
       ,[RetryCount]				= @RetryCount				  
       ,[FailureReason]				= @FailureReason			  
       ,[FullMessage]				= @FullMessage			  
             , [DateModified] = GetDate()
             , [UserModified] = @UserModified
           WHERE [HL7MessageLogId] = @HL7MessageLogId
        SELECT @intErrorCode = @@ERROR
    END
    IF @intErrorCode = 0
    BEGIN
           SELECT [HL7MessageLogId]
                , [HospitalId]
       ,[HospitalPatientIdentifier]
       ,[SendingApplication]
       ,[SendingFacility]
       ,[MessageControlId]
       ,[DateTimeOfMessage]
       ,[QueueStatusId]
       ,[ShouldRetry]
       ,[RetryCount]
       ,[FailureReason]
       ,[FullMessage]
       ,[DateCreated]
       ,[UserCreated]
       ,[DateModified]
       ,[UserModified]
                     FROM [hips].[HL7MessageLog] 
           WHERE [HL7MessageLogId] = @HL7MessageLogId
    END
    RETURN(@intErrorCode)
END

GO

DROP PROCEDURE [hips].[HL7MessageLogInsert]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Simon
-- Create date: 25 July 2012
-- Description:   Inserts a new record into the HL7 Message Log Table
-- 16/04/2013 - sbiber - Column IsSuccessful bit changed to QueueStatusId int
-- =============================================
CREATE PROCEDURE [hips].[HL7MessageLogInsert] 
(
		@HospitalID INT =NULL
       ,@HospitalPatientIdentifier VARCHAR(20)=NULL
       ,@SendingApplication	VARCHAR(180) =NULL
       ,@SendingFacility VARCHAR(180) = NULL
       ,@MessageControlId VARCHAR(25) = NULL
       ,@DateTimeOfMessage DATETIME = NULL
       ,@QueueStatusId INT 
       ,@ShouldRetry BIT
       ,@RetryCount INT 
       ,@FailureReason VARCHAR(256) = NULL
       ,@FullMessage VARCHAR(MAX) 
       ,@UserModified VARCHAR(256)
)								    
AS								   
BEGIN	
    SET NOCOUNT ON
    DECLARE @intErrorCode int
      ,@HL7MessageLogId INT
    SET @intErrorCode = 0						   
				   
	INSERT INTO [hips].[HL7MessageLog]
	        ( HospitalId ,
	          HospitalPatientIdentifier ,
	          SendingApplication ,
	          SendingFacility ,
	          MessageControlId ,
	          DateTimeOfMessage ,
	          QueueStatusId ,
	          ShouldRetry ,
	          RetryCount ,
	          FailureReason ,
	          FullMessage ,
	          DateCreated ,
	          UserCreated ,
	          DateModified ,
	          UserModified
	        )
VALUES
	(
		@HospitalID
       ,@HospitalPatientIdentifier
       ,@SendingApplication
       ,@SendingFacility
       ,@MessageControlId
       ,@DateTimeOfMessage
       ,@QueueStatusId
       ,@ShouldRetry
       ,@RetryCount
       ,@FailureReason
       ,@FullMessage
      , GETDATE()
      , @UserModified
      , GETDATE()
      , @UserModified
	)
SELECT @intErrorCode = @@ERROR 
           , @HL7MessageLogId = SCOPE_IDENTITY()
    IF @intErrorCode = 0
    BEGIN
        SELECT  [HL7MessageLogID]
			   ,[HospitalID]
			   ,[HospitalPatientIdentifier]
			   ,[SendingApplication]
			   ,[SendingFacility]
			   ,[MessageControlId]
			   ,[DateTimeOfMessage]
			   ,[QueueStatusId]
			   ,[ShouldRetry]
			   ,[RetryCount]
			   ,[FailureReason]
			   ,[FullMessage]
			   ,[DateCreated]
			   ,[UserCreated]
			   ,[DateModified]
			   ,[UserModified]
           FROM [hips].[HL7MessageLog] 
           WHERE [HL7MessageLogId] = @HL7MessageLogId
    END
    RETURN(@intErrorCode)
    
END

GO

-- Grant Permissions on the Changed Procedures

GRANT EXECUTE ON [hips].[HL7MessageLogGet] TO [RunStoredProcedure]
GRANT EXECUTE ON [hips].[HL7MessageLogInsert] TO [RunStoredProcedure]
GRANT EXECUTE ON [hips].[HL7MessageLogUpdate] TO [RunStoredProcedure]
GO