﻿using System;
using System.Collections.Generic;
using System.Net;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel;
using System.ServiceModel.Channels;
using HIPS.CommonBusinessLogic;
using HIPS.CommonSchemas;
using HIPS.Configuration;
using HIPS.HpiiSchemas;
using HIPS.PcehrHiBusinessLogic.Ihi;
using nehta.mcaR50.ProviderBatchSearchForProviderIndividual;
using nehta.mcaR50.ProviderSearchForProviderIndividual;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.HI;

namespace HIPS.PcehrHiBusinessLogic.Hpii
{
    /// <summary>
    /// Performs the HPII Search Queries for TECH.SIS.31
    /// </summary>
    public class HpiiSearch
    {
        #region Private Members

        #endregion Private Members

        #region Properties

        #endregion Properties

        #region Methods

        /// <summary>
        /// Performs an HPII search and validation.
        /// </summary>
        /// <param name="hpiiIdentifierQuery">The hpii identifier query.</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        public HpiiQueryResponse HpiiValidation(HpiiIdentifierQuery hpiiIdentifierQuery, UserDetails user)
        {
            HpiiQueryResponse response = new HpiiQueryResponse();

            ValidateQueryData(hpiiIdentifierQuery, response);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return response;
            }

            X509Certificate2 certificate = HpiiHelper.GetMedicareCertificate(hpiiIdentifierQuery.HpioNumber);
            X509Certificate2 signingCert = certificate;
            Uri uri = HpiiHelper.HiServiceUrl();
            ProductType product = HpiiHelper.GetProduct();

            if (!User.PopulateAndValidateUserFromHpio(hpiiIdentifierQuery.HpioNumber, user))
            {
                throw new Exception(ConstantsResource.InvalidUserDetails);
            }
            if (!User.ValidateUser(user))
            {
                throw new Exception(ConstantsResource.MissingUserValues);
            }

            QualifiedId userQualifiedId = HpiiHelper.GetUserQualifiedId(user);
            QualifiedId hpio = HpiiHelper.GetHpioQualifiedId(user, hpiiIdentifierQuery.HpioNumber);

            // Instantiate the client
            ProviderSearchForProviderIndividualClient providerIndividualClient = new ProviderSearchForProviderIndividualClient(
                uri,
                product,
                userQualifiedId,
                hpio,
                signingCert,
                certificate);

            try
            {
                System.Reflection.FieldInfo field = providerIndividualClient.GetType().GetField("providerSearchForProviderIndividualClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                ProviderSearchForProviderIndividualPortTypeClient port = field.GetValue(providerIndividualClient) as ProviderSearchForProviderIndividualPortTypeClient;
                System.ServiceModel.Channels.CustomBinding binding = port.Endpoint.Binding as System.ServiceModel.Channels.CustomBinding;
                System.ServiceModel.Channels.HttpsTransportBindingElement https = binding.Elements[1] as System.ServiceModel.Channels.HttpsTransportBindingElement;

                //Set the connection timeout for the ProviderBatchSearchForProviderIndividualClient service
                binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);

                if (Settings.Instance.AvoidProxy)
                {
                    https.UseDefaultWebProxy = false;
                }

                // Add server certificate validation callback
                ServicePointManager.ServerCertificateValidationCallback += HIPS.PcehrHiBusinessLogic.Pcehr.DocumentHelper.ValidateServiceCertificate;

                searchForProviderIndividual request = HpiiHelper.MapHpiiIdentifierRequestData(hpiiIdentifierQuery);
                searchForProviderIndividualResponse svcResponse = providerIndividualClient.ProviderIndividualSearch(request);
                response = HpiiHelper.MapHpiiResponseData(svcResponse.searchForProviderIndividualResult, hpiiIdentifierQuery.HpioNumber);
                response.HipsResponse.Status = HipsResponseIndicator.OK;
            }
            catch (FaultException<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType> fe)
            {
                MessageFault fault = fe.CreateMessageFault();
                if (fault.HasDetail)
                {
                    if (fault.Code.ToString() == "WSE9017")
                    {
                    }
                    else
                    {
                        nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType error = fault.GetDetail<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType>();
                        ServiceMessagesType serviceMessageTypes = new ServiceMessagesType();

                        response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                        response.HipsResponse.HipsErrorMessage = error.highestSeverity.ToString() + ":" + error.serviceMessage[0].ToString();

                        //grab the inner exception if there is one
                        if (fe.InnerException != null)
                        {
                            response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                        }
                        response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                        serviceMessageTypes.highestSeverity = error.highestSeverity;
                        serviceMessageTypes.serviceMessage = error.serviceMessage;
                        response.ServiceMessagesType = serviceMessageTypes;
                    }
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                    response.HipsResponse.HipsErrorMessage = fe.Message;
                    //grab the inner exception if there is one
                    if (fe.InnerException != null)
                    {
                        response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                    }
                    response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                }
            }
            catch (Exception e)
            {
                response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                response.HipsResponse.HipsErrorMessage = e.Message;
                response.HipsResponse.HipsErrorMessage = e.Message;
                //grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    response.HipsResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                response.HipsResponse.ResponseCodeDetails = e.StackTrace;
            }
            finally
            {
                HpiiHelper.InsertAudit(user, providerIndividualClient.SoapMessages, hpiiIdentifierQuery.HpioNumber, hpiiIdentifierQuery: hpiiIdentifierQuery, hpiiQueryResponse: response);
                providerIndividualClient.Dispose();
            }

            return response;
        }

        /// <summary>
        /// Performs an HPII demographic search.
        /// </summary>
        /// <param name="hpiiDemographicQuery">The hpii demographic query.</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        public HpiiQueryResponse HpiiDemographicSearch(HpiiDemographicQuery hpiiDemographicQuery, UserDetails user)
        {
            HpiiQueryResponse response = new HpiiQueryResponse();

            X509Certificate2 certificate = HpiiHelper.GetMedicareCertificate(hpiiDemographicQuery.HpioNumber);
            X509Certificate2 signingCert = certificate;
            Uri uri = HpiiHelper.HiServiceUrl();
            ProductType product = HpiiHelper.GetProduct();

            if (!User.PopulateAndValidateUserFromHpio(hpiiDemographicQuery.HpioNumber, user))
            {
                throw new Exception(ConstantsResource.InvalidUserDetails);
            }
            if (!User.ValidateUser(user))
            {
                throw new Exception(ConstantsResource.MissingUserValues);
            }

            QualifiedId userQualifiedId = HpiiHelper.GetUserQualifiedId(user);
            QualifiedId hpio = HpiiHelper.GetHpioQualifiedId(user, hpiiDemographicQuery.HpioNumber);

            // Instantiate the client
            ProviderSearchForProviderIndividualClient providerIndividualClient = new ProviderSearchForProviderIndividualClient(
                uri,
                product,
                userQualifiedId,
                hpio,
                signingCert,
                certificate);

            try
            {
                System.Reflection.FieldInfo field = providerIndividualClient.GetType().GetField("providerSearchForProviderIndividualClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                ProviderSearchForProviderIndividualPortTypeClient port = field.GetValue(providerIndividualClient) as ProviderSearchForProviderIndividualPortTypeClient;
                System.ServiceModel.Channels.CustomBinding binding = port.Endpoint.Binding as System.ServiceModel.Channels.CustomBinding;
                System.ServiceModel.Channels.HttpsTransportBindingElement https = binding.Elements[1] as System.ServiceModel.Channels.HttpsTransportBindingElement;

                //Set the connection timeout for the ProviderBatchSearchForProviderIndividualClient service
                binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);

                if (Settings.Instance.AvoidProxy)
                {
                    https.UseDefaultWebProxy = false;
                }

                // Add server certificate validation callback
                ServicePointManager.ServerCertificateValidationCallback += HIPS.PcehrHiBusinessLogic.Pcehr.DocumentHelper.ValidateServiceCertificate;

                nehta.mcaR50.ProviderSearchForProviderIndividual.searchForProviderIndividual request = HpiiHelper.MapHpiiDemographicRequestData(hpiiDemographicQuery);
                searchForProviderIndividualResponse svcResponse = providerIndividualClient.ProviderIndividualSearch(request);
                response = HpiiHelper.MapHpiiResponseData(svcResponse.searchForProviderIndividualResult, hpiiDemographicQuery.HpioNumber);
            }
            catch (FaultException<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType> fe)
            {
                MessageFault fault = fe.CreateMessageFault();

                if (fault.HasDetail)
                {
                    if (fault.Code.ToString() == "WSE9017")
                    {
                        //TODO TIDY
                    }
                    else
                    {
                        nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType error = fault.GetDetail<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType>();
                        ServiceMessagesType serviceMessageTypes = new ServiceMessagesType();

                        response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                        response.HipsResponse.HipsErrorMessage = error.highestSeverity.ToString() + ":" + error.serviceMessage[0].ToString();

                        //grab the inner exception if there is one
                        if (fe.InnerException != null)
                        {
                            response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                        }
                        response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                        serviceMessageTypes.highestSeverity = error.highestSeverity;
                        serviceMessageTypes.serviceMessage = error.serviceMessage;
                        response.ServiceMessagesType = serviceMessageTypes;
                    }
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                    response.HipsResponse.HipsErrorMessage = fe.Message;
                    //grab the inner exception if there is one
                    if (fe.InnerException != null)
                    {
                        response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                    }
                    response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                }
            }
            catch (Exception e)
            {
                response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                response.HipsResponse.HipsErrorMessage = e.Message;

                //grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    response.HipsResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                response.HipsResponse.ResponseCodeDetails = e.StackTrace;
            }
            finally
            {
                HpiiHelper.InsertAudit(user, providerIndividualClient.SoapMessages, hpiiDemographicQuery.HpioNumber, hpiiDemographicQuery: hpiiDemographicQuery, hpiiQueryResponse: response);
                providerIndividualClient.Dispose();
            }

            return response;
        }

        /// <summary>
        /// Performs an HPII search and validation.
        /// </summary>
        /// <param name="hpiiBatchAsyncSubmitRequest">The hpii batch asynchronous submit request.</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        /// <exception cref="System.Exception">
        /// </exception>
        public HpiiBatchAsyncSubmitResponse HpiiBatchSubmit(HpiiBatchAsyncSubmitRequest hpiiBatchAsyncSubmitRequest, UserDetails user)
        {
            HpiiBatchAsyncSubmitResponse response = new HpiiBatchAsyncSubmitResponse();

            ValidateQueryBatchData(hpiiBatchAsyncSubmitRequest, response);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return response;
            }

            X509Certificate2 certificate = HpiiHelper.GetMedicareCertificate(hpiiBatchAsyncSubmitRequest.HpioNumber);
            X509Certificate2 signingCert = certificate;
            Uri uri = HpiiHelper.HiServiceUrl();
            ProductType product = HpiiHelper.GetProduct();

            if (!User.PopulateAndValidateUserFromHpio(hpiiBatchAsyncSubmitRequest.HpioNumber, user))
            {
                throw new Exception(ConstantsResource.InvalidUserDetails);
            }
            if (!User.ValidateUser(user))
            {
                throw new Exception(ConstantsResource.MissingUserValues);
            }

            QualifiedId userQualifiedId = HpiiHelper.GetUserQualifiedId(user);
            QualifiedId hpio = HpiiHelper.GetHpioQualifiedId(user, hpiiBatchAsyncSubmitRequest.HpioNumber);

            // Instantiate the client
            ProviderBatchAsyncSearchForProviderIndividualClient providerBatchAsyncIndividualClient = new ProviderBatchAsyncSearchForProviderIndividualClient(
                uri,
                product,
                userQualifiedId,
                hpio,
                signingCert,
                certificate);

            try
            {
                System.Reflection.FieldInfo field = providerBatchAsyncIndividualClient.GetType().GetField("providerBatchAsyncSearchForProviderIndividualClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                SearchForProviderIndividualBatchAsyncPortTypeClient port = field.GetValue(providerBatchAsyncIndividualClient) as SearchForProviderIndividualBatchAsyncPortTypeClient;
                System.ServiceModel.Channels.CustomBinding binding = port.Endpoint.Binding as System.ServiceModel.Channels.CustomBinding;
                System.ServiceModel.Channels.HttpsTransportBindingElement https = binding.Elements[1] as System.ServiceModel.Channels.HttpsTransportBindingElement;

                //Set the connection timeout for the ProviderBatchSearchForProviderIndividualClient service
                binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);

                if (Settings.Instance.AvoidProxy)
                {
                    https.UseDefaultWebProxy = false;
                }

                // Add server certificate validation callback
                ServicePointManager.ServerCertificateValidationCallback += HIPS.PcehrHiBusinessLogic.Pcehr.DocumentHelper.ValidateServiceCertificate;

                //determine the total number of the batch items
                int batchCount = 0;
                if (hpiiBatchAsyncSubmitRequest.HpiiDemographicQueries != null)
                {
                    batchCount = hpiiBatchAsyncSubmitRequest.HpiiDemographicQueries.Count;
                }
                if (hpiiBatchAsyncSubmitRequest.HpiiIdentifierQueries != null)
                {
                    batchCount += hpiiBatchAsyncSubmitRequest.HpiiIdentifierQueries.Count;
                }

                if (batchCount == 0)
                {
                    response.HipsResponse.HipsErrorMessage = "No items to process within the Batch Asynchronous Submit";
                    return response;
                }

                //initialize the batch object
                BatchSearchForProviderIndividualCriteriaType[] requestBatch = new BatchSearchForProviderIndividualCriteriaType[batchCount];

                int itemAddedCount = 0;
                //loop over all items in the batch submission and add to the BatchSearchForProviderIndividualCriteriaType
                foreach (HpiiIdentifierQuery query in hpiiBatchAsyncSubmitRequest.HpiiIdentifierQueries)
                {
                    searchForProviderIndividual request = HpiiHelper.MapHpiiIdentifierRequestData(query);
                    requestBatch[itemAddedCount] = new BatchSearchForProviderIndividualCriteriaType();
                    requestBatch[itemAddedCount].searchForProviderIndividual = request;
                    requestBatch[itemAddedCount].requestIdentifier = (itemAddedCount + 1).ToString();
                    itemAddedCount += 1;
                }

                foreach (HpiiDemographicQuery query in hpiiBatchAsyncSubmitRequest.HpiiDemographicQueries)
                {
                    searchForProviderIndividual request = HpiiHelper.MapHpiiDemographicRequestData(query);
                    requestBatch[itemAddedCount] = new BatchSearchForProviderIndividualCriteriaType();
                    requestBatch[itemAddedCount].searchForProviderIndividual = request;
                    requestBatch[itemAddedCount].requestIdentifier = (itemAddedCount + 1).ToString();
                    itemAddedCount += 1;
                }

                submitSearchForProviderIndividualResponse svcResponse = providerBatchAsyncIndividualClient.BatchSubmitProviderIndividuals(requestBatch);

                if (svcResponse.submitSearchForProviderIndividualResult != null)
                {
                    //add in batch identifier and correct service messages
                    response.BatchIdentifier = svcResponse.submitSearchForProviderIndividualResult.batchIdentifier;
                    response.ServiceMessagesType = svcResponse.submitSearchForProviderIndividualResult.serviceMessages;
                    if (svcResponse.submitSearchForProviderIndividualResult.serviceMessages != null)
                    {
                        response.HipsResponse.ResponseCode = svcResponse.submitSearchForProviderIndividualResult.serviceMessages.serviceMessage[0].code;
                        response.HipsResponse.ResponseCodeDescription = svcResponse.submitSearchForProviderIndividualResult.serviceMessages.serviceMessage[0].reason;
                    }
                    response.HipsResponse.Status = HipsResponseIndicator.OK;
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                    response.HipsResponse.HipsErrorMessage = "Submit Result Missing - Batch Identifier could not be identified";
                    return response;
                }
            }
            catch (FaultException<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType> fe)
            {
                MessageFault fault = fe.CreateMessageFault();
                if (fault.HasDetail)
                {
                    if (fault.Code.ToString() == "WSE9017")
                    {
                    }
                    else
                    {
                        nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType error = fault.GetDetail<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType>();
                        ServiceMessagesType serviceMessageTypes = new ServiceMessagesType();

                        response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                        response.HipsResponse.HipsErrorMessage = error.highestSeverity.ToString() + ":" + error.serviceMessage[0].ToString();

                        //grab the inner exception if there is one
                        if (fe.InnerException != null)
                        {
                            response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                        }
                        response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                        serviceMessageTypes.highestSeverity = error.highestSeverity;
                        serviceMessageTypes.serviceMessage = error.serviceMessage;
                        response.ServiceMessagesType = serviceMessageTypes;
                    }
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                    response.HipsResponse.HipsErrorMessage = fe.Message;
                    //grab the inner exception if there is one
                    if (fe.InnerException != null)
                    {
                        response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                    }
                    response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                }
            }
            catch (Exception e)
            {
                response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                response.HipsResponse.HipsErrorMessage = e.Message;
                response.HipsResponse.HipsErrorMessage = e.Message;
                //grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    response.HipsResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                response.HipsResponse.ResponseCodeDetails = e.StackTrace;
            }
            finally
            {
                HpiiHelper.InsertAudit(user, providerBatchAsyncIndividualClient.SoapMessages, hpiiBatchAsyncSubmitRequest.HpioNumber, hpiiBatchAsyncSubmitResponse: response);
                providerBatchAsyncIndividualClient.Dispose();
            }

            return response;
        }

        /// <summary>
        /// Performs an HPII search and validation.
        /// </summary>
        /// <param name="hpiiIdentifierQuery">The hpii identifier query.</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        public HpiiBatchAsyncRetrieveResponse HpiiBatchRetrieve(HpiiBatchAsyncRetrieveRequest hpiiBatchAsyncRetrieveRequest, UserDetails user)
        {
            HpiiBatchAsyncRetrieveResponse response = new HpiiBatchAsyncRetrieveResponse();

            ValidateQueryData(hpiiBatchAsyncRetrieveRequest, response);
            if (response.HipsResponse.Status != HipsResponseIndicator.OK)
            {
                return response;
            }

            X509Certificate2 certificate = HpiiHelper.GetMedicareCertificate(hpiiBatchAsyncRetrieveRequest.HpioNumber);
            X509Certificate2 signingCert = certificate;
            Uri uri = HpiiHelper.HiServiceUrl();
            ProductType product = HpiiHelper.GetProduct();

            if (!User.PopulateAndValidateUserFromHpio(hpiiBatchAsyncRetrieveRequest.HpioNumber, user))
            {
                throw new Exception(ConstantsResource.InvalidUserDetails);
            }
            if (!User.ValidateUser(user))
            {
                throw new Exception(ConstantsResource.MissingUserValues);
            }

            QualifiedId userQualifiedId = HpiiHelper.GetUserQualifiedId(user);
            QualifiedId hpio = HpiiHelper.GetHpioQualifiedId(user, hpiiBatchAsyncRetrieveRequest.HpioNumber);

            // Instantiate the client
            ProviderBatchAsyncSearchForProviderIndividualClient providerBatchAsyncIndividualClient = new ProviderBatchAsyncSearchForProviderIndividualClient(
                uri,
                product,
                userQualifiedId,
                hpio,
                signingCert,
                certificate);

            try
            {
                System.Reflection.FieldInfo field = providerBatchAsyncIndividualClient.GetType().GetField("providerBatchAsyncSearchForProviderIndividualClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
                SearchForProviderIndividualBatchAsyncPortTypeClient port = field.GetValue(providerBatchAsyncIndividualClient) as SearchForProviderIndividualBatchAsyncPortTypeClient;
                System.ServiceModel.Channels.CustomBinding binding = port.Endpoint.Binding as System.ServiceModel.Channels.CustomBinding;
                System.ServiceModel.Channels.HttpsTransportBindingElement https = binding.Elements[1] as System.ServiceModel.Channels.HttpsTransportBindingElement;

                //Set the connection timeout for the ProviderBatchSearchForProviderIndividualClient service
                binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);
                binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.HpiiSearchTimeoutSeconds);

                if (Settings.Instance.AvoidProxy)
                {
                    https.UseDefaultWebProxy = false;
                }

                // Add server certificate validation callback
                ServicePointManager.ServerCertificateValidationCallback += HIPS.PcehrHiBusinessLogic.Pcehr.DocumentHelper.ValidateServiceCertificate;

                retrieveSearchForProviderIndividual requestBatch = new retrieveSearchForProviderIndividual();
                requestBatch.batchIdentifier = hpiiBatchAsyncRetrieveRequest.BatchIdentifier;

                retrieveSearchForProviderIndividualResponse svcResponse = providerBatchAsyncIndividualClient.BatchRetrieveProviderIndividuals(requestBatch);

                //extract the results
                if (svcResponse.retrieveSearchForProviderIndividualResult != null)
                {
                    //set the upper level batch identifier
                    response.BatchIdentifier = svcResponse.retrieveSearchForProviderIndividualResult.batchIdentifier;
                    response.ServiceMessagesType = svcResponse.retrieveSearchForProviderIndividualResult.serviceMessages;
                    response.HpiiBatchQueryResponses = new List<HpiiBatchQueryResponse>();

                    if (svcResponse.retrieveSearchForProviderIndividualResult.batchSearchForProviderIndividualResult != null && svcResponse.retrieveSearchForProviderIndividualResult.batchSearchForProviderIndividualResult != null)
                    {
                        //loop over all items the batchSearchForProviderIndividualResult to extract each individual result
                        foreach (BatchSearchForProviderIndividualResultType result in svcResponse.retrieveSearchForProviderIndividualResult.batchSearchForProviderIndividualResult)
                        {
                            //map the searchForProviderIndividualResult onto HpiiQueryResponse
                            HpiiBatchQueryResponse hpiiBatchQueryResponse = new HpiiBatchQueryResponse();

                            hpiiBatchQueryResponse.RequestIdentifier = result.requestIdentifier;
                            hpiiBatchQueryResponse.HpiiQueryResponse = HpiiHelper.MapHpiiResponseData(result.searchForProviderIndividualResult, hpiiBatchAsyncRetrieveRequest.HpioNumber);
                            response.HpiiBatchQueryResponses.Add(hpiiBatchQueryResponse);
                        }
                    }
                }

                //response = HpiiHelper.MapHpiiResponseData(svcResponse, hpiiIdentifierQuery.HpioNumber);
                response.HipsResponse.Status = HipsResponseIndicator.OK;
            }
            catch (FaultException<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType> fe)
            {
                MessageFault fault = fe.CreateMessageFault();
                if (fault.HasDetail)
                {
                    if (fault.Code.ToString() == "WSE9017")
                    {
                    }
                    else
                    {
                        nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType error = fault.GetDetail<nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType>();
                        ServiceMessagesType serviceMessageTypes = new ServiceMessagesType();

                        response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                        response.HipsResponse.HipsErrorMessage = error.highestSeverity.ToString() + ":" + error.serviceMessage[0].ToString();

                        //grab the inner exception if there is one
                        if (fe.InnerException != null)
                        {
                            response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                        }
                        response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                        serviceMessageTypes.highestSeverity = error.highestSeverity;
                        serviceMessageTypes.serviceMessage = error.serviceMessage;
                        response.ServiceMessagesType = serviceMessageTypes;
                    }
                }
                else
                {
                    response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                    response.HipsResponse.HipsErrorMessage = fe.Message;
                    //grab the inner exception if there is one
                    if (fe.InnerException != null)
                    {
                        response.HipsResponse.ResponseCodeDescription = fe.InnerException.Message;
                    }
                    response.HipsResponse.ResponseCodeDetails = fe.StackTrace;
                }
            }
            catch (Exception e)
            {
                response.HipsResponse.Status = HipsResponseIndicator.SystemError;
                response.HipsResponse.HipsErrorMessage = e.Message;
                response.HipsResponse.HipsErrorMessage = e.Message;
                //grab the inner exception if there is one
                if (e.InnerException != null)
                {
                    response.HipsResponse.ResponseCodeDescription = e.InnerException.Message;
                }
                response.HipsResponse.ResponseCodeDetails = e.StackTrace;
            }
            finally
            {
                HpiiHelper.InsertAudit(user, providerBatchAsyncIndividualClient.SoapMessages, hpiiBatchAsyncRetrieveRequest.HpioNumber, hpiiBatchAsyncRetrieveResponse: response);
                providerBatchAsyncIndividualClient.Dispose();
            }

            return response;
        }

        /// <summary>
        /// Validates the Identify query data.
        /// </summary>
        private void ValidateQueryData(HpiiIdentifierQuery hpiiIdentifierQuery, HpiiQueryResponse response)
        {
            //HPIO must always be provided
            if (hpiiIdentifierQuery.HpioNumber.IsNullOrEmptyWhitespace())
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.HPIONotProvided;
                return;
            }

            //no validation errors so at this time it is all ok
            response.HipsResponse.Status = HipsResponseIndicator.OK;
        }

        /// <summary>
        /// Validates the demographic query data.
        /// </summary>
        private void ValidateQueryData(HpiiDemographicQuery hpiiDemographicQuery, HpiiQueryResponse response)
        {
            //HPIO must always be provided
            if (hpiiDemographicQuery.HpioNumber.IsNullOrEmptyWhitespace())
            {
                response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.HPIONotProvided;
                return;
            }

            //no validation errors so at this time it is all ok
            response.HipsResponse.Status = HipsResponseIndicator.OK;
        }

        /// <summary>
        /// Validates the retrieve batch query data.
        /// </summary>
        private void ValidateQueryData(HpiiBatchAsyncRetrieveRequest hpiiBatchAsyncRetrieveRequest, HpiiBatchAsyncRetrieveResponse hpiiBatchAsyncRetrieveResponse)
        {
            //HPIO must always be provided
            if (hpiiBatchAsyncRetrieveRequest.HpioNumber.IsNullOrEmptyWhitespace())
            {
                hpiiBatchAsyncRetrieveResponse.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                hpiiBatchAsyncRetrieveResponse.HipsResponse.HipsErrorMessage = HpiiConstantsResource.HPIONotProvided;
                return;
            }

            //no validation errors so at this time it is all ok
            hpiiBatchAsyncRetrieveResponse.HipsResponse.Status = HipsResponseIndicator.OK;
        }

        /// <summary>
        /// Validates the batch query data.
        /// </summary>
        private void ValidateQueryBatchData(HpiiBatchAsyncSubmitRequest hpiiBatchAsyncSubmitRequest, HpiiBatchAsyncSubmitResponse response)
        {
            //loop over all items in the batch object
            foreach (HpiiIdentifierQuery query in hpiiBatchAsyncSubmitRequest.HpiiIdentifierQueries)
            {
                HpiiQueryResponse itemResponse = new HpiiQueryResponse();
                ValidateQueryData(query, itemResponse);
                if (itemResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                    response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.HPIONotProvided;
                    return;
                }
            }

            foreach (HpiiDemographicQuery query in hpiiBatchAsyncSubmitRequest.HpiiDemographicQueries)
            {
                HpiiQueryResponse itemResponse = new HpiiQueryResponse();
                ValidateQueryData(query, itemResponse);
                if (itemResponse.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    response.HipsResponse.Status = HipsResponseIndicator.ValidationError;
                    response.HipsResponse.HipsErrorMessage = HpiiConstantsResource.HPIONotProvided;
                    return;
                }
            }

            //no validation errors so at this time it is all ok
            response.HipsResponse.Status = HipsResponseIndicator.OK;
        }

        #endregion Methods
    }
}