﻿// -----------------------------------------------------------------------
// <copyright file="Participant.cs"  company="NEHTA">
// Developed by Chamonix for NEHTA.
// </copyright>
// -----------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.Serialization;
using System.Xml;
using HIPS.CommonBusinessLogic;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;

namespace HIPS.PcehrHiBusinessLogic.Pcehr
{
    /// <summary>
    /// Queries the status of queued operations (at time of writing, only upload or remove were queued operations) for an episode.
    /// </summary>
    public class OperationQuery
    {
        /// <summary>
        /// Gets the status of all operations for the specified patient and episode.
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier</param>
        /// <param name="admissionDate">The admission date and time</param>
        /// <param name="user">The user</param>
        /// <returns>List of queued operations and indicator of success or failure.</returns>
        public OperationStatus GetOperationStatus(PatientIdentifierBase patientIdentifier, DateTime admissionDate, UserDetails user)
        {
            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            OperationStatus response = new OperationStatus();
            response.Response = patientAccess.GetHospital(patientIdentifier, out hospital);

            if (response.Response.Status != HipsResponseIndicator.OK)
            {
                return response;
            }
            response.Response = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (response.Response.Status != HipsResponseIndicator.OK)
            {
                return response;
            }

            //A Specific Episode that has not been cancelled must be found in this case
            Episode episode = patientAccess.GetEpisodeWithoutCancelled(patientIdentifier, admissionDate, hospitalPatient);
            if (episode == null)
            {
                response.Response.Status = HipsResponseIndicator.InvalidEpisode;
                return response;
            }
            PcehrMessageQueueDl dataAccess = new PcehrMessageQueueDl(user);
            response.QueuedOperations = dataAccess.GetAll(episode.EpisodeId.Value);
            response.UploadedDocuments = patientAccess.DocumentDataAccess.GetAll(episode.EpisodeId.Value, null);
            foreach (ClinicalDocument document in response.UploadedDocuments)
            {
                response.UploadedDocumentVersions.AddRange(patientAccess.VersionDataAccess.GetAllVersions(document.ClinicalDocumentId.Value));
            }
            return response;
        }

        /// <summary>
        /// Gets the status of a single pcehr queue operation for the specified PcehrMessageQueueId
        /// </summary>
        /// <param name="pcehrMessageQueueId">The ID of the PcehrMessageQueue Table</param>
        /// <param name="user">The user</param>
        /// <returns>List of queued operations and indicator of success or failure.</returns>
        public IndividualOperationStatus GetIndividualOperationStatus(int pcehrMessageQueueId, UserDetails user)
        {
            IndividualOperationStatus response = new IndividualOperationStatus();

            //get the PcehrMessageQueue item
            PcehrMessageQueueDl dataAccess = new PcehrMessageQueueDl(user);
            PcehrMessageQueue pcehrMessageQueue = null;
            if (!dataAccess.Get(pcehrMessageQueueId, out pcehrMessageQueue))
            {
                response.ClinicalDocument = null;
                response.QueuedOperation = null;
                response.Response.Status = HipsResponseIndicator.CouldNotFindQueueItem;
                response.Response.HipsErrorMessage += "Could not find requested Pcehr Message Queue item";
                return response;
            }
            response.QueuedOperation = pcehrMessageQueue;

            //populate the ClinicalDocument and ClinicalDocumentVersion
            if (pcehrMessageQueue.QueueOperationId == (int)QueueOperation.UploadOrSupersede)
            {
                QueuedUploadOperation deserializedOperation = new QueuedUploadOperation(user);
                using (MemoryStream stream = new MemoryStream(pcehrMessageQueue.SerialisedObject))
                {
                    deserializedOperation = new DataContractSerializer(typeof(QueuedUploadOperation)).ReadObject(stream) as QueuedUploadOperation;
                }

                //extract the encoded document from the request and decode for the DecodedPackage string
                if (pcehrMessageQueue.Request != null)
                {
                    XmlDocument xmlDoc = new XmlDocument();
                    using (MemoryStream stream = new MemoryStream(System.Text.Encoding.UTF8.GetBytes(pcehrMessageQueue.Request)))
                    {
                        try
                        {
                            xmlDoc.Load(stream);
                            XmlNode docNode = xmlDoc.SelectSingleNode(XmlStringMap.RequestCDAPackageDocument, NamespaceManager());
                            if (docNode.FirstChild.InnerText != null)
                            {
                                response.DecodedPackage = Convert.FromBase64String(docNode.FirstChild.InnerText);
                            }
                        }
                        catch (XmlException ex)
                        {
                            response.Response.Status = HipsResponseIndicator.InvalidDocument;
                            response.Response.HipsErrorMessage += "[Cannot load Pcehr Message Queue Request as an XML document]";
                            response.Response.ResponseCodeDescription = ex.Message;
                        }
                    }
                }

                response.ClinicalDocument.DocumentTypeId = (int)deserializedOperation.DocumentType.DocumentTypeId;
                response.ClinicalDocument.DocumentTypeDescription = deserializedOperation.DocumentType.Description;
                response.ClinicalDocument.EpisodeId = (int)deserializedOperation.Episode.Id;
                response.ClinicalDocument.SourceSystemSetId = deserializedOperation.SourceSystemSetId;
            }
            else if (pcehrMessageQueue.QueueOperationId == (int)QueueOperation.Remove)
            {
                QueuedRemoveOperation deserializedOperation = new QueuedRemoveOperation(user);
                using (MemoryStream stream = new MemoryStream(pcehrMessageQueue.SerialisedObject))
                {
                    deserializedOperation = new DataContractSerializer(typeof(QueuedRemoveOperation)).ReadObject(stream) as QueuedRemoveOperation;
                }

                response.ClinicalDocument.EpisodeId = (int)deserializedOperation.Episode.Id;
                response.ClinicalDocument.SourceSystemSetId = deserializedOperation.SourceSystemSetId;
                response.ClinicalDocument.RemovalReasonDescription = deserializedOperation.AuditInformation.ToString();
            }

            return response;
        }

        /// <summary>
        /// An XmlNamespaceManager that includes the Soap Envelope Namespace (abbreviated to "s")
        /// and XDS-b 2007 Document Registry Namespace (abbreviated to "xdsb2007") namespaces.
        /// </summary>
        private XmlNamespaceManager NamespaceManager()
        {
            XmlNamespaceManager namespaceManagerField = new XmlNamespaceManager(new NameTable());
            namespaceManagerField.AddNamespace(XmlStringMap.SoapEnvelopeNamespaceAbbreviation, XmlStringMap.SoapEnvelopeNamespace);
            namespaceManagerField.AddNamespace(XmlStringMap.XDSb2007DocumentRegistryNamespaceAbbreviation, XmlStringMap.XDSb2007DocumentRegistryNamespace);
            return namespaceManagerField;
        }

        /// <summary>
        /// Gets a List of queued Operations
        /// </summary>
        /// <param name="dateTimeFrom">The date time from.</param>
        /// <param name="dateTimeTo">The date time to.</param>
        /// <param name="queueOperation">The queue operation.</param>
        /// <param name="queueStatus">The queue status.</param>
        /// <param name="patientIdentifier">The patient identifier.</param>
        /// <param name="user">The user</param>
        /// <returns>
        /// List of queued operations and indicator of success or failure.
        /// </returns>
        public QueuedOperationStatus GetQueuedOperationList(DateTime dateTimeFrom, DateTime dateTimeTo, IList<QueueOperation> queueOperation, IList<QueueStatus> queueStatus, IList<PatientIdentifierBase> patientIdentifier, UserDetails user)
        {
            QueuedOperationStatus queuedOperationList = new QueuedOperationStatus();

            //get the PcehrMessageQueue items between dates specified
            PcehrMessageQueueDl dataAccess = new PcehrMessageQueueDl(user);
            queuedOperationList.QueuedOperations = dataAccess.GetQueuedOperations(dateTimeFrom, dateTimeTo);

            //Filter the queueOperation criteria
            if (queueOperation != null && queueOperation.Count > 0)
            {
                queuedOperationList.QueuedOperations = queuedOperationList.QueuedOperations.Where(v => queueOperation.Count
                                                                                                 (a => a.ToString().Equals(v.QueueOperationName)) > 0)
                                                                                           .ToList<MessageQueueItem>();
            }

            //Filter the queueStatus criteria
            if (queueStatus != null && queueStatus.Count > 0)
            {
                queuedOperationList.QueuedOperations = queuedOperationList.QueuedOperations.Where(v => queueStatus.Count
                                                                                                 (a => a.ToString().Equals(v.QueueStatusName)) > 0)
                                                                                           .ToList<MessageQueueItem>();
            }

            //Filter the patient criteria
            if (patientIdentifier != null && patientIdentifier.Count > 0)
            {
                List<int> patientIds = new List<int>();

                foreach (PatientIdentifierBase patient in patientIdentifier)
                {
                    Hospital hospital;
                    HospitalPatient hospitalPatient;
                    PatientMaster patientMaster;
                    PatientAccess patientAccess = new PatientAccess(user);
                    HipsResponse response = patientAccess.GetHospital(patient, out hospital);
                    if (response.Status != HipsResponseIndicator.OK)
                    {
                        queuedOperationList.QueuedOperations = null;
                        queuedOperationList.Response.Status = HipsResponseIndicator.InvalidHospital;
                        queuedOperationList.Response.HipsErrorMessage += "Could not find hospital specified in the Patient Criteria";
                        return queuedOperationList;
                    }

                    patientAccess = new PatientAccess(user);
                    response = patientAccess.GetPatient(patient, hospital, out hospitalPatient, out patientMaster);
                    if (response.Status != HipsResponseIndicator.OK)
                    {
                        queuedOperationList.QueuedOperations = null;
                        queuedOperationList.Response.Status = HipsResponseIndicator.InvalidPatient;
                        queuedOperationList.Response.HipsErrorMessage += "Could not find Patient specified in the Patient Criteria";
                        return queuedOperationList;
                    }

                    patientIds.Add((int)hospitalPatient.Id);
                }

                if (patientIds != null && patientIds.Count() > 0)
                {
                    queuedOperationList.QueuedOperations = queuedOperationList.QueuedOperations.Where(v => patientIds.Count
                                                                                                 (a => a.Equals(v.PatientId)) > 0)
                                                                                                .ToList<MessageQueueItem>();
                }

            }

            foreach (MessageQueueItem messageQueueItem in queuedOperationList.QueuedOperations)
            {
                //populate the ClinicalDocument and ClinicalDocumentVersion if an UploadOrSupersede
                if (messageQueueItem.QueueOperationId == (int)QueueOperation.UploadOrSupersede)
                {
                    QueuedUploadOperation deserializedOperation = new QueuedUploadOperation(user);
                    using (MemoryStream stream = new MemoryStream(messageQueueItem.SerialisedObject))
                    {
                        deserializedOperation = new DataContractSerializer(typeof(QueuedUploadOperation)).ReadObject(stream) as QueuedUploadOperation;
                    }

                    messageQueueItem.DocumentTypeId = (deserializedOperation.DocumentType == null ? null : deserializedOperation.DocumentType.DocumentTypeId);
                    messageQueueItem.DocumentTypeDescription = (deserializedOperation.DocumentType == null ? null : deserializedOperation.DocumentType.Description);
                    messageQueueItem.DocumentTypeCode = (deserializedOperation.DocumentType == null ? null : deserializedOperation.DocumentType.Code);
                    messageQueueItem.DocumentFormatDescription = (deserializedOperation.DocumentFormat == null ? null : deserializedOperation.DocumentFormat.Description);
                    messageQueueItem.DocumentFormatCode = (deserializedOperation.DocumentFormat == null ? null : deserializedOperation.DocumentFormat.Code);
                }
            }

            //If none are found then response will indicate this
            if (queuedOperationList.QueuedOperations.Count == 0)
            {
                queuedOperationList.QueuedOperations = null;
                queuedOperationList.Response.Status = HipsResponseIndicator.CouldNotFindQueueItem;
                queuedOperationList.Response.HipsErrorMessage += "No Pcehr Message Queue items could be found with the criteria specified";
                return queuedOperationList;
            }

            return queuedOperationList;
        }
    }
}