﻿#region Using Directives

using System;
using System.Collections.Generic;
using System.ServiceModel;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.PCEHR;

#endregion Using Directives

namespace HIPS.ServiceContracts.ServiceInterfaces
{
    [ServiceContract(Namespace = "http://schemas.HIPS/Services/2012/01", Name = "IPCEHRService")]
    public interface IPCEHRService
    {
        /// <summary>
        /// Uploads a Level 3A discharge summary to the PCEHR National Repository, optionally superseding a previously-uploaded document.
        /// This takes in the XML document and any attachments and does the packaging itself.
        /// </summary>
        /// <param name="cdaDocument">The CDA root XML document.</param>
        /// <param name="patientIdentifier">The patient identifier at hospital, state, PDS or national level.</param>
        /// <param name="user">The user.</param>
        /// <param name="hospitalCode">The hospital code</param>
        /// <param name="hospitalCodeSystem">The hospital code system</param>
        /// <param name="attachments">The attachments</param>
        /// <param name="admissionDate">The admission date</param>
        /// <returns>Response indicating that the request was queued or the reason why not.</returns>
        [OperationContract]
        HipsResponse UploadOrSupersedeDocument(byte[] cdaDocument, PatientIdentifierBase patientIdentifier, UserDetails user, Attachment[] attachments, DateTime admissionDate, string documentFormatCode = null);

        /// <summary>
        /// Removes the specified discharge summary from the national PCEHR repository.
        /// </summary>
        /// <param name="patientIdentifier">An identifier of the patient (MRN, State ID, PatientMasterId or IHI)</param>
        /// <param name="hospitalCode">HospitalCode Code</param>
        /// <param name="hospitalCodeSystem">CodeSystem Code</param>
        /// <param name="admissionDate">Episode Admission Date</param>
        /// <param name="documentSetId">Clinical Document Set Id</param>
        /// <param name="reason">Reason for Removal</param>
        /// <param name="user">User details</param>
        /// <param name="auditInformation">Additional audit information to store</param>
        /// <returns>Response indicating that the request was queued or the reason why not.</returns>
        [OperationContract]
        HipsResponse Remove(PatientIdentifierBase patientIdentifier, DateTime admissionDate, string documentSetId, RemovalReason reason, UserDetails user, byte[] auditInformation);

        /// <summary>
        /// For a given MRN, State Patient Identifier, PCEHR Data Store PatientMasterId or a validated IHI,
        /// checks whether the patient has advertised the existence of his/her PCEHR,
        /// whether access has been gained, and whether access can be gained without a code.
        /// </summary>
        /// <param name="patientIdentifier">An identifier of the patient (MRN, State ID, PatientMasterId or IHI)</param>
        /// <param name="hospitalCode">Hospital code</param>
        /// <param name="hospitalCodeSystem">Hospital code system (pasFacCd)</param>
        /// <param name="user">The health provider individual, interactive user or authorised employee responsible for this action</param>
        /// <returns>Response containing operation success indicator, whether advertised, and access status</returns>
        [OperationContract]
        DoesPcehrExistResponse IsPcehrAdvertised(PatientIdentifierBase patientIdentifier, DateTime dateOfBirth, UserDetails user);

        /// <summary>
        /// Return a list of uploaded and changed documents from a single document set
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <param name="documentUniqueId"></param>
        /// <returns>ChangeHistoryViewResponse</returns>
        [OperationContract]
        ChangeHistoryViewResponse<PatientIdentifierBase> GetChangeHistoryView(PatientIdentifierBase patientIdentifier, UserDetails user, String documentUniqueId);

        /// <summary>
        /// This function is given the WS name of GetDocumentList.
        /// It returns all documents within the list without filtering the returned list
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <returns></returns>
        [OperationContractAttribute(Name = "GetDocumentList")]
        DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user);

        /// <summary>
        /// This function is given the WS name of GetDocumentListFilterDates.
        /// It returns all documents within the list with the ability to filter by start and end datetimes for creation and the service
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <param name="creationTimeStart"></param>
        /// <param name="creationTimeEnd"></param>
        /// <param name="serviceTimeStart"></param>
        /// <param name="serviceTimeEnd"></param>
        /// <returns></returns>
        [OperationContractAttribute(Name = "GetDocumentListFilterDates")]
        DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user, DateTime? creationTimeStart, DateTime? creationTimeEnd, DateTime? serviceTimeStart, DateTime? serviceTimeEnd);

        /// <summary>
        /// This function is given the WS name of GetDocumentListFilterStatusAndDates.
        /// It returns all documents within the list with the ability to filter by multiple statuses, start and end datetimes for creation and the service
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <param name="documentStatus"></param>
        /// <param name="creationTimeStart"></param>
        /// <param name="creationTimeEnd"></param>
        /// <param name="serviceTimeStart"></param>
        /// <param name="serviceTimeEnd"></param>
        /// <returns></returns>
        [OperationContractAttribute(Name = "GetDocumentListFilterStatusAndDates")]
        DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user, IList<DocumentStatus> documentStatus, DateTime? creationTimeStart, DateTime? creationTimeEnd, DateTime? serviceTimeStart, DateTime? serviceTimeEnd);

        /// <summary>
        /// This function is given the WS name of GetDocumentListActive.
        /// It returns all documents within the list that have the status as Approved
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <returns></returns>
        [OperationContractAttribute(Name = "GetDocumentListActive")]
        DocumentListResponse<PatientIdentifierBase> GetDocumentListActive(PatientIdentifierBase patientIdentifier, UserDetails user);

        /// <summary>
        /// This function is given the WS name of GetDocumentListActiveFilterDates.
        /// It returns all documents within the list that have the status as Approved and with the ability to filter by start and end datetimes for creation and the service
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <param name="creationTimeStart"></param>
        /// <param name="creationTimeEnd"></param>
        /// <param name="serviceTimeStart"></param>
        /// <param name="serviceTimeEnd"></param>
        /// <returns></returns>
        [OperationContractAttribute(Name = "GetDocumentListActiveFilterDates")]
        DocumentListResponse<PatientIdentifierBase> GetDocumentListActive(PatientIdentifierBase patientIdentifier, UserDetails user, DateTime? creationTimeStart, DateTime? creationTimeEnd, DateTime? serviceTimeStart, DateTime? serviceTimeEnd);

        /// <summary>
        /// This function is given the WS name of GetDocumentListWithQuery.
        /// The DocumentQuery object can be used to create the full spectrum of searches on the Document List.
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <param name="docQuery"></param>
        /// <returns></returns>
        [OperationContractAttribute(Name = "GetDocumentListWithQuery")]
        DocumentListResponse<PatientIdentifierBase> GetDocumentList(PatientIdentifierBase patientIdentifier, UserDetails user, DocumentQuery docQuery);

        /// <summary>
        /// Request for access without a code being required by PCEHR
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <returns>GainPcehrAccessResponse</returns>
        [OperationContract]
        GainPcehrAccessResponse GainAccessWithOutCode(PatientIdentifierBase patientIdentifier, UserDetails user);

        /// <summary>
        /// Request access with the access code being sent to the PCEHR
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="accessCode"></param>
        /// <param name="user"></param>
        /// <returns>GainPcehrAccessResponse</returns>
        [OperationContract]
        GainPcehrAccessResponse GainAccessWithCode(PatientIdentifierBase patientIdentifier, String AccessCode, UserDetails user);

        /// <summary>
        /// Requests emergency access to the patient's records on the PCEHR
        /// </summary>
        /// <param name="patientIdentifier"></param>
        /// <param name="user"></param>
        /// <returns>GainPcehrAccessResponse</returns>
        [OperationContract]
        GainPcehrAccessResponse GainAccessEmergency(PatientIdentifierBase patientIdentifier, UserDetails user);

        /// <summary>
        /// Returns the Document and Attachments required to display the requested document.
        /// </summary>
        /// <param name="patientIdentifier">An identifier for the patient.</param>
        /// <param name="user">Details of the person responsible for the action.</param>
        /// <param name="request">Identifiers required to locate the document.</param>
        /// <returns>DocumentResponse will also pass back success and error messages</returns>
        [OperationContract]
        DocumentResponse RetrieveDocument(PatientIdentifierBase patientIdentifier, UserDetails user, DocumentRequest request);

        /// <summary>
        /// Returns the Document and Attachments required to display the requested view.
        /// </summary>
        /// <param name="patientIdentifier">An identifier for the patient.</param>
        /// <param name="user">Details of the person responsible for the action.</param>
        /// <param name="request">The view request, which must be one of the concrete view types under the declared base type.</param>
        /// <returns>DocumentResponse will also pass back success and error messages</returns>
        [OperationContract]
        ViewResponse GetView(PatientIdentifierBase patientIdentifier, UserDetails user, ViewRequestBase request);

        /// <summary>
        /// Gets the status of all operations for the specified patient and episode.
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier</param>
        /// <param name="admissionDate">The admission date and time</param>
        /// <param name="user">The user</param>
        /// <returns>List of queued operations and indicator of success or failure.</returns>
        [OperationContract]
        OperationStatus GetOperationStatus(PatientIdentifierBase patientIdentifier, DateTime admissionDate, UserDetails user);

        /// <summary>
        /// Gets the status of a single pcehr queue operation for the specified PcehrMessageQueueId.
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier</param>
        /// <param name="admissionDate">The admission date and time</param>
        /// <param name="user">The user</param>
        /// <returns>List of queued operations and indicator of success or failure.</returns>
        [OperationContract]
        IndividualOperationStatus GetIndividualOperationStatus(int pcehrMessageQueueId, UserDetails user);

        /// <summary>
        /// Gets a List of queued Operations
        /// </summary>
        /// <param name="dateTimeFrom">The date time from.</param>
        /// <param name="dateTimeTo">The date time to.</param>
        /// <param name="queueOperation">The queue operation.</param>
        /// <param name="queueStatus">The queue status.</param>
        /// <param name="patientIdentifier">The patient identifier.</param>
        /// <param name="user">The user</param>
        /// <returns>
        /// List of queued operations and indicator of success or failure.
        /// </returns>
        [OperationContract]
        QueuedOperationStatus GetQueuedOperationList(DateTime dateTimeFrom, DateTime dateTimeTo, IList<QueueOperation> queueOperation, IList<QueueStatus> queueStatus, IList<PatientIdentifierBase> patientIdentifier, UserDetails user);
    }
}