﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Security.Cryptography.X509Certificates;
using Nehta.VendorLibrary.PCEHR;
using System.Net;
using Nehta.VendorLibrary.PCEHR.DocumentRepository;
using System.IO;
using System.ServiceModel;
using System.Net.Security;
using Nehta.VendorLibrary.PCEHR.PCEHRProfile;
using Nehta.VendorLibrary.PCEHR.RemoveDocument;

using Registry = Nehta.VendorLibrary.PCEHR.DocumentRegistry;
using IhiPcehrCommonSchemas;

namespace PcehrBusinessLogic
{
    public static class DocumentHelper
    {
        public static bool Upload(string package, string ihi, UserDetails user, string parentDocumentId)
        {
            // Obtain the certificate for use with TLS and signing
            X509Certificate2 certificate = Helpers.GetHpioCertificate();
            Uri url = Helpers.GetUploadDocumentUrl();

            // Create PCEHR header
            CommonPcehrHeader header = Helpers.GetHeader(ihi, user);

            // Create the client
            UploadDocumentClient uploadDocumentClient = new UploadDocumentClient(url, certificate, certificate);

            // Add server certificate validation callback
            ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;

            byte[] packageBytes = File.ReadAllBytes(package); // Create a package

            // Create a request from an existing package
            ProvideAndRegisterDocumentSetRequestType request;
            if (string.IsNullOrEmpty(parentDocumentId))
            {
                request = uploadDocumentClient.CreateRequestForNewDocument(
                packageBytes,
                FormatCodes.DischargeSummaryConformance3A,
                HealthcareFacilityTypeCodes.Hospitals,
                PracticeSettingTypes.GeneralHospital
               );
            }
            else
            {

                request = uploadDocumentClient.CreateRequestForReplacement(
                packageBytes,
                FormatCodes.DischargeSummaryConformance3A,
                HealthcareFacilityTypeCodes.Hospitals,
                PracticeSettingTypes.GeneralHospital,
                GetDocumentId(ihi, user, parentDocumentId)
               );
            }


            StringBuilder sb = new StringBuilder();

            try
            {
                // Invoke the service
                RegistryResponseType registryResponse = uploadDocumentClient.UploadDocument(header, request);

                // Get the soap request and response
                string soapRequest = uploadDocumentClient.SoapMessages.SoapRequest;
                string soapResponse = uploadDocumentClient.SoapMessages.SoapResponse;
                return (registryResponse.status.ToUpper().EndsWith("RESPONSESTATUSTYPE:SUCCESS"));
            }
            catch (FaultException<Nehta.VendorLibrary.PCEHR.GetTemplate.StandardErrorType> ex)
            {
                // The PCEHR StandardErrorType contains an errorCode and message
                sb.Append("[" + ex.Detail.errorCode + "] " + ex.Detail.message);
            }
            catch (Exception ex)
            {
                // Some other identifier of exception - e.g. could not connect
                if (ex.InnerException != null)
                {
                    sb.AppendLine(ex.InnerException.Message);
                }
                sb.Append(ex.Message);
            }
            throw new Exception(sb.ToString());
        }

        public static bool Replace(string package, string ihi, UserDetails user, string originalDocumentId)
        {
            // Obtain the certificate for use with TLS and signing
            X509Certificate2 certificate = Helpers.GetHpioCertificate();
            Uri url = Helpers.GetUploadDocumentUrl();

            // Create PCEHR header
            CommonPcehrHeader header = Helpers.GetHeader(ihi, user);

            // Create the client
            UploadDocumentClient uploadDocumentClient = new UploadDocumentClient(url, certificate, certificate);

            // Add server certificate validation callback
            ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;

            byte[] packageBytes = File.ReadAllBytes(package); // Create a package

            // Create a request from an existing package
            ProvideAndRegisterDocumentSetRequestType request = uploadDocumentClient.CreateRequestForReplacement(
                packageBytes,
                FormatCodes.DischargeSummaryConformance3A,
                HealthcareFacilityTypeCodes.Hospitals,
                PracticeSettingTypes.GeneralHospital,
                originalDocumentId
                );

            StringBuilder sb = new StringBuilder();

            try
            {
                // Invoke the service
                RegistryResponseType registryResponse = uploadDocumentClient.UploadDocument(header, request);

                // Get the soap request and response
                string soapRequest = uploadDocumentClient.SoapMessages.SoapRequest;
                string soapResponse = uploadDocumentClient.SoapMessages.SoapResponse;
                return (registryResponse.status.ToUpper().EndsWith("RESPONSESTATUSTYPE:SUCCESS"));
            }
            catch (FaultException<Nehta.VendorLibrary.PCEHR.GetTemplate.StandardErrorType> ex)
            {
                // The PCEHR StandardErrorType contains an errorCode and message
                sb.Append("[" + ex.Detail.errorCode + "] " + ex.Detail.message);
            }
            catch (Exception ex)
            {
                // Some other identifier of exception - e.g. could not connect
                if (ex.InnerException != null)
                {
                    sb.AppendLine(ex.InnerException.Message);
                }
                sb.Append(ex.Message);
            }
            throw new Exception(sb.ToString());
        }

        /// <summary>
        /// Removes the specified document id.
        /// </summary>
        /// <param name="documentId">The document id.</param>
        /// <param name="ihi">The ihi.</param>
        /// <param name="reason">The reason.</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        public static bool Remove(string documentId, string ihi, removeDocumentReasonForRemoval reason, UserDetails user)
        {
            // Obtain the certificate for use with TLS and signing
            X509Certificate2 certificate = Helpers.GetHpioCertificate();
            Uri url = Helpers.GetRemoveDocumentUrl();

            // Create PCEHR header
            CommonPcehrHeader header = Helpers.GetHeader(ihi, user);
            RemoveDocumentClient removeDocumentClient = new RemoveDocumentClient(url, certificate, certificate);

            // Add server certificate validation callback
            ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;
            StringBuilder sb = new StringBuilder();
            try
            {
                var request = new removeDocument()
                {
                    documentID = documentId,
                    reasonForRemoval = reason
                };

                // Invoke the service
                var responseStatus = removeDocumentClient.RemoveDocument(header, request);
                //responseStatus.responseStatus 
                // Get the soap request and response
                string soapRequest = removeDocumentClient.SoapMessages.SoapRequest;
                string soapResponse = removeDocumentClient.SoapMessages.SoapResponse;
                return responseStatus.responseStatus.code.ToUpper().Equals("PCEHR_SUCCESS");
               
            }
            catch (FaultException<Nehta.VendorLibrary.PCEHR.RemoveDocument.StandardErrorType> ex)
            {
                // The PCEHR StandardErrorType contains an errorCode and message
                sb.Append("[" + ex.Detail.errorCode + "] " + ex.Detail.message);
            }
            catch (Exception ex)
            {
                // Some other identifier of exception - e.g. could not connect
                if (ex.InnerException != null)
                {
                    sb.AppendLine(ex.InnerException.Message);
                }
                sb.Append(ex.Message);
            }
            throw new Exception(sb.ToString());
        }


        /// <summary>
        /// Validates the service certificate. - Not sure if we need to do anything here
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="certificate">The certificate.</param>
        /// <param name="chain">The chain.</param>
        /// <param name="sslPolicyErrors">The SSL policy errors.</param>
        /// <returns></returns>
        private static bool ValidateServiceCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            // Checks can be done here to validate the service certificate.
            // If the service certificate contains any problems or is invalid, return false. Otherwise, return true.
            // This example returns true to indicate that the service certificate is valid.
            return true;
        }

        /// <summary>
        /// Gets the document id.
        /// </summary>
        /// <param name="externalDocumentId">The external document id.</param>
        /// <returns></returns>
        private static string GetDocumentId(string ihi, UserDetails user, string parentDocumentId)
        {
            // Obtain the certificate for use with TLS and signing
            X509Certificate2 certificate = Helpers.GetHpioCertificate();
            Uri url = Helpers.GetDocumentListUrl();

            // Create PCEHR header
            CommonPcehrHeader header = Helpers.GetHeader(ihi, user);

            // Create the client
            GetDocumentListClient documentListClient = new GetDocumentListClient(url, certificate, certificate);

            // Add server certificate validation callback
            ServicePointManager.ServerCertificateValidationCallback += ValidateServiceCertificate;


            // Create a query
            AdhocQueryBuilder adhocQueryBuilder = new AdhocQueryBuilder(ihi, new[] { DocumentStatus.Approved });

            // Create the request using the query
            Registry.AdhocQueryRequest queryRequest = adhocQueryBuilder.BuildRequest();

            try
            {
                // Invoke the service
                Registry.AdhocQueryResponse queryResponse = documentListClient.GetDocumentList(header, queryRequest);

                foreach (Registry.ExtrinsicObjectType current in queryResponse.RegistryObjectList.ExtrinsicObject)
                {
                    
                    foreach (Registry.ExternalIdentifierType slot in current.ExternalIdentifier)
                    {
                        if (slot.value == parentDocumentId)
                        {
                            return current.id;
                        }
                    }
                }
               



                // Get the soap request and response
                string soapRequest = documentListClient.SoapMessages.SoapRequest;
                string soapResponse = documentListClient.SoapMessages.SoapResponse;
                throw new Exception("Parent Document not found");
            }
            catch (FaultException e)
            {
                throw new Exception("OK NO LUCK", e);
            }

        }
    }
}
