﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.IO;
using System.Security.Cryptography.X509Certificates;
using System.Configuration;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.PCEHR;
using enums = HIPS.PcehrDataStore.Schemas.Enumerators;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Reflection;
using IhiPcehrCommonSchemas;

namespace PcehrBusinessLogic
{
    public static class Helpers
    {
        /// <summary>
        /// XMLs the view model.
        /// </summary>
        /// <param name="xml">The XML.</param>
        /// <returns></returns>
        public static string XmlViewModel(string xml)
        {

            XmlDocument document = new XmlDocument();
            document.Load(new StringReader(xml));

            StringBuilder builder = new StringBuilder();
            using (XmlTextWriter writer = new XmlTextWriter(new StringWriter(builder)))
            {
                writer.Formatting = Formatting.Indented;
                document.Save(writer);
            }

            return builder.ToString();
        }

        /// <summary>
        /// Gets the nash certificate.
        /// </summary>
        /// <param name="pcehrServiceUrl">The pcehr service URL.</param>
        /// <param name="tlsCert">The TLS certificate.</param>
        public static X509Certificate2 GetHpioCertificate()
        {
            string serialNumber = ConfigurationManager.AppSettings["PcehrHpioCertificate"];
            if (string.IsNullOrEmpty(serialNumber))
            {
                throw new Exception("PCEHR certificate Serial has not been defined in the App.Config");
            }
            // Obtain the certificate for use with TLS
#if DEBUG
            return X509CertificateUtil.GetCertificate(serialNumber, X509FindType.FindBySerialNumber, StoreName.My, StoreLocation.CurrentUser, false);
#else
            return X509CertificateUtil.GetCertificate(serialNumber, X509FindType.FindBySerialNumber, StoreName.My, StoreLocation.LocalMachine, false);
#endif
        }

        /// <summary>
        /// Gets the does pcehr URL.
        /// </summary>
        /// <returns></returns>
        public static Uri GetDoesPcehrUrl()
        {
            string url = ConfigurationManager.AppSettings["DoesPCEHRExistUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Does PCEHR Exist Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the Upload document URL.
        /// </summary>
        /// <returns></returns>
        public static Uri GetUploadDocumentUrl()
        {
            string url = ConfigurationManager.AppSettings["UploadDocumentUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Upload Document Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the document list URL.
        /// </summary>
        /// <returns></returns>
        public static Uri GetDocumentListUrl()
        {
            string url = ConfigurationManager.AppSettings["ListDocumentUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("List Document Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Gets the Remove document URL.
        /// </summary>
        /// <returns></returns>
        public static Uri GetRemoveDocumentUrl()
        {
            string url = ConfigurationManager.AppSettings["RemoveDocumentUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("Remove Document Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }


        /// <summary>
        /// Gets the PCEHR header.
        /// </summary>
        public static CommonPcehrHeader GetHeader(string ihi, UserDetails user)
        {
            CommonPcehrHeader pcehrHeader = new CommonPcehrHeader();
            if (!string.IsNullOrEmpty(user.HpiI))
            {
                pcehrHeader.UserIdType = CommonPcehrHeaderUserIDType.HPII;
                pcehrHeader.UserId = user.HpiI;
            }
            else
            {
                pcehrHeader.UserIdType = CommonPcehrHeaderUserIDType.LocalSystemIdentifier;
                pcehrHeader.UserId = user.Login;
            }

            pcehrHeader.UserName = user.Name;
            pcehrHeader.IhiNumber = ihi;

            pcehrHeader.OrganisationName = ConfigurationManager.AppSettings["PcehrVendorId"];
            pcehrHeader.OrganisationId = ConfigurationManager.AppSettings["PcehrHpio"];

            pcehrHeader.ClientSystemType = CommonPcehrHeaderClientSystemType.CIS;
            pcehrHeader.ProductName = ConfigurationManager.AppSettings["PcehrProductName"];
            pcehrHeader.ProductPlatform = System.Environment.OSVersion.ToString();
            pcehrHeader.ProductVersion = ConfigurationManager.AppSettings["PcehrProductVersion"];
            pcehrHeader.ProductVendor = ConfigurationManager.AppSettings["PcehrVendorId"];
            return pcehrHeader;
        }

        /// <summary>
        /// Extracts the ihi from the fully qualified string.
        /// </summary>
        /// <param name="ihi">The ihi.</param>
        /// <returns></returns>
        public static string ExtractIhi(string ihi)
        {
            if (string.IsNullOrEmpty(ihi))
            {
                return string.Empty;
            }
            string result = ihi;
            if (ihi != null && ihi.StartsWith(HIQualifiers.IHIQualifier))
            {
                result = ihi.Substring(HIQualifiers.IHIQualifier.Length);
            }

            return result;
        }

        /// <summary>
        /// Inserts the ihi identifier in fromt of the IHI if required
        /// This should not need to be done, but for some reason the Medicare
        /// API needs it in the IHI field.
        /// </summary>
        /// <param name="ihi">The ihi.</param>
        /// <returns></returns>
        public static string InsertIhiIdentifier(string ihi)
        {
            string result = ihi;
            if (ihi != null && !ihi.StartsWith(HIQualifiers.IHIQualifier))
            {
                result = HIQualifiers.IHIQualifier + ihi;
            }

            return result;
        }

        /// <summary>
        /// Clones the specified source.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="source">The source.</param>
        /// <returns></returns>
        public static T Clone<T>(this T source)
        {
            if (!typeof(T).IsSerializable)
            {
                throw new ArgumentException("The type must be serializable.", "source");
            }

            // Don't serialize a null object, simply return the default for that object 
            if (Object.ReferenceEquals(source, null))
            {
                return default(T);
            }

            IFormatter formatter = new BinaryFormatter();
            Stream stream = new MemoryStream();
            using (stream)
            {
                formatter.Serialize(stream, source);
                stream.Seek(0, SeekOrigin.Begin);
                return (T)formatter.Deserialize(stream);
            }
        }

        /// <summary>
        /// Updates the specified target item. 
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="targetItem">The target item.</param>
        /// <param name="sourceItem">The source item.</param>
        public static void Update<T>(T targetItem, T sourceItem)
        {
            PropertyInfo[] fromFields = typeof(T).GetProperties();
            for (int count = 0; count < fromFields.Length; count++)
            {
                PropertyInfo fromField = (PropertyInfo)fromFields[count];
                if (fromField.CanWrite)
                {
                    object value = fromField.GetValue(sourceItem, null);
                    fromField.SetValue(targetItem, value, null);
                }
            }
        }

    }
}
