﻿// -----------------------------------------------------------------------
// <copyright file="CdaHelper.cs" company="NEHTA">
// Developed by Chamonix for NEHTA.
// </copyright>
// -----------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Security.Cryptography;
using System.Security.Cryptography.Xml;
using System.Text;
using System.Xml;
using System.Xml.Schema;

namespace Test.CommonCcaNoc.Helpers
{
    /// <summary>
    /// This class represents a CDA signature file (CDA_SIGN.XML) and is used for CCA testing.
    /// </summary>
    public class CdaSignature
    {
        #region Private Properties

        private const string SIGNED_PAYLOAD_NAMESPACE = "http://ns.electronichealth.net.au/xsp/xsd/SignedPayload/2010";
        private const string CDA_SIGNATURE_NAMESPACE = "http://ns.electronichealth.net.au/cdaPackage/xsd/eSignature/2012";
        private const string XML_SIGNATURE_NAMESPACE = "http://www.w3.org/2000/09/xmldsig#";

        private XmlDocument documentField;
        private XmlNamespaceManager namespaceManagerField;
        private CcaTest testField;

        /// <summary>
        /// The XmlDocument object that backs this CdaSignature object.
        /// </summary>
        private XmlDocument Document
        {
            get
            {
                if (documentField == null)
                {
                    documentField = new XmlDocument();
                }
                return documentField;
            }
        }

        /// <summary>
        /// An XmlNamespaceManager that includes the HL7 v3 (abbreviated to "x")
        /// and Australian CDA Extensions (abbreviated to "ext") namespaces.
        /// </summary>
        public XmlNamespaceManager NamespaceManager
        {
            get
            {
                if (namespaceManagerField == null)
                {
                    namespaceManagerField = new XmlNamespaceManager(new NameTable());
                    namespaceManagerField.AddNamespace("sp", SIGNED_PAYLOAD_NAMESPACE);
                    namespaceManagerField.AddNamespace("es", CDA_SIGNATURE_NAMESPACE);
                    namespaceManagerField.AddNamespace("ds", XML_SIGNATURE_NAMESPACE);
                }
                return namespaceManagerField;
            }
        }

        #endregion Private Properties

        /// <summary>
        /// Loads a CDA signature from the binary contents.
        /// </summary>
        /// <param name="cdaSignature"></param>
        public CdaSignature(byte[] cdaSignature, CcaTest test)
        {
            using (MemoryStream stream = new MemoryStream(cdaSignature))
            {
                Document.Load(stream);
            }
            testField = test;
        }

        /// <summary>
        /// Validates the CDA signature.
        /// </summary>
        /// <returns>True if the CDA signature passed schema validation.</returns>
        public bool ValidateCdaSignature(bool validateSignedPayload = true, bool validateESignatureNode = true)
        {
            XmlNode eSignatureNode = Document.SelectSingleNode("/sp:signedPayload/sp:signedPayloadData/es:eSignature", NamespaceManager);

            Document.Schemas.Add(SIGNED_PAYLOAD_NAMESPACE, "xsp-SignedPayload-2010.xsd");
            Document.Schemas.Add(CDA_SIGNATURE_NAMESPACE, "cdaPackage-eSignature-2012.xsd");
            Document.Schemas.Add(XML_SIGNATURE_NAMESPACE, "xmldsig-core-schema.xsd");
            schemaErrorsWereFound = false;

            try
            {
                if (validateSignedPayload)
                {
                    // Validate the signed payload document (where the signedPayload is lax)
                    Document.Validate(new ValidationEventHandler(ValidationEventHandler));
                    XmlQualifiedName typeName = Document.DocumentElement.SchemaInfo.SchemaType.QualifiedName;
                    testField.LogAssert.AreEqual(SIGNED_PAYLOAD_NAMESPACE, typeName.Namespace, DialogueResource.CdaSignDocumentElementTypeNamespace);
                    testField.LogAssert.AreEqual("SignedPayloadType", typeName.Name, DialogueResource.CdaSignDocumentElementTypeName);
                }

                if (validateESignatureNode)
                {
                    // Validate the eSignature node
                    Document.Validate(new ValidationEventHandler(ValidationEventHandler), eSignatureNode);
                    XmlQualifiedName typeName = eSignatureNode.SchemaInfo.SchemaType.QualifiedName;
                    testField.LogAssert.AreEqual(CDA_SIGNATURE_NAMESPACE, typeName.Namespace, DialogueResource.CdaSignESignatureElementTypeNamespace);
                    testField.LogAssert.AreEqual("eSignatureType", typeName.Name, DialogueResource.CdaSignESignatureElementTypeName);
                }
            }
            catch (XmlSchemaValidationException ex)
            {
                testField.Log(ex.Message);
                schemaErrorsWereFound = true;
            }
            return !schemaErrorsWereFound;
        }

        private bool schemaErrorsWereFound;

        private void ValidationEventHandler(object sender, ValidationEventArgs e)
        {
            if (e.Severity == XmlSeverityType.Error)
            {
                schemaErrorsWereFound = true;
            }
            testField.Log(e.Message);
        }

        /// <summary>
        /// Gets the reference URI attribute value from a reference element
        /// (either the one in Signature/SignedInfo or the one in
        /// eSignature/Manifest).
        /// </summary>
        /// <returns></returns>
        public string GetReferenceUri(XmlNode referenceNode)
        {
            XmlAttribute referenceUriAttribute = referenceNode.Attributes["URI"];
            if (referenceUriAttribute == null)
            {
                return null;
            }
            return referenceUriAttribute.Value;
        }

        /// <summary>
        /// Gets the digest value of the given reference node.
        /// </summary>
        /// <param name="referenceNode"></param>
        /// <returns></returns>
        public string GetReferenceDigestValue(XmlNode referenceNode)
        {
            XmlNode digestValue = referenceNode.SelectSingleNode("ds:DigestValue", NamespaceManager);
            if (digestValue == null)
            {
                return null;
            }
            return digestValue.InnerText;
        }

        /// <summary>
        /// Gets the exclusive canonicalisation of the signed payload data.
        /// </summary>
        /// <returns></returns>
        public byte[] GetExclusiveCanonicalisationOfSignedPayloadData()
        {
            XmlNodeList signedPayloadDataNode = Document.SelectNodes("/sp:signedPayload/sp:signedPayloadData", NamespaceManager);
            XmlDsigExcC14NTransform transform = new XmlDsigExcC14NTransform();
            transform.Context = signedPayloadDataNode[0] as XmlElement;
            string[] inclusiveNamespaces = new string[]
            {
                SIGNED_PAYLOAD_NAMESPACE, CDA_SIGNATURE_NAMESPACE, XML_SIGNATURE_NAMESPACE
            };
            transform.InclusiveNamespacesPrefixList = string.Join(" ", inclusiveNamespaces);
            byte[] xmlToTransform = UTF8Encoding.UTF8.GetBytes(signedPayloadDataNode[0].OuterXml);
            MemoryStream streamToTransform = new MemoryStream(xmlToTransform);
            transform.LoadInput(streamToTransform);
            Stream transformedStream = transform.GetOutput() as Stream;
            MemoryStream memoryStream = new MemoryStream();
            transformedStream.CopyTo(memoryStream);
            return memoryStream.ToArray();
        }

        /// <summary>
        /// Gets the value of the Algorithm attribute on the CanonicalizationMethod element.
        /// </summary>
        /// <returns></returns>
        public string GetCanonicalizationMethodAlgorithm()
        {
            XmlNode canonicalizationMethod = Document.SelectSingleNode("/sp:signedPayload/sp:signatures/ds:Signature/ds:SignedInfo/ds:CanonicalizationMethod", NamespaceManager);
            if (canonicalizationMethod == null)
            {
                return null;
            }
            XmlAttribute algorithmAttribute = canonicalizationMethod.Attributes["Algorithm"];
            if (algorithmAttribute == null)
            {
                return null;
            }
            return algorithmAttribute.Value;
        }

        /// <summary>
        /// Gets the value of the Algorithm attribute on the SignatureMethod element.
        /// </summary>
        /// <returns></returns>
        public string GetSignatureMethodAlgorithm()
        {
            XmlNode signatureMethod = Document.SelectSingleNode("/sp:signedPayload/sp:signatures/ds:Signature/ds:SignedInfo/ds:SignatureMethod", NamespaceManager);
            if (signatureMethod == null)
            {
                return null;
            }
            XmlAttribute algorithmAttribute = signatureMethod.Attributes["Algorithm"];
            if (algorithmAttribute == null)
            {
                return null;
            }
            return algorithmAttribute.Value;
        }

        /// <summary>
        /// Gets the signing key, from which we can determine the algorithm used to calculate the signature value.
        /// </summary>
        /// <returns></returns>
        public AsymmetricAlgorithm GetSigningKey()
        {
            SignedXml signedXml = new SignedXml(Document);
            XmlNode signatureNode = Document.SelectSingleNode("/sp:signedPayload/sp:signatures/ds:Signature", NamespaceManager);
            signedXml.LoadXml((XmlElement)signatureNode);
            AsymmetricAlgorithm signingKey;
            signedXml.CheckSignatureReturningKey(out signingKey);
            return signingKey;
        }

        /// <summary>
        /// Gets the ds:Transforms element from within the ds:Reference element.
        /// </summary>
        /// <param name="referenceElement"></param>
        /// <returns></returns>
        public XmlNode GetTransformsElement(XmlNode referenceElement)
        {
            return referenceElement.SelectSingleNode("ds:Transforms", NamespaceManager);
        }

        /// <summary>
        /// Gets a list of ds:Transform elements nested within the ds:Transforms element in the specified reference element.
        /// </summary>
        /// <param name="referenceElement"></param>
        /// <returns></returns>
        public XmlNodeList GetTransformElements(XmlNode referenceElement)
        {
            return referenceElement.SelectNodes("ds:Transforms/ds:Transform", NamespaceManager);
        }

        /// <summary>
        /// Calculates the Base64 representation of the SHA-1 hash over the given byte array.
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        public string CalculateBase64OfSha1Hash(byte[] input)
        {
            return Convert.ToBase64String(SHA1.Create().ComputeHash(input));
        }

        public XmlNodeList GetESignatureElements()
        {
            return Document.SelectNodes("/sp:signedPayload/sp:signedPayloadData/es:eSignature", NamespaceManager);
        }

        public XmlNode GetKeyInfoElement()
        {
            return Document.SelectSingleNode("/sp:signedPayload/sp:signatures/ds:Signature/ds:KeyInfo", NamespaceManager);
        }

        public XmlNodeList GetReferenceElementsInManifest()
        {
            return Document.SelectNodes("/sp:signedPayload/sp:signedPayloadData/es:eSignature/ds:Manifest/ds:Reference", NamespaceManager);
        }

        public XmlNodeList GetReferenceElementsInSignedInfo()
        {
            return Document.SelectNodes("/sp:signedPayload/sp:signatures/ds:Signature/ds:SignedInfo/ds:Reference", NamespaceManager);
        }

        public string GetSignedPayloadDataId()
        {
            XmlNode signedPayloadData = Document.SelectSingleNode("/sp:signedPayload/sp:signedPayloadData", NamespaceManager);
            if (signedPayloadData == null)
            {
                return null;
            }
            XmlAttribute idAttribute = signedPayloadData.Attributes["id"];
            if (idAttribute == null)
            {
                return null;
            }
            return idAttribute.Value;
        }

        /// <summary>
        /// Gets the person ID of the approver of the signature.
        /// </summary>
        /// <returns></returns>
        public string GetApproverPersonId()
        {
            XmlNode personId = Document.SelectSingleNode("/sp:signedPayload/sp:signedPayloadData/es:eSignature/es:approver/es:personId", NamespaceManager);
            if (personId == null)
            {
                return null;
            }
            return personId.InnerText;
        }

        /// <summary>
        /// Gets the signing time of the signature.
        /// </summary>
        /// <returns></returns>
        public string GetSigningTime()
        {
            XmlNode signingTime = Document.SelectSingleNode("/sp:signedPayload/sp:signedPayloadData/es:eSignature/es:signingTime", NamespaceManager);
            if (signingTime == null)
            {
                return null;
            }
            return signingTime.InnerText;
        }

        /// <summary>
        /// Gets the value of the Algorithm attribute on the ds:DigestMethod element.
        /// </summary>
        /// <param name="reference"></param>
        /// <returns></returns>
        public string GetDigestMethodAlgorithm(XmlNode reference)
        {
            XmlNode digestMethod = reference.SelectSingleNode("ds:DigestMethod", NamespaceManager);
            if (digestMethod == null)
            {
                return null;
            }
            XmlAttribute algorithm = digestMethod.Attributes["Algorithm"];
            if (algorithm == null)
            {
                return null;
            }
            return algorithm.Value;
        }

        /// <summary>
        /// Gets the ds:X509Data element inside the ds:KeyInfo element.
        /// </summary>
        /// <returns></returns>
        public XmlNode GetX509DataElement()
        {
            return Document.SelectSingleNode("/sp:signedPayload/sp:signatures/ds:Signature/ds:KeyInfo/ds:X509Data", NamespaceManager);
        }

        /// <summary>
        /// Gets the ds:X509Certificate element inside the ds:X509Data element.
        /// </summary>
        /// <returns></returns>
        public XmlNode GetX509CertificateElement()
        {
            return Document.SelectSingleNode("/sp:signedPayload/sp:signatures/ds:Signature/ds:KeyInfo/ds:X509Data/ds:X509Certificate", NamespaceManager);
        }

        /// <summary>
        /// Gets the ds:Object element under the ds:Signature element. The expected value is null.
        /// </summary>
        /// <returns></returns>
        public XmlNode GetObjectElement()
        {
            return Document.SelectSingleNode("/sp:signedPayload/sp:signatures/ds:Signature/ds:Object", NamespaceManager);
        }

        /// <summary>
        /// Gets the ds:Signature elements under the sp:signatures element.
        /// </summary>
        /// <returns></returns>
        public XmlNodeList GetSignatureElements()
        {
            return Document.SelectNodes("/sp:signedPayload/sp:signatures/ds:Signature", NamespaceManager);
        }

        /// <summary>
        /// Gets the family name of the approver of the signature.
        /// </summary>
        /// <returns></returns>
        public string GetApproverFamilyName()
        {
            XmlNode personId = Document.SelectSingleNode("/sp:signedPayload/sp:signedPayloadData/es:eSignature/es:approver/es:personName/es:familyName", NamespaceManager);
            if (personId == null)
            {
                return null;
            }
            return personId.InnerText;
        }

        /// <summary>
        /// Gets the given names of the approver of the signature.
        /// </summary>
        /// <returns></returns>
        public List<string> GetApproverGivenNames()
        {
            XmlNodeList givenNameElements = Document.SelectNodes("/sp:signedPayload/sp:signedPayloadData/es:eSignature/es:approver/es:personName/es:givenName", NamespaceManager);
            List<string> givenNames = new List<string>();
            foreach (XmlNode givenNameElement in givenNameElements)
            {
                givenNames.Add(givenNameElement.InnerText);
            }
            return givenNames;
        }

        /// <summary>
        /// Gets the text of the ds:SignatureValue element in the ds:Signature element.
        /// </summary>
        /// <returns></returns>
        public string GetSignatureValue()
        {
            XmlNode signatureValue = Document.SelectSingleNode("/sp:signedPayload/sp:signatures/ds:Signature/ds:SignatureValue", NamespaceManager);
            if (signatureValue == null)
            {
                return null;
            }
            return signatureValue.InnerText;
        }
    }
}