﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Test.CommonCcaNoc.Helpers
{
    /// <param name="assigningAuthorityName"></param>
    /// <param name="assigningGeographicArea"></param>
    /// <param name="code"></param>
    /// <param name="codeSystem"></param>
    /// <param name="codeSystemName"></param>
    class EntityIdentifier
    {
        /// <summary>
        /// This is the OID prefix defined in the NEHTA FAQ as a qualifier for a local system identifier.
        /// It must have the HPI-O appended to be a valid OID.
        /// </summary>
        public const string HPIO_QUALIFIED_LOCAL_IDENTIFIER_OID_PREFIX = "1.2.36.1.2001.1005.41.";

        /// <summary>
        /// An OID that is the identifier or qualifies the extension.
        /// </summary>
        public string Root { get; set; }

        /// <summary>
        /// An identifier qualified by the root. For an HPI-I this should be null.
        /// </summary>
        public string Extension { get; set; }

        /// <summary>
        /// The name of the system that assigned the identifier. For an HPI-I this should be "HPI-I".
        /// </summary>
        public string AssigningAuthorityName { get; set; }

        /// <summary>
        /// The geographic area within which the identifier is assigned. For an HPI-I this should be "National Identifier".
        /// </summary>
        public string AssigningGeographicArea { get; set; }

        /// <summary>
        /// The identifier type code, "EI" for a local identifier, null for an HPI-I.
        /// </summary>
        public string Code { get; set; }

        /// <summary>
        /// The identifier type codeSystem, "2.16.840.1.113883.12.203" for a local identifier, null for an HPI-I.
        /// </summary>
        public string CodeSystem { get; set; }

        /// <summary>
        /// The identifier type codeSystemName, "Identifier Type (HL7)" for a local identifier, null for an HPI-I.
        /// </summary>
        public string CodeSystemName { get; set; }

        /// <summary>
        /// Creates an entity identifier for the HPI-I of a provider individual.
        /// </summary>
        /// <param name="hpii"></param>
        public static EntityIdentifier HpiI(string hpii)
        {
            EntityIdentifier ei = new EntityIdentifier();
             ei.Root = string.Format("1.2.36.1.2001.1003.0.{0}", hpii);
             ei.Extension = null;
             ei.AssigningAuthorityName = "HPI-I";
             ei.AssigningGeographicArea = "National Identifier";
             ei.Code = null;
             ei.CodeSystem = null;
             ei.CodeSystemName = null;
            return ei;
        }

        /// <summary>
        /// Creates an entity identifier for the HPI-O of a provider organisation.
        /// </summary>
        /// <param name="hpii"></param>
        public static EntityIdentifier HpiO(string hpio)
        {
            EntityIdentifier ei = new EntityIdentifier();
            ei.Root = string.Format("1.2.36.1.2001.1003.0.{0}", hpio);
            ei.Extension = null;
            ei.AssigningAuthorityName = "HPI-O";
            ei.AssigningGeographicArea = "National Identifier";
            ei.Code = null;
            ei.CodeSystem = null;
            ei.CodeSystemName = null;
            return ei;
        }

        /// <summary>
        /// Creates an entity identifier for the local system identifier of a provider individual.
        /// </summary>
        /// <param name="hpio">The assigning HPI-O</param>
        /// <param name="local">The local system identifier</param>
        public static EntityIdentifier Local(string hpio, string local)
        {
            EntityIdentifier ei = new EntityIdentifier();
            ei.Root = string.Format("{0}{1}", HPIO_QUALIFIED_LOCAL_IDENTIFIER_OID_PREFIX, hpio);
            ei.Extension = local;
            ei.AssigningAuthorityName = "Local ID";
            ei.AssigningGeographicArea = null;
            ei.Code = "EI";
            ei.CodeSystem = "2.16.840.1.113883.12.203";
            ei.CodeSystemName = "Identifier Type (HL7)";
            return ei;
        }
    }
}
