﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data.SqlClient;
using HIPS.Base.Schemas;
using HIPS.Base.Schemas.Enumerators;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Schemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace Test.Helpers
{
    public class PatientTestHelpersBase : BaseDl
    {
        #region Private Members

        private PatientMaster targetPatient;
        private SqlCommand testCommand;

        #endregion Private Members

        #region Properties

        /// <summary>
        /// Gets or sets the target patient.
        /// </summary>
        /// <value>
        /// The target patient.
        /// </value>
        public PatientMaster TargetPatient
        {
            get
            {
                return targetPatient;
            }
            set
            {
                targetPatient = value;
            }
        }

        /// <summary>
        /// Gets the test SQL command object that uses a different connection string the base command object.
        /// </summary>
        public SqlCommand TestCommand
        {
            get
            {
                if (testCommand == null)
                {
                    testCommand = new SqlCommand();
                    SqlConnection connection = new SqlConnection(ConfigurationManager.ConnectionStrings["TestPcehrDataStoreConnectionString"].ToString());
                    testCommand.Connection = connection;
                }
                if (testCommand.Connection.State != System.Data.ConnectionState.Open)
                {
                    testCommand.Connection.Open();
                }
                testCommand.Parameters.Clear();
                return testCommand;
            }
        }

        #endregion Properties

        #region Constructors

        /// <summary>
        /// Initializes a new instance of the <see cref="PatientTestHelpersBase"/> class.
        /// </summary>
        public PatientTestHelpersBase()
        {
            targetPatient = new PatientMaster();
        }

        #endregion Constructors

        #region PatientMasterDL generic Test data

        /// <summary>
        /// Sets the test data.
        /// </summary>
        /// <returns></returns>
        virtual public int SetTestData()
        {
            ResetPatientData();

            GetTestInsertPatient();
            PatientMasterDl patientMasterDl = new PatientMasterDl(GetTestUser());
            bool result = patientMasterDl.Insert(TargetPatient, null);
            if (!result || !TargetPatient.PatientMasterId.HasValue)
            {
                Assert.Fail("The test patient was not saved");
            }
            return TargetPatient.PatientMasterId.Value;
        }

        public UserDetails GetTestUser()
        {
            UserDetails user = new UserDetails();
            user.Role = UserRole.InteractiveUser;
            user.Login = "TestLogin";
            user.Domain = "TestDomain";
            user.Name = "TestName";
            return user;
        }

        /// <summary>
        /// Gets the insert patient.
        /// </summary>
        /// <returns></returns>
        virtual public void GetTestInsertPatient()
        {
            targetPatient.CurrentSexId = 1;
            targetPatient.DateOfBirth = new DateTime(1969, 07, 20);
            targetPatient.DateOfDeath = new DateTime(2023, 08, 21);
            targetPatient.MedicareNumber = "1234567890";
            targetPatient.MedicareIrn = "1";
            targetPatient.IsMedicareNumberValid = false;
            targetPatient.DvaNumber = "DvaNumber";
            targetPatient.SetNewCurrentName(null, "FirstName", "LastName", null);
            targetPatient.SetNewCurrentName(null, "FirstName2", "LastName", null);
            targetPatient.SetNewCurrentName(null, "FirstName3", "LastName", null);

            targetPatient.Addresses = GetAddresses();
            targetPatient.Contacts = GetContacts();
        }

        /// <summary>
        /// Gets the names.
        /// Provide an override if you require a specific address
        /// </summary>
        /// <returns></returns>
        virtual public List<Address> GetAddresses()
        {
            List<Address> results = new List<Address>();
            results.Add(new Address("1-Line1", "1-Line2", "1-PlaceName", 1, "1-ISC", "1-PC", 1, 1));
            results.Add(new Address("2-Line1", "2-Line2", "2-PlaceName", 1, "2-ISC", "2-PC", 1, 1));
            results.Add(new Address("3-Line1", "3-Line2", "3-PlaceName", 1, "3-ISC", "3-PC", 1, 1));
            results.Add(new Address("4-Line1", "4-Line2", "4-PlaceName", 1, "4-ISC", "4-PC", 1, 1));
            results.Add(new Address("5-Line1", "5-Line2", "5-PlaceName", 1, "5-ISC", "5-PC", 1, 1));
            return results;
        }

        /// <summary>
        /// Gets the contacts.
        /// </summary>
        /// <returns></returns>
        virtual public List<Contact> GetContacts()
        {
            List<Contact> results = new List<Contact>();
            results.Add(new Contact("1-Detail1", 1, "First 1"));
            results.Add(new Contact("1-Detail2", 1, "First 2"));
            results.Add(new Contact("1-Detail3", 1, "First 3"));
            results.Add(new Contact("1-Detail4", 1, "First 4"));
            results.Add(new Contact("1-Detail5", 1, "First 5"));
            return results;
        }

        /// <summary>
        /// Sets the name of the current.
        /// </summary>
        /// <returns></returns>
        private PatientMasterName SetCurrentName()
        {
            PatientMasterName name = new PatientMasterName(null, null, "Test Name", "Family1", null, null, (int)NameTypes.Current, null);
            return name;
        }

        #endregion PatientMasterDL generic Test data

        #region Generic

        /// <summary>
        /// Resets the Patient data. This can now delete multiple PatientMaster records from the database,
        /// which is required for certain cases when a test creates multiple copies of a patient record.
        /// The subclass of PatientTestHelpersBase can override SqlCommandText method to provide a query
        /// to find the patient master IDs that should be deleted.
        /// </summary>
        virtual public void ResetPatientData()
        {
            GetTestInsertPatient();
            GenericDataAccess dataAccess = new GenericDataAccess();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = SqlCommandText();
                command.CommandType = System.Data.CommandType.Text;
                SqlDataReader reader = command.ExecuteReader();
                List<int> patientMasterIds = new List<int>();
                while (reader.Read())
                {
                    patientMasterIds.Add(reader.GetInt32(0));
                }
                reader.Close();
                foreach (int patientMasterId in patientMasterIds)
                {
                    DeleteChildData(command, patientMasterId);
                    command.CommandText = string.Format("DELETE hips.PatientMaster WHERE PatientMasterId = {0}", patientMasterId);
                    int rows = command.ExecuteNonQuery();
                    if (rows == 0)
                    {
                        throw new Exception("Patient test data has not been deleted");
                    }
                }
                command.Connection.Close();
            }
        }

        /// <summary>
        /// Deletes the child data from the existing patient.
        /// </summary>
        /// <param name="command">The command.</param>
        /// <param name="patientMasterId">The patient master ID.</param>
        private void DeleteChildData(SqlCommand command, int? patientMasterId)
        {
            command.CommandText = string.Format("DELETE [hips].[PatientMasterIhiHistory] Where PatientMasterId = {0}", patientMasterId);
            command.ExecuteNonQuery();

            command.CommandText = string.Format("DELETE [hips].[PatientMasterIhi] Where PatientMasterId = {0}", patientMasterId);
            command.ExecuteNonQuery();

            command.CommandText = string.Format("DELETE [hips].[IhiLookupAlert] WHERE PatientMasterId = {0}", patientMasterId);
            command.ExecuteNonQuery();

            command.CommandText = string.Format("DELETE [hips].[IhiLookupAudit] Where PatientMasterId = {0}", patientMasterId);
            command.ExecuteNonQuery();

            command.CommandText = string.Format("DELETE [hips].[PcehrAudit] Where PatientMasterId = {0}", patientMasterId);
            command.ExecuteNonQuery();

            command.CommandText = string.Format("DELETE [hips].[HealthProviderOrganisationPatient] Where PatientMasterId = {0}", patientMasterId);
            command.ExecuteNonQuery();

            command.CommandText = string.Format("DELETE [hips].[Address] WHERE AddressId IN (SELECT AddressId FROM [hips].[PatientMasterAddress] WHERE PatientMasterId = {0})", patientMasterId);
            command.ExecuteNonQuery();

            command.CommandText = string.Format("DELETE [hips].[Contact] WHERE ContactId IN (SELECT ContactId FROM [hips].[PatientMasterContact] WHERE PatientMasterId = {0})", patientMasterId);
            command.ExecuteNonQuery();

            command.CommandText = string.Format("DELETE hips.PatientMasterName WHERE PatientMasterId = {0}", patientMasterId);
            command.ExecuteNonQuery();

            command.CommandText = string.Format("DELETE hips.DownloadedDocument WHERE PatientMasterId = {0}", patientMasterId);
            command.ExecuteNonQuery();

            foreach (int patientId in GetPatientIds(patientMasterId))
            {
                foreach (int episodeId in GetEpisodeIds(patientId))
                {
                    foreach (int clinicalDocumentId in GetClinicalDocumentIds(episodeId))
                    {
                        command.CommandText = string.Format("DELETE [hips].[ClinicalDocumentVersion] WHERE ClinicalDocumentId = {0}", clinicalDocumentId);
                        command.ExecuteNonQuery();

                        command.CommandText = string.Format("DELETE [hips].[RemoveAudit] WHERE ClinicalDocumentId = {0}", clinicalDocumentId);
                        command.ExecuteNonQuery();
                    }
                    command.CommandText = string.Format("DELETE [hips].[ClinicalDocument] WHERE EpisodeId = {0}", episodeId);
                    command.ExecuteNonQuery();

                    command.CommandText = string.Format("DELETE [hips].[ConsentAudit] WHERE EpisodeId = {0}", episodeId);
                    command.ExecuteNonQuery();
                }
                command.CommandText = string.Format("DELETE hips.Episode WHERE PatientId = {0}", patientId);
                command.ExecuteNonQuery();
            }

            command.CommandText = string.Format("DELETE [hips].[HospitalPatient] Where PatientMasterId = {0}", patientMasterId);

            command.ExecuteNonQuery();

            // Do not reseed the identity columns here! That will make it reuse the IDs within a test run, corrupting the integrity of the test reports.

            //IhiLookupAudit lastOne = GetLastIhiLookupAudit();
            //if (lastOne.IhiLookupAuditId.HasValue)
            //{
            //    command.CommandText = string.Format("DBCC CHECKIDENT ('[hips].[IhiLookupAudit]', reseed, {0})", lastOne.IhiLookupAuditId);
            //    command.ExecuteNonQuery();
            //}

            //IhiLookupAlert lastOneAlert = GetLastIhiLookupAlert();
            //command.CommandText = string.Format("DBCC CHECKIDENT ('[hips].[IhiLookupAlert]', reseed, {0})", lastOneAlert.IhiLookupAlertId);
            //command.ExecuteNonQuery();

            //PcehrAudit lastPcehrAudit = GetLastPcehrAudit();
            //if (lastPcehrAudit.PcehrAuditId.HasValue)
            //{
            //    command.CommandText = string.Format("DBCC CHECKIDENT ('[hips].[PcehrAudit]', reseed, {0})", lastPcehrAudit.PcehrAuditId);
            //    command.ExecuteNonQuery();
            //}
        }

        private List<int> GetPatientIds(int? patientMasterId)
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("SELECT PatientId FROM [hips].[HospitalPatient] WHERE PatientMasterId = {0}", patientMasterId);
                SqlDataReader reader = command.ExecuteReader();
                List<int> patientIds = new List<int>();
                while (reader.Read())
                {
                    patientIds.Add(reader.GetInt32(0));
                }
                reader.Close();
                return patientIds;
            }
        }

        private List<int> GetEpisodeIds(int? patientId)
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("SELECT EpisodeId FROM [hips].[Episode] WHERE PatientId = {0}", patientId);
                SqlDataReader reader = command.ExecuteReader();
                List<int> episodeIds = new List<int>();
                while (reader.Read())
                {
                    episodeIds.Add(reader.GetInt32(0));
                }
                reader.Close();
                return episodeIds;
            }
        }

        public List<int> GetClinicalDocumentIds(int? episodeId)
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("SELECT ClinicalDocumentId FROM [hips].[ClinicalDocument] WHERE EpisodeId = {0}", episodeId);
                SqlDataReader reader = command.ExecuteReader();
                List<int> clinicalDocumentIds = new List<int>();
                while (reader.Read())
                {
                    clinicalDocumentIds.Add(reader.GetInt32(0));
                }
                reader.Close();
                return clinicalDocumentIds;
            }
        }

        private List<int> GetClinicalDocumentVersionIds(int? clinicalDocumentId)
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("SELECT ClinicalDocumentVersionId FROM [hips].[ClinicalDocumentVersion] WHERE ClinicalDocumentId = {0}", clinicalDocumentId);
                SqlDataReader reader = command.ExecuteReader();
                List<int> clinicalDocumentVersionIds = new List<int>();
                while (reader.Read())
                {
                    clinicalDocumentVersionIds.Add(reader.GetInt32(0));
                }
                reader.Close();
                return clinicalDocumentVersionIds;
            }
        }

        /// <summary>
        /// Deletes the patient.
        /// </summary>
        /// <param name="mrn">The MRN.</param>
        public void DeletePatientByMrn(string mrn)
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            using (SqlCommand command = TestCommand) //dataAccess.Command(this.SqlCommandText()))
            {
                command.CommandText = string.Format("SELECT PatientMasterId FROM [hips].[Hospitalpatient] WHERE mrn = '{0}'", mrn);
                int? result = System.Convert.ToInt32(command.ExecuteScalar());
                if (result.HasValue && result > 0)
                {
                    DeleteChildData(command, result);

                    command.CommandText = string.Format("DELETE hips.PatientMaster WHERE PatientMasterId = {0}", result);
                    int rows = command.ExecuteNonQuery();
                    if (rows == 0)
                    {
                        throw new Exception("Patient test data has not been deleted");
                    }
                }
                command.Connection.Close();
            }
            return;
        }

        /// <summary>
        /// Gets the patient by MRN.
        /// </summary>
        /// <param name="mrn">The MRN.</param>
        /// <returns></returns>
        public PatientMaster GetPatientByMrn(string mrn)
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            PatientMaster result = new PatientMaster();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("SELECT *, pmi.DateLastValidated AS IhiLastValidated FROM [hips].[PatientMaster] pm LEFT JOIN [hips].[HospitalPatient] AS p on p.PatientMasterId = pm.PatientMasterId LEFT OUTER JOIN [hips].[PatientMasterIhi] AS pmi ON pmi.PatientMasterId = p.PatientMasterId WHERE mrn = '{0}'", mrn);
                PopulateBusinessObject<PatientMaster>(command.ExecuteReader(), result);
                command.CommandText = string.Format("SELECT *, t.Code AS Title, s.Code AS Suffix FROM [hips].[PatientMasterName] pmn LEFT JOIN [hips].[HospitalPatient] AS p on p.PatientMasterId = pmn.PatientMasterId LEFT JOIN [hips].[Title] AS t ON t.TitleId = pmn.TitleId LEFT JOIN [hips].[Suffix] AS s ON s.SuffixId = pmn.SuffixId WHERE mrn = '{0}'", mrn);
                result.Names = GetPopulatedBusinessList<PatientMasterName>(command.ExecuteReader());
                command.Connection.Close();
            }
            return result;
        }

        /// <summary>
        /// Returns a SQL command to obtain the patient master ID of the target patient if it already exists in the database.
        /// The implementation in this base class looks for a patient with the same date of birth, date of death, medicare number and dva number.
        /// The implementation can be overridden in a subclass.
        /// Provide an override to match your target patient if the supplied query won't work.
        /// </summary>
        /// <returns>A SQL command to obtain the patient master ID of the target patient if it already exists in the database.</returns>
        virtual public string SqlCommandText()
        {
            return string.Format("SELECT PatientMasterId FROM hips.PatientMaster  WHERE DateOfBirth = '{0:yyyy-MM-dd}' AND DateOfDeath = '{1:yyyy-MM-dd}' AND MedicareNumber = '{2}' AND DvaNumber = '{3}'", TargetPatient.DateOfBirth, TargetPatient.DateOfDeath, TargetPatient.MedicareNumber, TargetPatient.DvaNumber);
        }

        /// <summary>
        /// Deletes the patient master ihi by MRN.
        /// </summary>
        /// <param name="mrn">The MRN.</param>
        public void DeletePatientMasterIhiByMrn(string mrn)
        {
            PatientMaster patient = GetPatientByMrn(mrn);
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("DELETE [hips].[PatientMasterIhi] Where PatientMasterId = {0}", patient.PatientMasterId);
                command.ExecuteNonQuery();
            }
        }

        /// <summary>
        /// Gets the last error message.
        /// </summary>
        /// <returns></returns>
        public SystemErrorLog GetLastErrorMessage()
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            SystemErrorLog result = new SystemErrorLog();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = "SELECT TOP 1 * from [hips].[SystemErrorLog] ORDER BY SystemErrorLogId DESC";
                PopulateBusinessObject<SystemErrorLog>(command.ExecuteReader(), result);
            }
            return result;
        }

        /// <summary>
        /// Gets the last ihi lookup alert.
        /// </summary>
        /// <returns></returns>
        public IhiLookupAlert GetLastIhiLookupAlert()
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            IhiLookupAlert result = new IhiLookupAlert();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = "SELECT TOP 1 * from [hips].[IhiLookupAlert] ORDER BY IhiLookupAlertId DESC";
                PopulateBusinessObject<IhiLookupAlert>(command.ExecuteReader(), result);
            }
            return result;
        }

        /// <summary>
        /// Gets the last IHI lookup audit record.
        /// </summary>
        /// <returns></returns>
        public IhiLookupAudit GetLastIhiLookupAudit()
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            IhiLookupAudit result = new IhiLookupAudit();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = "SELECT TOP 1 * from [hips].[IhiLookupAudit] ORDER BY IhiLookupAuditId DESC";
                PopulateBusinessObject<IhiLookupAudit>(command.ExecuteReader(), result);
            }
            return result;
        }

        /// <summary>
        /// Gets the last PCEHR access audit record.
        /// </summary>
        /// <returns></returns>
        public PcehrAudit GetLastPcehrAudit()
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            PcehrAudit result = new PcehrAudit();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = "SELECT TOP 1 * from [hips].[PcehrAudit] ORDER BY PcehrAuditId DESC";
                PopulateBusinessObject<PcehrAudit>(command.ExecuteReader(), result);
            }
            return result;
        }

        /// <summary>
        /// Gets the last PCEHR access audit record for a specific service name.
        /// </summary>
        /// <returns></returns>
        public PcehrAudit GetLastPcehrAudit(string serviceName)
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            PcehrAudit result = new PcehrAudit();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = String.Format("SELECT TOP 1 * from [hips].[PcehrAudit] WHERE ServiceName = '{0}' ORDER BY PcehrAuditId DESC", serviceName);
                PopulateBusinessObject<PcehrAudit>(command.ExecuteReader(), result);
            }
            return result;
        }

        /// <summary>
        /// Gets the last clinical document version.
        /// </summary>
        /// <returns>The last clinical document version</returns>
        public ClinicalDocumentVersion GetLastClinicalDocumentVersion()
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            ClinicalDocumentVersion result = new ClinicalDocumentVersion();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = "SELECT TOP 1 * from [hips].[ClinicalDocumentVersion] ORDER BY ClinicalDocumentVersionId DESC";
                PopulateBusinessObject<ClinicalDocumentVersion>(command.ExecuteReader(), result);
            }
            return result;
        }

        /// <summary>
        /// Gets the last ihi history by MRN.
        /// </summary>
        /// <param name="mrn">The MRN.</param>
        /// <returns></returns>
        public PatientMasterIhiHistory GetLastIhiHistoryByMrn(string mrn)
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            PatientMaster patient = GetPatientByMrn(mrn);
            PatientMasterIhiHistory result = new PatientMasterIhiHistory();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("SELECT TOP 1 * from [hips].[PatientMasterIhiHistory] WHERE PatientMasterId = {0} ORDER BY IhiHistoryId DESC", patient.PatientMasterId.Value);
                PopulateBusinessObject<PatientMasterIhiHistory>(command.ExecuteReader(), result);
            }
            return result;
        }

        /// <summary>
        /// Inserts the PatientMasterIhi item.
        /// </summary>
        /// <param name="item">The item.</param>
        public void PatientMasterIhiInsert(PatientMasterIhi item)
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = "[hips].[PatientMasterIhiInsert]";
                command.CommandType = System.Data.CommandType.StoredProcedure;
                Insert<PatientMasterIhi>(item, command);
                command.CommandType = System.Data.CommandType.Text;
            }
        }

        /// <summary>
        /// Deletes the PatientMasterIhi item.
        /// </summary>
        /// <param name="item">The item.</param>
        public void PatientMasterIhiDelete(PatientMasterIhi item)
        {
            GenericDataAccess dataAccess = new GenericDataAccess();
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = "[hips].[PatientMasterIhiDelete]";
                command.CommandType = System.Data.CommandType.StoredProcedure;
                Delete<PatientMasterIhi>(item, command);
                command.CommandType = System.Data.CommandType.Text;
            }
        }

        /// <summary>
        /// Ihis the lookup alerts occured.
        /// </summary>
        /// <param name="lastOne">The last one.</param>
        /// <param name="expectedErrors">The expected errors.</param>
        /// <returns></returns>
        public bool IhiLookupAlertsOccured(IhiLookupAlert lastOne, int expectedErrors = 1)
        {
            IhiLookupAlert latestOne = GetLastIhiLookupAlert();
            bool result = (lastOne.IhiLookupAlertId + expectedErrors) != (latestOne.IhiLookupAlertId);
            if (result)
            {
                Assert.Fail(string.Format("An IHI Lookup alert occured that was not expected {0}", latestOne.Message));
            }
            return result;
        }

        /// <summary>
        /// Ihis the lookup audits occured.
        /// </summary>
        /// <param name="lastOne">The last one.</param>
        /// <param name="expectedErrors">The expected errors.</param>
        /// <returns></returns>
        public bool IhiLookupAuditsOccured(IhiLookupAudit lastOne, int expectedErrors = 1)
        {
            IhiLookupAudit latestOne = GetLastIhiLookupAudit();
            bool result = (lastOne.IhiLookupAuditId + expectedErrors) != (latestOne.IhiLookupAuditId);
            if (result)
            {
                Assert.Fail(string.Format("An IHI Lookup audit occured that was not expected {0}", latestOne.Message));
            }
            return result;
        }

        /// <summary>
        /// Anies the system errors occur.
        /// </summary>
        /// <param name="lastOne">The last error message.</param>
        /// <returns></returns>
        public bool AnySystemErrorsOccur(SystemErrorLog lastErrorMessage, int expectedErrors = 0)
        {
            SystemErrorLog latestErrorMessage = GetLastErrorMessage();
            bool result = (lastErrorMessage.SystemErrorLogId + expectedErrors) != (latestErrorMessage.SystemErrorLogId);
            if (result)
            {
                Assert.Fail(string.Format("A System Error Occured {0}", latestErrorMessage.ExceptionMessage));
            }
            return result;
        }

        /// <summary>
        /// Sets the ihi for a known patient.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <param name="ihi">The ihi.</param>
        public void SetIhi(int patientMasterId, string ihi)
        {
            using (SqlCommand command = TestCommand)
            {
                command.CommandText = string.Format("UPDATE [hips].[PatientMasterIhi] SET IHI = '{0}' WHERE PatientMasterId = {1}", ihi, patientMasterId);
                command.ExecuteScalar();
                command.Connection.Close();
            }
        }

        #endregion Generic
    }
}