﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using HIPS.PcehrDataStore.Schemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.MedicareCca.Helpers;

namespace Test.MedicareCCA.Helpers
{
    /// <summary>
    /// Base class for all Medicare CCA test classes, which handles logging the test results into an HTML report.
    /// </summary>
    [TestClass]
    public class CcaTest
    {
        #region Private Properties

        /// <summary>
        /// The date and time when the test method started.
        /// </summary>
        private DateTime TestStartTime { get; set; }

        protected bool batching = false;

        private StringWriter testReportField;
        private const string TEST_FINISHED_KEY = "{TEST_FINISHED}";
        private const string TEST_OUTCOME_KEY = "{TEST_OUTCOME}";
        private LogAssert logAssertField;

        private bool InLogTable { get; set; }

        private string ResultsDirectory
        {
            get
            {
                DirectoryInfo directory = new DirectoryInfo(TestContext.TestRunDirectory);
                if (!directory.Exists)
                {
                    directory.Create();
                }
                return directory.FullName;
            }
        }

        /// <summary>
        /// Writes the database records created during the test into the test report.
        /// </summary>
        private void LogDatabaseRecords()
        {
            if (patient != null)
            {
                Assert.IsNotNull(patient);
                Log(patient.GetIhiLookupAudits());
            }
            else if (provider != null && !batching)
            {
                Assert.IsNotNull(provider);
                Log(provider.GetLastHpiiLookupAudit());
            }
            else if (provider != null && batching)
            {
                Assert.IsNotNull(provider);
                LogBatch(provider.GetLastHpiiLookupAudit());
            }
            else
            {
                LeaveLogTable();
                TestReport.WriteLine("<h3>No Patient Record Created</h3>");
            }
        }

        #endregion Private Properties

        #region Public Properties

        public CcaPatient patient { get; set; }

        public CcaProviders provider { get; set; }

        /// <summary>
        /// The unit testing framework sets this property when starting the test.
        /// </summary>
        public TestContext TestContext { get; set; }

        /// <summary>
        /// A stream to write to the test results log file.
        /// </summary>
        public StringWriter TestReport
        {
            get
            {
                if (testReportField == null)
                {
                    testReportField = new StringWriter();
                    WriteTestReportHeader();
                }
                return testReportField;
            }
        }

        /// <summary>
        /// The logging assert instance that logs to the current test context.
        /// </summary>
        public LogAssert LogAssert
        {
            get
            {
                if (logAssertField == null)
                {
                    logAssertField = new LogAssert(this);
                }
                return logAssertField;
            }
        }

        #endregion Public Properties

        #region Public Methods

        /// <summary>
        /// Logs a message to the Test Assertion Log table in the test report.
        /// </summary>
        /// <param name="message">Format string</param>
        /// <param name="parameters">Substitution parameters for format string</param>
        public void Log(string message, params object[] parameters)
        {
            EnterLogTable();
            string msg = string.Format(message, parameters);
            msg = msg.Replace("&", "&amp;");
            msg = msg.Replace("<", "&lt;");
            msg = msg.Replace(">", "&gt;");
            TestReport.WriteLine(HtmlResource.AssertionLogRowFormat, DateTime.Now, msg);
        }

        /// <summary>
        /// The unit testing framework calls this method after the test
        /// completes (regardless of whether the test passed, failed or was
        /// inconclusive).
        ///
        /// This method writes the contents of the audit tables and records of
        /// uploaded and downloaded documents into the test report stream,
        /// replaces the keys for the finish time and outcome, and writes the
        /// report to a file.
        /// </summary>
        [TestCleanup()]
        public void FinishTestReportFile()
        {
            LogDatabaseRecords();
            TestReport.WriteLine(HtmlResource.BodyEnd);
            TestReport.WriteLine(HtmlResource.HtmlEnd);
            TestReport.Flush();

            // Replace Keys
            StringBuilder sb = TestReport.GetStringBuilder();
            sb.Replace(TEST_FINISHED_KEY, string.Format(HtmlResource.DateTimeFormat, DateTime.Now));
            sb.Replace(TEST_OUTCOME_KEY, string.Format(HtmlResource.TestOutcomeFormat, TestContext.CurrentTestOutcome));

            // Write File
            string testReportPath = string.Format(@"{0}\{1}.html", ResultsDirectory, TestContext.TestName);
            StreamWriter testReportFile = new StreamWriter(testReportPath);
            testReportFile.Write(sb);
            testReportFile.Flush();

            TestReport.Close();
            testReportFile.Close();

            // Clean up state for next test
            testReportField = null;
        }

        #endregion Public Methods

        #region Private Methods

        /// <summary>
        /// Writes the HTML headers and the Test Overview table.
        /// The items that won't be known until the test has finished are written with a key
        /// and that key is replaced at the end.
        /// </summary>
        private void WriteTestReportHeader()
        {
            TestReport.WriteLine(HtmlResource.HtmlStart);
            TestReport.WriteLine(HtmlResource.HeadStart);
            TestReport.WriteLine(HtmlResource.TitleElementFormat, TestContext.TestName);
            TestReport.WriteLine(HtmlResource.StyleElement);
            TestReport.WriteLine(HtmlResource.HeadEnd);
            TestReport.WriteLine(HtmlResource.BodyStart);
            TestReport.WriteLine(HtmlResource.HeaderTestOverview);
            TestReport.WriteLine(HtmlResource.TableStart);
            TestReport.WriteLine("<tr><th>Test Name:</th><td>{0}</td></tr>", TestContext.TestName);
            TestReport.WriteLine("<tr><th>Test Context:</th><td>{0}</td></tr>", TestContext.FullyQualifiedTestClassName);
            TestReport.WriteLine("<tr><th>Test Started:</th><td>{0:dd/MM/yyyy HH:mm:ss}</td></tr>", DateTime.Now);
            TestReport.WriteLine("<tr><th>Test Finished:</th><td>{0}</td></tr>", TEST_FINISHED_KEY);
            TestReport.WriteLine("<tr><th>Test Outcome:</th><td>{0}</td></tr>", TEST_OUTCOME_KEY);
            TestReport.WriteLine(HtmlResource.TableEnd);
        }

        private void EnterLogTable()
        {
            if (!InLogTable)
            {
                InLogTable = true;
                TestReport.WriteLine("<h3>Test Assertion Log</h3>");
                TestReport.WriteLine(HtmlResource.TableStart);
                TestReport.WriteLine("<tr><th>Date/Time</th><th>Message</th></tr>");
            }
        }

        private void LeaveLogTable()
        {
            if (InLogTable)
            {
                InLogTable = false;
                TestReport.WriteLine(HtmlResource.TableEnd);
            }
        }

        /// <summary>
        /// Constructs a relative link to a file in the current directory.
        /// </summary>
        /// <param name="path">Full path to file</param>
        /// <param name="text">Text for link</param>
        /// <returns></returns>
        private string Link(string path, string text)
        {
            path = Uri.EscapeUriString(new FileInfo(path).Name);
            return string.Format(HtmlResource.LinkFormat, path, text);
        }

        /// <summary>
        /// Logs the PCEHR Audit records.
        /// </summary>
        /// <param name="audits"></param>
        private void Log(List<PcehrAudit> audits)
        {
            LeaveLogTable();
            TestReport.WriteLine("<h3>PCEHR Service Audit Log</h3>");
            if (audits.Count == 0)
            {
                TestReport.WriteLine("<p>No PCEHR services were invoked in this test run.</p>");
                return;
            }
            TestReport.WriteLine(HtmlResource.TableStart);
            TestReport.WriteLine("<tr><th>PCEHR Audit ID</th><th>Patient Master ID</th><th>Service Name</th><th>Service Message</th><th>IHI</th><th>HPI-O</th><th>Date/Time Audited</th><th>Request</th><th>Response</th></tr>");
            foreach (PcehrAudit audit in audits)
            {
                string requestPath = string.Format(@"{0}\PCEHR_{1}_Request.xml", ResultsDirectory, audit.PcehrAuditId);
                string responsePath = string.Format(@"{0}\PCEHR_{1}_Response.xml", ResultsDirectory, audit.PcehrAuditId);
                File.WriteAllText(requestPath, audit.Request);
                File.WriteAllText(responsePath, audit.Response);
                TestReport.WriteLine("<tr><td>{0}</td><td>{1}</td><td>{2}</td><td>{3}</td><td>{4}</td><td>{5}</td><td>{6:dd/MM/yyyy HH:mm:ss}</td><td>{7}</td><td>{8}</td></tr>",
                    audit.PcehrAuditId, audit.PatientMasterId, audit.ServiceName, audit.ServiceMessage, audit.Ihi, audit.HpiO, audit.DateCreated, Link(requestPath, "Request"), Link(responsePath, "Response"));
            }
            TestReport.WriteLine(HtmlResource.TableEnd);
        }

        /// <summary>
        /// Logs the IHI audits
        /// </summary>
        /// <param name="audits"></param>
        private void Log(List<IhiLookupAudit> audits)
        {
            LeaveLogTable();
            TestReport.WriteLine("<h3>IHI Lookup Audit Log</h3>");
            if (audits.Count == 0)
            {
                TestReport.WriteLine("<p>No IHI lookups occurred in this test run.</p>");
                return;
            }
            TestReport.WriteLine(HtmlResource.TableStart);
            TestReport.WriteLine("<tr><th>IHI Lookup Audit ID</th><th>Patient Master ID</th><th>Family Name</th><th>Given Name</th><th>DOB</th><th>Sex</th><th>Message</th><th>IHI</th><th>IHI Status</th><th>IHI Record Status</th><th>Date/Time Audited</th><th>Request</th><th>Response</th></tr>");
            foreach (IhiLookupAudit audit in audits)
            {
                string requestPath = string.Format(@"{0}\HI_{1}_Request.xml", ResultsDirectory, audit.IhiLookupAuditId);
                string responsePath = string.Format(@"{0}\HI_{1}_Response.xml", ResultsDirectory, audit.IhiLookupAuditId);
                File.WriteAllText(requestPath, audit.Request);
                File.WriteAllText(responsePath, audit.Response);
                TestReport.WriteLine("<tr><td>{0}</td><td>{1}</td><td>{2}</td><td>{3}</td><td>{4:dd/MM/yyyy}</td><td>{5}</td><td>{6}</td><td>{7}</td><td>{8}</td><td>{9}</td><td>{10:dd/MM/yyyy HH:mm:ss}</td><td>{11}</td><td>{12}</td></tr>",
                    audit.IhiLookupAuditId, audit.PatientMasterId, audit.FamilyName, audit.GivenName, audit.DateOfBirth, audit.SexID, audit.Message, audit.IhiNumber, audit.IhiStatus, audit.IhiRecordStatus, audit.DateCreated, Link(requestPath, "Request"), Link(responsePath, "Response"));
            }
            TestReport.WriteLine(HtmlResource.TableEnd);
        }

        /// <summary>
        /// Logs the HPII audits
        /// </summary>
        /// <param name="audits"></param>
        private void Log(HpiiLookupAudit audit)
        {
            LeaveLogTable();
            TestReport.WriteLine("<h3>HPII Lookup Audit Log</h3>");
            if (audit == null)
            {
                TestReport.WriteLine("<p>No HPII lookups occurred in this test run.</p>");
                return;
            }
            TestReport.WriteLine(HtmlResource.TableStart);
            TestReport.WriteLine("<tr><th>HPII Lookup Audit ID</th><th>Family Name</th><th>Given Name</th><th>DOB</th><th>Sex</th><th>Message</th><th>HPII</th><th>HPII Status</th><th>Registration Id</th><th>Date/Time Audited</th><th>Request</th><th>Response</th></tr>");
            //foreach (HpiiLookupAudit audit in audits)
            //{
            string requestPath = string.Format(@"{0}\HI_{1}_Request.xml", ResultsDirectory, audit.HpiiLookupAuditId);
            string responsePath = string.Format(@"{0}\HI_{1}_Response.xml", ResultsDirectory, audit.HpiiLookupAuditId);
            File.WriteAllText(requestPath, audit.Request);
            File.WriteAllText(responsePath, audit.Response);
            TestReport.WriteLine("<tr><td>{0}</td><td>{1}</td><td>{2}</td><td>{3:dd/MM/yyyy}</td><td>{4}</td><td>{5}</td><td>{6}</td><td>{7}</td><td>{8}</td><td>{9:dd/MM/yyyy HH:mm:ss}</td><td>{10}</td><td>{11}</td></tr>",
                audit.HpiiLookupAuditId, audit.FamilyName, audit.GivenNames, audit.DateOfBirth, audit.Sex, audit.Message, audit.HpiiNumber, audit.HpiiStatusId, audit.RegistrationId, audit.DateCreated, Link(requestPath, "Request"), Link(responsePath, "Response"));
            //}
            TestReport.WriteLine(HtmlResource.TableEnd);
        }

        /// <summary>
        /// Logs the HPII audits
        /// </summary>
        /// <param name="audit">The audit.</param>
        private void LogBatch(HpiiLookupAudit audit)
        {
            LeaveLogTable();
            TestReport.WriteLine("<h3>HPII Lookup Batch Audit Log</h3>");
            if (audit == null)
            {
                TestReport.WriteLine("<p>No HPII lookups occurred in this test run.</p>");
                return;
            }
            TestReport.WriteLine(HtmlResource.TableStart);
            TestReport.WriteLine("<tr><th>HPII Lookup Audit ID</th><th>BatchIdentifier</th><th>Message</th><th>HPIO</th><th>Date/Time Audited</th><th>Request Message Id</th><th>Request</th><th>Response Message Id</th><th>Response</th></tr>");
            //foreach (HpiiLookupAudit audit in audits)
            //{
            string requestPath = string.Format(@"{0}\HI_{1}_Request.xml", ResultsDirectory, audit.HpiiLookupAuditId);
            string responsePath = string.Format(@"{0}\HI_{1}_Response.xml", ResultsDirectory, audit.HpiiLookupAuditId);
            File.WriteAllText(requestPath, audit.Request);
            File.WriteAllText(responsePath, audit.Response);
            TestReport.WriteLine("<tr><td>{0}</td><td>{1}</td><td>{2}</td><td>{3}</td><td>{4:dd/MM/yyyy}</td><td>{5}</td><td>{6}</td><td>{7}</td><td>{8}</td></tr>",
                audit.HpiiLookupAuditId, audit.BatchId, audit.Message, audit.HpiO, audit.DateCreated, audit.ServiceMessageRequestID, Link(requestPath, "Request"), audit.ServiceMessageResponseID, Link(responsePath, "Response"));
            //}
            TestReport.WriteLine(HtmlResource.TableEnd);
        }

        /// <summary>
        /// Logs the consent audits.
        /// </summary>
        /// <param name="audits"></param>
        private void Log(List<ConsentAudit> audits)
        {
            LeaveLogTable();
            TestReport.WriteLine("<h3>Consent Audit Log</h3>");
            if (audits.Count == 0)
            {
                TestReport.WriteLine("<p>No withdrawals of consent occurred in this test run.</p>");
                return;
            }
            TestReport.WriteLine(HtmlResource.TableStart);
            TestReport.WriteLine("<tr><th>Consent Audit ID</th><th>Action</th><th>Audit Information</th><th>Date Audited</th></tr>");
            foreach (ConsentAudit audit in audits)
            {
                string action = audit.ConsentWithdrawn ? "Withdrawn" : "Restored";

                // Assume that the audit info was a UTF-8 encoded string
                // For the scripted CCA tests here it will be a UTF-8 string, even though
                // in the general case it could store a scanned form (e.g. JPG, TIFF or PDF).
                string auditInfo = UTF8Encoding.UTF8.GetString(audit.AuditInformation);
                TestReport.WriteLine("<tr><td>{0}</td><td>{1}</td><td>{2}</td><td>{3:dd/MM/yyyy HH:mm:ss}</td></tr>",
                    audit.ConsentAuditId, action, auditInfo, audit.DateCreated);
            }
            TestReport.WriteLine(HtmlResource.TableEnd);
        }

        /// <summary>
        /// Logs the uploaded clinical documents.
        /// </summary>
        /// <param name="documents"></param>
        private void Log(List<ClinicalDocument> documents)
        {
            LeaveLogTable();
            TestReport.WriteLine("<h3>Uploaded Clinical Documents</h3>");
            if (documents.Count == 0)
            {
                TestReport.WriteLine("<p>No uploaded documents</p>");
                return;
            }
            TestReport.WriteLine(HtmlResource.TableStart);
            TestReport.WriteLine("<tr><th>Clinical Document ID</th><th>Set ID</th><th>Date/Time Uploaded</th><th>Type</th><th>Status</th><th>Removal Reason</th><th>Date/Time Removed</th></tr>");
            foreach (ClinicalDocument document in documents)
            {
                TestReport.WriteLine("<tr><td>{0}</td><td>{1}</td><td>{2:dd/MM/yyyy HH:mm:ss}</td><td>{3}</td><td>{4}</td><td>{5}</td><td>{6:dd/MM/yyyy HH:mm:ss}</td></tr>",
                    document.ClinicalDocumentId,
                    document.SourceSystemSetId,
                    document.DateCreated,
                    document.DocumentTypeDescription,
                    (HIPS.PcehrDataStore.Schemas.Enumerators.ClinicalDocumentStatus)document.ClinicalDocumentStatusId,
                    document.RemovalReasonDescription,
                    document.RemovedDate);
            }
            TestReport.WriteLine(HtmlResource.TableEnd);
        }

        /// <summary>
        /// Logs the uploaded clinical document versions.
        /// </summary>
        /// <param name="versions"></param>
        private void Log(List<ClinicalDocumentVersion> versions)
        {
            LeaveLogTable();
            TestReport.WriteLine("<h3>Uploaded Clinical Document Versions</h3>");
            TestReport.WriteLine(HtmlResource.TableStart);
            TestReport.WriteLine("<tr><th>Clinical Document Version ID</th><th>Clinical Document ID</th><th>Document ID</th><th>Date/Time Uploaded</th><th>Date/Time Superseded</th><th>Package</th></tr>");
            foreach (ClinicalDocumentVersion version in versions)
            {
                string packagePath = string.Format(@"{0}\Clinical_Document_Version_{1}_Package.zip", ResultsDirectory, version.ClinicalDocumentVersionId);
                File.WriteAllBytes(packagePath, version.Package);
                TestReport.WriteLine("<tr><td>{0}</td><td>{1}</td><td>{2}</td><td>{3:dd/MM/yyyy HH:mm:ss}</td><td>{4:dd/MM/yyyy HH:mm:ss}</td><td>{5}</td></tr>",
                    version.ClinicalDocumentVersionId,
                    version.ClinicalDocumentId,
                    version.SourceSystemDocumentId,
                    version.UploadedDate,
                    version.SupersededDate,
                    Link(packagePath, "Package"));
            }
            TestReport.WriteLine(HtmlResource.TableEnd);
        }

        /// <summary>
        /// Logs the downloaded documents that were saved into the local patient record.
        /// </summary>
        /// <param name="documents"></param>
        private void Log(List<DownloadedDocument> documents)
        {
            LeaveLogTable();
            TestReport.WriteLine("<h3>Downloaded Clinical Documents</h3>");
            if (documents.Count == 0)
            {
                TestReport.WriteLine("<p>No documents were saved after being downloaded in this test run.</p>");
                return;
            }
            TestReport.WriteLine(HtmlResource.TableStart);
            TestReport.WriteLine("<tr><th>Downloaded Document ID</th><th>Document ID</th><th>Date/Time Downloaded</th><th>Package</th></tr>");
            foreach (DownloadedDocument document in documents)
            {
                string packagePath = string.Format(@"{0}\Downloaded_Document_{1}_Package.zip", ResultsDirectory, document.DownloadedDocumentId);
                File.WriteAllBytes(packagePath, document.Package);
                TestReport.WriteLine("<tr><td>{0}</td><td>{1}</td><td>{2:dd/MM/yyyy HH:mm:ss}</td><td>{3}</td></tr>",
                    document.DownloadedDocumentId,
                    document.SourceSystemDocumentId,
                    document.DownloadedDate,
                    Link(packagePath, "Package"));
            }
            TestReport.WriteLine(HtmlResource.TableEnd);
        }

        #endregion Private Methods
    }
}