﻿// -----------------------------------------------------------------------
// <copyright file="LogAssert.cs" company="NEHTA">
// Developed by Chamonix for NEHTA.
// </copyright>
// -----------------------------------------------------------------------

namespace Test.MedicareCca.Helpers
{
    using System;
    using System.Linq;
    using System.Text.RegularExpressions;
    using HIPS.CommonSchemas;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Test.MedicareCCA.Helpers;

    /// <summary>
    /// TODO: Update summary.
    /// </summary>
    public class LogAssert
    {
        private CcaTest test;
        private TestContext context;
        private readonly Regex numericalRegex = new Regex(@"^\d+$");

        public LogAssert(CcaTest test)
        {
            this.test = test;
            this.context = test.TestContext;
        }

        /// <summary>
        /// Asserts that the actual HipsResponse from the HIPS service has the expected HipsResponseIndicator.
        /// </summary>
        /// <param name="expected">The expected HipsResponseIndicator enumeration value.</param>
        /// <param name="actual">The actual HipsResponse object.</param>
        /// <param name="serviceName">The name of the HIPS service that was called.</param>
        public void ExpectResponse(HipsResponseIndicator expected, HipsResponse actual, string serviceName)
        {
            if (actual.Status == expected)
            {
                test.Log("Pass: HIPS {0} returned status {1}", serviceName, actual);
            }
            else
            {
                test.Log("Fail: HIPS {0} return status was not {1} but rather {2}", serviceName, expected, actual);
            }
            Assert.AreEqual(expected, actual.Status);
        }

        /// <summary>
        /// Asserts that the actual value of the named item is equal to the expected value for that item.
        /// Use this overload when the source of the expected item is obvious and does not need to be logged.
        /// </summary>
        /// <param name="expected"></param>
        /// <param name="actual"></param>
        /// <param name="itemName"></param>
        /// <param name="parameters"></param>
        public void AreEqual(object expected, object actual, string itemName)
        {
            if (object.Equals(expected, actual))
            {
                test.Log("Pass: {0} was {1}", itemName, actual);
            }
            else
            {
                test.Log("Fail: {0} was not {1} but instead {2}", itemName, expected, actual);
            }
            Assert.AreEqual(expected, actual);
        }

        /// <summary>
        /// Asserts that the actual value of the named item is equal to the expected value for that item.
        /// Use this overload when the sources of the actual and expected items should both be logged.
        /// </summary>
        /// <param name="expected">The expected item</param>
        /// <param name="actual">The actual item</param>
        /// <param name="expectedName">The source of the expected item</param>
        /// <param name="actualName">The source of the actual item</param>
        public void AreEqual(object expected, object actual, string expectedName, string actualName)
        {
            if (object.Equals(expected, actual))
            {
                test.Log("Pass: {0} and {1} were both {2}", actualName, expectedName, actual);
            }
            else
            {
                test.Log("Fail: {0} was {1} but {2} was {3}", actualName, actual, expectedName, expected);
            }
            Assert.AreEqual(expected, actual);
        }

        /// <summary>
        /// Asserts that the actual value of the named item is not equal to a certain value.
        /// </summary>
        /// <param name="expected"></param>
        /// <param name="actual"></param>
        /// <param name="itemName"></param>
        /// <param name="parameters"></param>
        public void AreNotEqual(object expected, object actual, string itemName, params object[] parameters)
        {
            if (!object.Equals(expected, actual))
            {
                test.Log("Pass: {0} was not {1} but instead {2}", string.Format(itemName, parameters), expected, actual);
            }
            else
            {
                test.Log("Fail: {0} was {1}", string.Format(itemName, parameters), actual);
            }
            Assert.AreNotEqual(expected, actual);
        }

        /// <summary>
        /// Asserts that a condition is false. Logs one message if this assertion passes and another if the assertion fails.
        /// </summary>
        /// <param name="condition"></param>
        /// <param name="passMessage"></param>
        /// <param name="failMessage"></param>
        public void IsFalse(bool condition, string passMessage, string failMessage)
        {
            if (condition)
            {
                test.Log("Fail: {0}", failMessage);
            }
            else
            {
                test.Log("Pass: {0}", passMessage);
            }
            Assert.IsFalse(condition, failMessage);
        }

        /// <summary>
        /// Asserts that a condition is true. Logs one message if this assertion passes and another if the assertion fails.
        /// </summary>
        /// <param name="condition"></param>
        /// <param name="passMessage"></param>
        /// <param name="failMessage"></param>
        public void IsTrue(bool condition, string passMessage, string failMessage)
        {
            if (condition)
            {
                test.Log("Pass: {0}", passMessage);
            }
            else
            {
                test.Log("Fail: {0}", failMessage);
            }
            Assert.IsTrue(condition, failMessage);
        }

        /// <summary>
        /// Unconditionally fails the test with a message explaining what has gone wrong.
        /// </summary>
        /// <param name="message"></param>
        /// <param name="parameters"></param>
        public void Fail(string message, params object[] parameters)
        {
            test.Log("Fail: {0}", string.Format(message, parameters));
            Assert.Fail(message, parameters);
        }

        /// <summary>
        /// Asserts that the actual value of the named binary item is equal to the expected value for that item.
        /// </summary>
        /// <param name="expected"></param>
        /// <param name="actual"></param>
        /// <param name="passMessage"></param>
        /// <param name="failMessage"></param>
        public void AreEqualBinary(byte[] expected, byte[] actual, string passMessage, string failMessage)
        {
            int expectedLength = expected.Length;
            int actualLength = actual.Length;
            if (expectedLength != actualLength)
            {
                test.Log("Fail: {0}", failMessage);
                Assert.Fail(failMessage);
            }
            for (int i = 0; i < expectedLength; i++)
            {
                if (expected[i] != actual[i])
                {
                    test.Log("Fail: {0}", failMessage);
                    Assert.Fail(failMessage);
                }
            }
            test.Log("Pass: {0}", passMessage);
        }

        /// <summary>
        /// Asserts that the actual value of the named binary item is not equal to a certain value.
        /// </summary>
        /// <param name="expected"></param>
        /// <param name="actual"></param>
        /// <param name="passMessage"></param>
        /// <param name="failMessage"></param>
        public void AreNotEqualBinary(byte[] expected, byte[] actual, string passMessage, string failMessage)
        {
            int expectedLength = expected.Length;
            int actualLength = actual.Length;
            if (expectedLength != actualLength)
            {
                test.Log("Pass: {0}", passMessage);
                return;
            }
            for (int i = 0; i < expectedLength; i++)
            {
                if (expected[i] != actual[i])
                {
                    test.Log("Pass: {0}", passMessage);
                    return;
                }
            }
            test.Log("Fail: {0}", failMessage);
            Assert.Fail(failMessage);
        }

        /// <summary>
        /// Asserts that the result of the test is inconclusive because of the reason specified in the message.
        /// </summary>
        /// <param name="message"></param>
        public void Inconclusive(string message)
        {
            test.Log(message);
            Assert.Inconclusive(message);
        }

        /// <summary>
        /// Asserts that the actual value for the item with name itemName is one of the values in the expected array.
        /// </summary>
        /// <param name="expected"></param>
        /// <param name="actual"></param>
        /// <param name="itemName"></param>
        public void Contains(string[] expected, string actual, string itemName)
        {
            if (expected.Contains(actual))
            {
                string message = string.Format("Pass: {0} was {1} which is {2}", itemName, actual, string.Join(" or ", expected));
                test.Log(message);
            }
            else
            {
                string message = string.Format("Fail: {0} was {1} which is not {2}", itemName, actual, string.Join(" or ", expected));
                test.Log(message);
                Assert.Fail(message);
            }
        }

        /// <summary>
        /// Asserts that the actual value for the item with name itemName is between the minimumExpected and the maximumExpected values.
        /// </summary>
        /// <param name="minimumExpected"></param>
        /// <param name="maximumExpected"></param>
        /// <param name="actual"></param>
        /// <param name="itemName"></param>
        public void IsBetween(IComparable minimumExpected, IComparable maximumExpected, IComparable actual, string itemName)
        {
            if (actual.CompareTo(minimumExpected) >= 0 && actual.CompareTo(maximumExpected) <= 0)
            {
                string message = string.Format("Pass: {0} was {1} which is between {2} and {3}", itemName, actual, minimumExpected, maximumExpected);
                test.Log(message);
            }
            else
            {
                string message = string.Format("Fail: {0} was {1} which is not between {2} and {3}", itemName, actual, minimumExpected, maximumExpected);
                test.Log(message);
                Assert.Fail(message);
            }
        }

        /// <summary>
        /// Asserts that the actual value for the item with name itemName is not null.
        /// </summary>
        /// <param name="actual"></param>
        /// <param name="itemName"></param>
        public void IsNotNull(object actual, string itemName)
        {
            string message;
            if (actual == null)
            {
                message = string.Format("Fail: {0} was null", itemName);
            }
            else
            {
                message = string.Format("Pass: {0} was not null", itemName);
            }
            test.Log(message);
            Assert.IsNotNull(actual, message);
        }

        /// <summary>
        /// Asserts that the actual value for the item with name itemName has no leading or trailing spaces and is not blank.
        /// </summary>
        /// <param name="actual">The actual value</param>
        /// <param name="itemName">The name of the item</param>
        public void HasNoLeadingOrTrailingSpacesAndIsNotBlank(string actual, string itemName)
        {
            if (actual.Trim() != actual)
            {
                string message = string.Format("Fail: {0} '{1}' has leading or trailing spaces", itemName, actual);
                test.Log(message);
                Assert.Fail(message);
            }
            else if (actual.Length == 0)
            {
                string message = string.Format("Fail: {0} is blank", itemName);
                test.Log(message);
                Assert.Fail(message);
            }
            else
            {
                string message = string.Format("Pass: {0} '{1}' has no leading or trailing spaces and is not blank", itemName, actual);
                test.Log(message);
            }
        }

        /// <summary>
        /// Asserts that the actual value for the item with name itemName is numerical.
        /// </summary>
        /// <param name="actual">The actual value</param>
        /// <param name="itemName">The name of the item</param>
        public void IsNumerical(string actual, string itemName)
        {
            if (numericalRegex.IsMatch(actual))
            {
                string message = string.Format("Pass: {0} '{1}' is numerical", itemName, actual);
                test.Log(message);
            }
            else
            {
                string message = string.Format("Fail: {0} '{1}' is not numerical", itemName, actual);
                test.Log(message);
                Assert.Fail(message);
            }
        }

        /// <summary>
        /// Asserts that the actual value for the item with name itemName is not an empty string.
        /// </summary>
        /// <param name="actual">The actual value</param>
        /// <param name="itemName">The name of the item</param>
        internal void IsNotBlank(string actual, string itemName)
        {
            if (actual.Length == 0)
            {
                string message = string.Format("Fail: {0} is blank", itemName);
                test.Log(message);
                Assert.Fail(message);
            }
            else
            {
                string message = string.Format("Pass: {0} '{1}' is not blank", itemName, actual);
                test.Log(message);
            }
        }
    }
}