﻿using System;
using HIPS.Base.Schemas;
using HIPS.CommonSchemas;
using HIPS.HibIntegration.HL7;
using HIPS.HibIntegration.HL7.Message;
using HIPS.HibIntegration.Loader;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.Helpers;

namespace Test.HibIntegration
{
    /// <summary>
    ///This is a test class for PasLoaderTest and is intended
    ///to contain all PasLoaderTest Unit Tests
    ///</summary>
    [TestClass()]
    public class PasLoaderTest
    {
        private PatientTestHelpersBase helpers = new PatientTestHelpersBase();

        /// <summary>
        /// Gets the Helpers.
        /// </summary>
        private PatientTestHelpersBase Helpers
        {
            get
            {
                if (this.helpers == null)
                {
                    helpers = new PatientTestHelpersBase();
                }
                return helpers;
            }
        }

        #region Constructors

        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        #endregion Constructors

        #region Additional test attributes

        //
        //You can use the following additional attributes as you write your tests:
        //
        //Use ClassInitialize to run code before running the first test in the class
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //Use ClassCleanup to run code after all tests in a class have run
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //Use TestInitialize to run code before running each test
        //[TestInitialize()]
        //public void MyTestInitialize()
        //{
        //}
        //
        //Use TestCleanup to run code after each test has run
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //

        #endregion Additional test attributes

        #region Test Methods

        /// <summary>
        ///A test to see if the PAS Loader runs with
        ///</summary>
        [TestMethod()]
        public void ProcessGeneralLoaderTest()
        {
            try
            {
                SystemErrorLog errorLog = Helpers.GetLastErrorMessage();
                HL7Message newPatientMessage = HL7Message.Parse(HL7MessageResource.Message1);
                HL7GenericPasMessage pasMessage = newPatientMessage as HL7GenericPasMessage;
                MessageLoader pasLoader = new MessageLoader();
                HL7MessageLog log = new HL7MessageLog();
                UserDetails user = GetUser();
                pasLoader.Process(pasMessage, pasMessage.ToString(), user, out log);
                Helpers.AnySystemErrorsOccur(errorLog);
            }
            catch (Exception ex)
            {
                if (!ex.Message.StartsWith(" - (INFORMATION) No Healthcare Individual records have been found."))
                {
                    Assert.Fail(string.Format("The process did not complete - {0}", ex.Message));
                }
            }
        }

        /// <summary>
        ///A test to see if the PAS Loader runs with a new outpatient appointment message.
        ///</summary>
        [TestMethod()]
        public void ProcessNewOutpatientAppointmentTest()
        {
            try
            {
                SystemErrorLog errorLog = Helpers.GetLastErrorMessage();
                HL7Message newPatientMessage = HL7Message.Parse(HL7MessageResource.NewOutpatientAppointment);
                HL7GenericPasMessage pasMessage = newPatientMessage as HL7GenericPasMessage;
                MessageLoader pasLoader = new MessageLoader();
                HL7MessageLog log = new HL7MessageLog();
                UserDetails user = GetUser();
                pasLoader.Process(pasMessage, pasMessage.ToString(), user, out log);
                Helpers.AnySystemErrorsOccur(errorLog);
            }
            catch (Exception ex)
            {
                if (!ex.Message.StartsWith(" - (INFORMATION) No Healthcare Individual records have been found."))
                {
                    Assert.Fail(string.Format("The process did not complete - {0}", ex.Message));
                }
            }
        }

        #endregion Test Methods

        #region CCA Test Methods HI_005_005807 (9)

        /// <summary>
        /// Process_s the H i_005_005808_a test.
        /// </summary>
        [TestMethod()]
        public void Process_HI_005_005808_aTest()
        {
            TestDateTimeValidation(HL7MessageResource.HI_005_005808_a);
        }

        [TestMethod()]
        public void Process_HI_005_005808_bTest()
        {
            TestDateTimeValidation(HL7MessageResource.HI_005_005808_b);
        }

        [TestMethod()]
        public void Process_HI_005_005808_cTest()
        {
            TestDateTimeValidation(HL7MessageResource.HI_005_005808_c);
        }

        [TestMethod()]
        public void Process_HI_005_005808_dTest()
        {
            TestDateTimeValidation(HL7MessageResource.HI_005_005808_d);
        }

        [TestMethod()]
        public void Process_HI_005_005808_eTest()
        {
            TestDateTimeValidation(HL7MessageResource.HI_005_005808_e);
        }

        [TestMethod()]
        public void Process_HI_005_005808_e1Test()
        {
            TestDateTimeValidation(HL7MessageResource.HI_005_005808_e1);
        }

        [TestMethod()]
        public void Process_HI_005_005808_e2Test()
        {
            TestDateTimeValidation(HL7MessageResource.HI_005_005808_e2);
        }

        [TestMethod()]
        public void Process_HI_005_005808_e3Test()
        {
            TestDateTimeValidation(HL7MessageResource.HI_005_005808_e3);
        }

        [TestMethod()]
        public void Process_HI_005_005808_fTest()
        {
            TestDateTimeValidation(HL7MessageResource.HI_005_005808_f, false);
        }

        #endregion CCA Test Methods HI_005_005807 (9)

        #region CCA Test Methods HI_005_005807

        [TestMethod()]
        public void Process_HI_005_005807a_Test()
        {
            TestMedicareValidation(HL7MessageResource.HI_005_005807a, "TEST-CCA_85", null, null, false, true);
            TestMedicareValidation(HL7MessageResource.HI_005_005807a, "TEST-CCA_85", "2850250921", "1", false, false);
        }

        [TestMethod()]
        public void Process_HI_005_005807a2_Test()
        {
            TestMedicareValidation(HL7MessageResource.HI_005_005807a2, "TEST-CCA_85", null, null, false, true);
            TestMedicareValidation(HL7MessageResource.HI_005_005807a2, "TEST-CCA_85", "295025O921", "1", false, false);
        }

        [TestMethod()]
        public void Process_HI_005_005807b_Test()
        {
            TestMedicareValidation(HL7MessageResource.HI_005_005807b, "TEST-CCA_85", null, null, false, true);
            TestMedicareValidation(HL7MessageResource.HI_005_005807b, "TEST-CCA_85", "2950250921", "1", true, false);
        }

        #endregion CCA Test Methods HI_005_005807

        #region CCA Test Methods HI_005_005804

        [TestMethod()]
        public void Process_HI_005_005804a_Test()
        {
            Helpers.DeletePatientByMrn("TEST-CCA_42");
            TestName(HL7MessageResource.HI_005_005804a, "TEST-CCA_42", "Parker", "Liana Winola Jessamine", "Parker", "Winola");
        }

        [TestMethod()]
        public void Process_HI_005_005804b_Test()
        {
            Helpers.DeletePatientByMrn("TEST-CCA_42");
            ProcessMessage(HL7MessageResource.HI_005_005804a);
            TestName(HL7MessageResource.HI_005_005804b, "TEST-CCA_42", "Smith", "Winola Jessamine Liana", "Parker", "Winola");
        }

        #endregion CCA Test Methods HI_005_005804

        #region CCA Test Methods HI_010_005805

        [TestMethod()]
        public void Process_HI_010_005805a_Test()
        {
            string mrn = "TEST-CCA_01";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_010_005805a);
            InsertCca01Ihi();

            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_010_005805a);
            IhiLookupAudit latestAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiLookupAuditId + 2 != latestAudit.IhiLookupAuditId)
            {
                Assert.Fail("Audit results are not as expected");
            }
            if (latestAudit.FamilyName.Length != 40)
            {
                Assert.Fail("Family Name is not 40 characters long");
            }
            if (!latestAudit.FamilyName.EndsWith("GHIJ"))
            {
                Assert.Fail("Family name does not end with 'GHIJ'");
            }
        }

        [TestMethod()]
        public void Process_HI_010_005805b_Test()
        {
            string mrn = "TEST-CCA_01";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_010_005805a);
            InsertCca01Ihi();

            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_010_005805a);
            IhiLookupAudit latestAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiLookupAuditId + 2 != latestAudit.IhiLookupAuditId)
            {
                Assert.Fail("Audit results are not as expected");
            }
            if (latestAudit.GivenName.Length != 40)
            {
                Assert.Fail("Given name is not 40 characters long");
            }
            if (!latestAudit.GivenName.EndsWith("FGHI"))
            {
                Assert.Fail("Given name does not end with 'FGHI'");
            }
        }

        [TestMethod()]
        public void Process_HI_010_005805c_Test()
        {
            string mrn = "TEST-CCA_01";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_010_005805a);
            InsertCca01Ihi();

            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_010_005805a);
            IhiLookupAudit latestAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiLookupAuditId + 2 != latestAudit.IhiLookupAuditId)
            {
                Assert.Fail("Audit results are not as expected");
            }
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            string[] givenName = patient.CurrentName.GivenNames.Split(' ');
            if (givenName[0].Length != 44)
            {
                Assert.Fail("Current given name is the wrong length");
            }
            if (!givenName[0].EndsWith("0042"))
            {
                Assert.Fail("Current given name does not end with 0042");
            }
            if (patient.CurrentName.FamilyName.Length != 44)
            {
                Assert.Fail("Current family name is the wrong length");
            }
            if (!patient.CurrentName.FamilyName.EndsWith("0042"))
            {
                Assert.Fail("Current family name does not end with 0042");
            }
        }

        [TestMethod()]
        public void Process_HI_010_005805d_Test()
        {
            string mrn = "TEST-CCA_01";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_010_005805a);
            InsertCca01Ihi();

            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_010_005805a);
            IhiLookupAudit latestAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiLookupAuditId + 2 != latestAudit.IhiLookupAuditId)
            {
                Assert.Fail("Audit results are not as expected");
            }
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            string[] givenName = patient.RegisteredGivenName.Split(' ');
            if (givenName[0].Length != 44)
            {
                Assert.Fail("Current given name is the wrong length");
            }
            if (!givenName[0].EndsWith("0042"))
            {
                Assert.Fail("Registered given name does not end with 0042");
            }
            if (patient.RegisteredFamilyName.Length != 44)
            {
                Assert.Fail("Registered family name is the wrong length");
            }
            if (!patient.RegisteredFamilyName.EndsWith("0042"))
            {
                Assert.Fail("Registered family name does not end with 0042");
            }
        }

        #endregion CCA Test Methods HI_010_005805

        #region CCA Test Methods HI_010_005817

        [TestMethod()]
        public void Process_HI_010_005817_Test()
        {
            string mrn = "TEST-CCA_90";
            Helpers.DeletePatientByMrn(mrn);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.PatientMasterId != null)
            {
                Assert.Fail("Patient exists");
            }
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            SystemErrorLog lastLog = Helpers.GetLastErrorMessage();
            ProcessMessage(HL7MessageResource.HI_010_005817);
            Helpers.IhiLookupAlertsOccured(lastAlert, 0);
            Helpers.IhiLookupAuditsOccured(lastAudit, 0);
            Helpers.AnySystemErrorsOccur(lastLog, 0);
            patient = Helpers.GetPatientByMrn(mrn);
            if (patient.PatientMasterId == null)
            {
                Assert.Fail("Patient was not stored correctly");
            }
        }

        #endregion CCA Test Methods HI_010_005817

        #region CCA Test Methods HI_010_005820

        [TestMethod()]
        public void Process_HI_010_005820a_Test()
        {
            PatientMaster patient = SetDataHI010005820();
            if (patient.Ihi != "8003606790010791")
            {
                Assert.Fail("Patient IHI is incorrect");
            }
        }

        [TestMethod()]
        public void Process_HI_010_005820b_Test()
        {
            PatientMaster patient = SetDataHI010005820();
            if (patient.IhiStatusId != (int)IhiStatus.Active)
            {
                Assert.Fail("Patient IHI Status is incorrect");
            }
        }

        [TestMethod()]
        public void Process_HI_010_005820c_Test()
        {
            PatientMaster patient = SetDataHI010005820();
            if ((patient.IhiLastValidated < patient.DateModified) || (patient.IhiLastValidated < DateTime.Now.AddSeconds(-4)))
            {
                Assert.Fail("IHI Last update is incorrect");
            }
        }

        [TestMethod()]
        public void Process_HI_010_005820d_Test()
        {
            PatientMaster patient = SetDataHI010005820();
            if (patient.IhiRecordStatusId != (int)IhiRecordStatus.Verified)
            {
                Assert.Fail("Patient IHI Record Status is incorrect");
            }
        }

        #endregion CCA Test Methods HI_010_005820

        #region CCA Test Methods HI_010_005839

        [TestMethod()]
        public void Process_HI_010_005839_Test()
        {
            string mrn = "TEST-CCA_30";
            Helpers.DeletePatientByMrn(mrn);
            PatientMaster patient = SetDataHI010005820();
            Helpers.SetIhi(patient.PatientMasterId.Value, "8003602346555439");
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            ProcessMessage(HL7MessageResource.HI_010_005839);
            Helpers.IhiLookupAuditsOccured(lastAudit, 1);
            Helpers.IhiLookupAlertsOccured(lastAlert, 1);
            patient = Helpers.GetPatientByMrn(mrn);
            IhiLookupAlert latestAlert = Helpers.GetLastIhiLookupAlert();
            if (latestAlert.Message != "The IHI status has changed to DuplicateIhi. This IHI can not be used.")
            {
                Assert.Fail("Alert does not hold the duplicate message");
            }
            if (patient.Ihi != "8003602346555439")
            {
                Assert.Fail("IHI was not stored");
            }
            if (patient.IhiStatusId != (int)IhiStatus.DuplicateIhi)
            {
                Assert.Fail("IHI was not flagged as a duplicate");
            }
        }

        #endregion CCA Test Methods HI_010_005839

        #region CCA Test Methods HI_010_005873

        #endregion CCA Test Methods HI_010_005873

        #region CCA Test Methods HI_010_008028

        [TestMethod()]
        public void Process_HI_010_008028a_Test()
        {
            string mrn = "TEST-CCA_B1";
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            SetHi010008028Data(mrn);
            Helpers.IhiLookupAuditsOccured(lastAudit, 1);
        }

        [TestMethod()]
        public void Process_HI_010_008028b_Test()
        {
            string mrn = "TEST-CCA_B1";
            SetHi010008028Data(mrn);
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            if (string.IsNullOrEmpty(lastAudit.UserCreated))
            {
                Assert.Fail("User details have not been stored");
            }
        }

        [TestMethod()]
        public void Process_HI_010_008028c_Test()
        {
            string mrn = "TEST-CCA_B1";
            SetHi010008028Data(mrn);
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiNumber != "8003601011850042")
            {
                Assert.Fail("IHI not updated correctly");
            }
        }

        [TestMethod()]
        public void Process_HI_010_008028d_Test()
        {
            string mrn = "TEST-CCA_B1";
            SetHi010008028Data(mrn);
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.Operator.ToUpper() != @"David Johnston (CIO)".ToUpper())
            {
                Assert.Fail("Operator not updated correctly");
            }
        }

        [TestMethod()]
        public void Process_HI_010_008028e_Test()
        {
            string mrn = "TEST-CCA_B1";
            SetHi010008028Data(mrn);
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            if (!string.IsNullOrEmpty(lastAudit.HpiI))
            {
                Assert.Fail("No HPI-I should be saved");
            }
        }

        [TestMethod()]
        public void Process_HI_010_008028f_Test()
        {
            string mrn = "TEST-CCA_B1";
            SetHi010008028Data(mrn);
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.HpiO != "8003624900000594")
            {
                Assert.Fail("HPI-O is incorrect");
            }
        }

        [TestMethod()]
        public void Process_HI_010_008028h_Test()
        {
            string mrn = "TEST-CCA_B1";
            SetHi010008028Data(mrn);
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            if (!lastAudit.Request.Contains(@"<a:Action s:mustUnderstand=""1"">http://ns.electronichealth.net.au/hi/svc/ConsumerSearchIHI/3.0/ConsumerSearchIHIPortType/searchIHIRequest</a:Action>"))
            {
                Assert.Fail("Audit does not hold the request");
            }
        }

        [TestMethod()]
        public void Process_HI_010_008028i_Test()
        {
            string mrn = "TEST-CCA_B1";
            DateTime now = DateTime.Now.AddSeconds(-5);
            SetHi010008028Data(mrn);
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.DateCreated < now)
            {
                Assert.Fail("Audit date is not correct");
            }
        }

        [TestMethod()]
        public void Process_HI_010_008028j_Test()
        {
            string mrn = "TEST-CCA_B1";
            DateTime now = DateTime.Now;
            SetHi010008028Data(mrn);
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            if (!lastAudit.Response.Contains(@"<wsa:Action xmlns:wsa=""http://www.w3.org/2005/08/addressing"">http://ns.electronichealth.net.au/hi/svc/ConsumerSearchIHI/3.0/ConsumerSearchIHIPortType/searchIHIResponse</wsa:Action>"))
            {
                Assert.Fail("Audit does not hold the HI Web Service Version");
            }
        }

        #endregion CCA Test Methods HI_010_008028

        #region CCA Test Methods HI_010_008526

        [TestMethod()]
        public void Process_HI_010_008526a_Test()
        {
            string mrnA = "TEST-CCA_B3";
            string mrnB = "TEST-CCA_B1";
            SetHi010008526Data(mrnA, mrnB);
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_010_008526a);
            Helpers.IhiLookupAlertsOccured(lastAlert, 0);
            Helpers.IhiLookupAuditsOccured(lastAudit, 1);
            IhiLookupAudit latestAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiNumber != "8003603456797233")
            {
                Assert.Fail("IHI Search by Medicare Card Number failed.");
            }
        }

        [TestMethod()]
        public void Process_HI_010_008526b_Test()
        {
            string mrnA = "TEST-CCA_B3";
            string mrnB = "TEST-CCA_B1";
            SetHi010008526Data(mrnA, mrnB);
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_010_008526b);
            Helpers.IhiLookupAlertsOccured(lastAlert, 0);
            Helpers.IhiLookupAuditsOccured(lastAudit, 0);
        }

        [TestMethod()]
        public void Process_HI_010_008526c_Test()
        {
            string mrnA = "TEST-CCA_B3";
            string mrnB = "TEST-CCA_B1";
            SetHi010008526Data(mrnA, mrnB);
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_010_008526c);
            Helpers.IhiLookupAlertsOccured(lastAlert, 2);
            Helpers.IhiLookupAuditsOccured(lastAudit, 2);
        }

        #endregion CCA Test Methods HI_010_008526

        #region CCA Test Methods HI_010_005809

        [TestMethod()]
        public void Process_HI_010_005809_Test()
        {
            string mrn = "TEST-CCA_36";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_010_005809);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.CurrentName.GivenNames.ToUpper() != "Kaili Liza Kameko".ToUpper())
            {
                Assert.Fail("First names are not stored correctly");
            }
        }

        #endregion CCA Test Methods HI_010_005809

        #region CCA Test Methods HI_010_005813

        [TestMethod()]
        public void Process_HI_010_005813_Test()
        {
            string mrn = "TEST-CCA_B3";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_B3);
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.Ihi != "8003603456797233")
            {
                Assert.Fail("IHI was not stored correctly");
            }
            if (!lastAudit.Request.Contains("2950156481"))
            {
                Assert.Fail("The medicare card number was not passed through to the Web Service");
            }
        }

        #endregion CCA Test Methods HI_010_005813

        #region CCA Test Methods HI_010_005814

        [TestMethod()]
        public void Process_HI_010_005814_Test()
        {
            string mrn = "TEST-CCA_16";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_010_005814);
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.Ihi != "8003605678922788")
            {
                Assert.Fail("IHI was not stored correctly");
            }
            if (!lastAudit.Request.Contains("N 909030E"))
            {
                Assert.Fail("The DVA card number was not passed through to the Web Service");
            }
        }

        #endregion CCA Test Methods HI_010_005814

        #region CCA Test Methods HI_010_005830

        [TestMethod()]
        public void Process_HI_010_005830_Test()
        {
            string mrnA = "TEST-CCA_B3";
            string mrnB = "TEST-CCA_34";
            string mrnC = "TEST-CCA_85";
            Helpers.DeletePatientByMrn(mrnA);
            Helpers.DeletePatientByMrn(mrnB);
            Helpers.DeletePatientByMrn(mrnC);
            ProcessMessage(HL7MessageResource.TEST_CCA_B3);
            ProcessMessage(HL7MessageResource.TEST_CCA_34);
            ProcessMessage(HL7MessageResource.TEST_CCA_85);
            PatientMaster patient = Helpers.GetPatientByMrn(mrnA);

            if (patient.MedicareNumber != "2950156481")
            {
                Assert.Fail("Medicare number is incorrect");
            }
            if (!string.IsNullOrEmpty(patient.MedicareIrn))
            {
                Assert.Fail("Medicare IRN should be empty");
            }

            patient = Helpers.GetPatientByMrn(mrnB);

            if (patient.DvaNumber != "N 568908J")
            {
                Assert.Fail("DVA number is incorrect");
            }

            patient = Helpers.GetPatientByMrn(mrnC);

            if (patient.MedicareNumber != "2950250921")
            {
                Assert.Fail("Medicare number is incorrect");
            }
            if (patient.MedicareIrn != "1")
            {
                Assert.Fail("Medicare IRN is not set");
            }
        }

        #endregion CCA Test Methods HI_010_005830

        #region CCA Test Methods HI_010_005917

        [TestMethod()]
        public void Process_HI_010_005917_Test()
        {
            IhiLookupAudit audit = Helpers.GetLastIhiLookupAudit();
            if (string.IsNullOrEmpty(audit.UserCreated))
            {
                Assert.Fail("Audit does not contain an operator");
            }
        }

        #endregion CCA Test Methods HI_010_005917

        #region CCA Test Methods HI_015_005805

        [TestMethod()]
        public void Process_HI_015_005805a_Test()
        {
            string mrn = "TEST-CCA_01";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_01);
            InsertCca01Ihi();

            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.TEST_CCA_01);
            IhiLookupAudit latestAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiLookupAuditId + 2 != latestAudit.IhiLookupAuditId)
            {
                Assert.Fail("Audit results are not as expected");
            }
            if (latestAudit.FamilyName.Length != 40)
            {
                Assert.Fail("Family Name is not 40 characters long");
            }
            if (!latestAudit.FamilyName.EndsWith("GHIJ"))
            {
                Assert.Fail("Family name does not end with 'GHIJ'");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005805b_Test()
        {
            string mrn = "TEST-CCA_01";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_01);
            InsertCca01Ihi();

            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.TEST_CCA_01);
            IhiLookupAudit latestAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiLookupAuditId + 2 != latestAudit.IhiLookupAuditId)
            {
                Assert.Fail("Audit results are not as expected");
            }
            if (latestAudit.GivenName.Length != 40)
            {
                Assert.Fail("Given name is not 40 characters long");
            }
            if (!latestAudit.GivenName.EndsWith("FGHI"))
            {
                Assert.Fail("Given name does not end with 'FGHI'");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005805c_Test()
        {
            string mrn = "TEST-CCA_01";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_01);
            InsertCca01Ihi();

            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.TEST_CCA_01);
            IhiLookupAudit latestAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiLookupAuditId + 2 != latestAudit.IhiLookupAuditId)
            {
                Assert.Fail("Audit results are not as expected");
            }
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            string[] givenName = patient.CurrentName.GivenNames.Split(' ');
            if (givenName[0].Length != 44)
            {
                Assert.Fail("Current given name is the wrong length");
            }
            if (!givenName[0].EndsWith("0042"))
            {
                Assert.Fail("Current given name does not end with 0042");
            }
            if (patient.CurrentName.FamilyName.Length != 44)
            {
                Assert.Fail("Current family name is the wrong length");
            }
            if (!patient.CurrentName.FamilyName.EndsWith("0042"))
            {
                Assert.Fail("Current family name does not end with 0042");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005805d_Test()
        {
            string mrn = "TEST-CCA_01";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_01);
            InsertCca01Ihi();

            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.TEST_CCA_01);
            IhiLookupAudit latestAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiLookupAuditId + 2 != latestAudit.IhiLookupAuditId)
            {
                Assert.Fail("Audit results are not as expected");
            }
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            string[] givenName = patient.RegisteredGivenName.Split(' ');
            if (givenName[0].Length != 44)
            {
                Assert.Fail("Current given name is the wrong length");
            }
            if (!givenName[0].EndsWith("0042"))
            {
                Assert.Fail("Registered given name does not end with 0042");
            }
            if (patient.RegisteredFamilyName.Length != 44)
            {
                Assert.Fail("Registered family name is the wrong length");
            }
            if (!patient.RegisteredFamilyName.EndsWith("0042"))
            {
                Assert.Fail("Registered family name does not end with 0042");
            }
        }

        #endregion CCA Test Methods HI_015_005805

        #region CCA Test Methods HI_015_005807

        /// <summary>
        /// Process_s the H i_005_005808_a test.
        /// </summary>
        [TestMethod()]
        public void Process_HI_015_005808_aTest()
        {
            SetHi015005808Data();
            TestDateTimeValidation(HL7MessageResource.HI_015_005808a);
        }

        [TestMethod()]
        public void Process_HI_015_005808_bTest()
        {
            SetHi015005808Data();
            TestDateTimeValidation(HL7MessageResource.HI_015_005808b);
        }

        [TestMethod()]
        public void Process_HI_015_005808_cTest()
        {
            SetHi015005808Data();
            TestDateTimeValidation(HL7MessageResource.HI_015_005808c);
        }

        [TestMethod()]
        public void Process_HI_015_005808_dTest()
        {
            SetHi015005808Data();
            TestDateTimeValidation(HL7MessageResource.HI_015_005808d);
        }

        [TestMethod()]
        public void Process_HI_015_005808_eTest()
        {
            SetHi015005808Data();
            TestDateTimeValidation(HL7MessageResource.HI_015_005808e);
        }

        [TestMethod()]
        public void Process_HI_015_005808_e1Test()
        {
            SetHi015005808Data();
            TestDateTimeValidation(HL7MessageResource.HI_015_005808e1);
        }

        [TestMethod()]
        public void Process_HI_015_005808_e2Test()
        {
            SetHi015005808Data();
            TestDateTimeValidation(HL7MessageResource.HI_015_005808e2);
        }

        [TestMethod()]
        public void Process_HI_015_005808_e3Test()
        {
            SetHi015005808Data();
            TestDateTimeValidation(HL7MessageResource.HI_015_005808e3);
        }

        [TestMethod()]
        public void Process_HI_015_005808_fTest()
        {
            SetHi015005808Data();
            TestDateTimeValidation(HL7MessageResource.HI_015_005808f, false);
            PatientMaster patient = Helpers.GetPatientByMrn("TEST-CCA_02");
            if (patient.DateOfBirth != new DateTime(1953, 11, 23))
            {
                Assert.Fail("Date is not correct");
            }
        }

        #endregion CCA Test Methods HI_015_005807

        #region CCA Test Methods HI_015_005820

        [TestMethod()]
        public void Process_HI_015_005820a_Test()
        {
            string mrn = "TEST-CCA_23";
            SetHi015005820Data(mrn);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.Ihi != "8003606790010791")
            {
                Assert.Fail("IHI was not stored correctly");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005820b_Test()
        {
            string mrn = "TEST-CCA_23";
            SetHi015005820Data(mrn);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.IhiStatusId != (int)IhiStatus.Active)
            {
                Assert.Fail("IHI Status is not correct");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005820c_Test()
        {
            string mrn = "TEST-CCA_23";
            SetHi015005820Data(mrn);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if ((patient.IhiLastValidated < patient.DateModified) || (patient.IhiLastValidated > patient.DateModified.AddSeconds(4)))
            {
                Assert.Fail("IHI Last update is incorrect");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005820d_Test()
        {
            string mrn = "TEST-CCA_23";
            SetHi015005820Data(mrn);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.IhiRecordStatusId != (int)IhiRecordStatus.Verified)
            {
                Assert.Fail("IHI record status was not stored correctly");
            }
        }

        #endregion CCA Test Methods HI_015_005820

        #region CCA Test Methods HI_015_005839

        [TestMethod()]
        public void Process_HI_015_005839_Test()
        {
            string mrnA = "TEST-CCA_30";
            string mrnB = "TEST-CCA_23";
            string ihi = "8003602346555439";
            Helpers.DeletePatientByMrn(mrnA);
            Helpers.DeletePatientByMrn(mrnB);
            ProcessMessage(HL7MessageResource.TEST_CCA_30);
            ProcessMessage(HL7MessageResource.TEST_CCA_23);
            Helpers.DeletePatientMasterIhiByMrn(mrnA);
            PatientMaster patient = Helpers.GetPatientByMrn(mrnB);
            Helpers.SetIhi(patient.PatientMasterId.Value, ihi);
            patient = Helpers.GetPatientByMrn(mrnB);
            if (patient.Ihi != ihi)
            {
                Assert.Fail("IHI test data is not correct");
            }

            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            ProcessMessage(HL7MessageResource.TEST_CCA_30);
            Helpers.IhiLookupAuditsOccured(lastAudit, 1);
            Helpers.IhiLookupAlertsOccured(lastAlert, 1);
            patient = Helpers.GetPatientByMrn(mrnA);
            IhiLookupAlert latestAlert = Helpers.GetLastIhiLookupAlert();
            if (latestAlert.Message != "The IHI status has changed to DuplicateIhi. This IHI can not be used.")
            {
                Assert.Fail("Alert does not hold the duplicate message");
            }
            if (patient.Ihi != ihi)
            {
                Assert.Fail("IHI was not stored");
            }
            if (patient.IhiStatusId != (int)IhiStatus.DuplicateIhi)
            {
                Assert.Fail("IHI was not flagged as a duplicate");
            }
        }

        #endregion CCA Test Methods HI_015_005839

        #region CCA Test Methods HI_015_005843

        [TestMethod()]
        public void Process_HI_015_005843a_Test()
        {
            string mrn = "TEST-CCA_01";
            PatientMaster patient = SetHi015005843Data(mrn);
            patient = Helpers.GetPatientByMrn(mrn);

            if (patient.Ihi != "8003602345688553")
            {
                Assert.Fail("IHI has not been stored correctly");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005843b_Test()
        {
            string mrn = "TEST-CCA_01";
            PatientMaster patient = SetHi015005843Data(mrn);
            patient = Helpers.GetPatientByMrn(mrn);
            if (patient.IhiStatusId != (int)IhiStatus.Active)
            {
                Assert.Fail("IHI status has not been stored correctly");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005843c_Test()
        {
            string mrn = "TEST-CCA_01";
            PatientMaster patient = SetHi015005843Data(mrn);
            patient = Helpers.GetPatientByMrn(mrn);
            if (patient.IhiRecordStatusId != (int)IhiRecordStatus.Verified)
            {
                Assert.Fail("IHI record status has not been stored correctly");
            }
        }

        #endregion CCA Test Methods HI_015_005843

        #region CCA Test Methods HI_015_005847

        [TestMethod()]
        public void Process_HI_015_005847a_Test()
        {
            string mrn = "TEST-CCA_05";
            PatientMaster patient = SetHi015005847Data(mrn);
            if (patient.Ihi != "8003601240025671")
            {
                Assert.Fail("IHI has not been stored correctly");
            }
            if (patient.IhiRecordStatusId != (int)IhiRecordStatus.Verified)
            {
                Assert.Fail("IHI record status is incorrect");
            }
            if (patient.IhiStatusId != (int)IhiStatus.Active)
            {
                Assert.Fail("IHI status is incorrect");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005847b_Test()
        {
            string mrn = "TEST-CCA_05";
            PatientMaster patient = SetHi015005847Data(mrn);
            PatientMasterIhiHistory history = Helpers.GetLastIhiHistoryByMrn(mrn);
            if (history.Ihi != "8003601240025663")
            {
                Assert.Fail("IHI has not been stored correctly");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005847c_Test()
        {
            string mrn = "TEST-CCA_05";
            PatientMaster patient = SetHi015005847Data(mrn);
            PatientMasterIhiHistory history = Helpers.GetLastIhiHistoryByMrn(mrn);
            if (history.IhiRecordStatusId != (int)IhiRecordStatus.Verified)
            {
                Assert.Fail("IHI record status is incorrect");
            }
            if (history.IhiStatusId != (int)IhiStatus.Resolved)
            {
                Assert.Fail("IHI status is incorrect");
            }
        }

        #endregion CCA Test Methods HI_015_005847

        #region CCA Test Methods HI_015_005872

        [TestMethod()]
        public void Process_HI_015_005872a_Test()
        {
            PatientMaster updated = SetHi015005872Data();

            if (updated.Ihi != "8003606790010791")
            {
                Assert.Fail("IHI is not correct");
            }
        }

        [TestMethod()]
        public void Process_HI_015_05872b_Test()
        {
            PatientMaster updated = SetHi015005872Data();

            if (updated.IhiStatusId != (int)IhiStatus.Active)
            {
                Assert.Fail("IHI status is not correct");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005872c_Test()
        {
            PatientMaster updated = SetHi015005872Data();

            if (updated.IhiRecordStatusId != (int)IhiRecordStatus.Verified)
            {
                Assert.Fail("IHI record status is not correct");
            }
        }

        #endregion CCA Test Methods HI_015_005872

        #region CCA Test Methods HI_015_005873

        [TestMethod()]
        public void Process_HI_015_005873a_Test()
        {
            IhiLookupAlert alert = SetHi015005873Data();
        }

        [TestMethod()]
        public void Process_HI_015_005873b_Test()
        {
            IhiLookupAlert alert = SetHi015005873Data();
            if (alert.DateCreated == null)
            {
                Assert.Fail("Date/Time has not been stored");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005873c_Test()
        {
            IhiLookupAlert alert = SetHi015005873Data();
            if (!alert.Response.Contains("out5:code"))
            {
                Assert.Fail("Error number has not been recorded in the response message");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005873d_Test()
        {
            IhiLookupAlert alert = SetHi015005873Data();
            if (string.IsNullOrEmpty(alert.Message))
            {
                Assert.Fail("Error message has not been stored");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005873e_Test()
        {
            IhiLookupAlert alert = SetHi015005873Data();
            if (!alert.Response.Contains("out5:reason"))
            {
                Assert.Fail("Message ID has not been recorded in the response message");
            }
        }

        #endregion CCA Test Methods HI_015_005873

        #region CCA TestMethods HI_015_005875

        [TestMethod()]
        public void Process_HI_015_005875a_Test()
        {
            string mrn = "TEST-CCA_B3";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_B3);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.Ihi != "8003603456797233" || patient.IhiRecordStatusId == (int)IhiRecordStatus.Verified && patient.IhiLastValidated < DateTime.Now.AddSeconds(-5))
            {
                Assert.Fail("The IHI was not retrieved");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005875b_Test()
        {
            SetHi015005875Data();
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            if (!lastAlert.Message.ToUpper().StartsWith("This patient has an IHI search criteria that matches another patient. The IHI cannot be used.".ToUpper()))
            {
                Assert.Fail("Duplicate patient demographics not found");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005875c_Test()
        {
            SetHi015005875Data();
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            PatientMaster patient = Helpers.GetPatientByMrn("TEST-CCA_B3B");
            if (!string.IsNullOrEmpty(patient.Ihi))
            {
                Assert.Fail("IHI was stored");
            }
        }

        #endregion CCA TestMethods HI_015_005875

        #region CCA TestMethods HI_015_006105

        [TestMethod()]
        public void Process_HI_015_006105_Test()
        {
            string mrn = "TEST-CCA_B1";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_015_006105);
            InsertCcaB1Ihi("Susan Georgian");
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.CurrentName.GivenNames != "Susan Georgian")
            {
                Assert.Fail("Test data is not correct");
            }
            ProcessMessage(HL7MessageResource.TEST_CCA_B1);
            patient = Helpers.GetPatientByMrn(mrn);
            if (patient.IhiLastValidated < DateTime.Now.AddSeconds(-5))
            {
                Assert.Fail("IHI was not revalidated");
            }
        }

        #endregion CCA TestMethods HI_015_006105

        #region CCA TestMethods HI_015_006119

        [TestMethod()]
        public void Process_HI_015_006119a_Test()
        {
            string mrn = SetHi015006119aData();
        }

        [TestMethod()]
        public void Process_HI_015_006119b_Test()
        {
            string mrn = SetHi015006119aData();
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            if (lastAlert.Message != "The IHI Record status has a status of Deceased. Please check that this is correct.")
            {
                Assert.Fail("Invalid alert produced");
            }
        }

        [TestMethod()]
        public void Process_HI_015_006119c1_Test()
        {
            string mrn = SetHi015006119aData();
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.IhiStatusId != (int)IhiStatus.Deceased)
            {
                Assert.Fail("IHI Status was not changed to deceased");
            }
            Helpers.DeletePatientByMrn(mrn);
        }

        [TestMethod()]
        public void Process_HI_015_006119c2_Test()
        {
            string mrn = SetHi015006119bData();
            Helpers.DeletePatientByMrn(mrn);
        }

        /// <summary>
        /// Sets the hi015006119a data.
        /// </summary>
        /// <returns></returns>
        private string SetHi015006119aData()
        {
            string mrn = "TEST-CCA_11";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_11);
            Helpers.DeletePatientMasterIhiByMrn(mrn);
            InsertCca11Ihi(IhiStatus.Active);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.IhiStatusId != (int)IhiStatus.Active)
            {
                Assert.Fail("Test data is incorrect");
            }
            IhiLookupAudit audit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_015_006119a);
            Helpers.IhiLookupAuditsOccured(audit, 1);
            return mrn;
        }

        /// <summary>
        /// Sets the hi015006119b data.
        /// </summary>
        /// <returns></returns>
        private string SetHi015006119bData()
        {
            string mrn = "TEST-CCA_06";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_06);
            Helpers.DeletePatientMasterIhiByMrn(mrn);
            InsertCca06Ihi();
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.Ihi != "8003601240025663")
            {
                Assert.Fail("Test data is incorrect");
            }
            IhiLookupAudit lastAudit = helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_015_006119b);
            Helpers.IhiLookupAuditsOccured(lastAudit, 3);
            return mrn;
        }

        #endregion CCA TestMethods HI_015_006119

        #region CCA TestMethods HI_015_006218

        [TestMethod()]
        public void Process_HI_015_006218a_Test()
        {
            string mrn = "TEST-CCA_02";
            SetHi015008218Data(mrn);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (!string.IsNullOrEmpty(patient.Ihi) || patient.IhiStatusId != -1 || patient.IhiRecordStatusId != -1)
            {
                Assert.Fail("IHI was not cleared");
            }
        }

        [TestMethod()]
        public void Process_HI_015_006218b_Test()
        {
            string mrn = "TEST-CCA_02";
            SetHi015008218Data(mrn);
            IhiLookupAlert alert = Helpers.GetLastIhiLookupAlert();
            if (alert.Message != "The IHI record status is Unverified. This IHI can not be used or recorded.")
            {
                Assert.Fail("Expected alert was not found");
            }
        }

        /// <summary>
        /// Sets the hi015008218 data.
        /// </summary>
        /// <param name="mrn">The MRN.</param>
        private void SetHi015008218Data(string mrn)
        {
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_02);
            InsertCca02Ihi(IhiStatus.Active);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.IhiStatusId != (int)IhiStatus.Active || patient.Ihi != "8003601240025168")
            {
                Assert.Fail("Test Data is incorrect");
            }
            ProcessMessage(HL7MessageResource.HI_015_008218);
        }

        #endregion CCA TestMethods HI_015_006218

        #region CCA TestMethods HI_015_008219

        [TestMethod()]
        public void Process_HI_015_008219a_Test()
        {
            string mrn;
            PatientMaster patient;
            SetHi015008219Data(out mrn, out patient);
            patient = Helpers.GetPatientByMrn(mrn);
            if (!string.IsNullOrEmpty(patient.Ihi) || patient.IhiStatusId != -1 || patient.IhiRecordStatusId != -1)
            {
                Assert.Fail("IHI was not updated correctly");
            }
        }

        [TestMethod()]
        public void Process_HI_015_008219b_Test()
        {
            string mrn;
            PatientMaster patient;
            SetHi015008219Data(out mrn, out patient);
            IhiLookupAlert alert = Helpers.GetLastIhiLookupAlert();
            if (alert.Message != "The IHI record status is Unverified. This IHI can not be used or recorded.")
            {
                Assert.Fail("IHI was not updated correctly");
            }
        }

        private void SetHi015008219Data(out string mrn, out PatientMaster patient)
        {
            mrn = "TEST-CCA_03";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_03);
            InsertCca03Ihi(IhiStatus.Active);
            patient = Helpers.GetPatientByMrn(mrn);
            if (patient.Ihi != "8003601240025176" || patient.IhiStatusId != (int)IhiStatus.Active || patient.IhiRecordStatusId != (int)IhiRecordStatus.Verified)
            {
                Assert.Fail("Invalid test data");
            }
            ProcessMessage(HL7MessageResource.HI_015_008219);
        }

        #endregion CCA TestMethods HI_015_008219

        #region CCA TestMethods HI_015_005809

        [TestMethod()]
        public void Process_HI_015_005809_Test()
        {
            string mrn = "TEST-CCA_36";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_015_005809);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.CurrentName.GivenNames != "Kaili Kameko")
            {
                Assert.Fail("Test data is incorrect");
            }
            ProcessMessage(HL7MessageResource.TEST_CCA_36);
            patient = Helpers.GetPatientByMrn(mrn);
            if (patient.CurrentName.GivenNames != "Kaili Liza Kameko")
            {
                Assert.Fail("The names were not updated correctly");
            }
        }

        #endregion CCA TestMethods HI_015_005809

        #region CCA TEST Methods HI_015_005812

        [TestMethod()]
        public void Process_HI_015_005812_Test()
        {
            string mrn = "TEST-CCA_B1";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_B1);
            InsertCcaB1Ihi();
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.Ihi != "8003601011850042")
            {
                Assert.Fail("Incorrect test data");
            }
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_015_005812);
            Helpers.IhiLookupAuditsOccured(lastAudit, 1);
            lastAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiNumber != "8003601011850042")
            {
                Assert.Fail("IHI Audit is not correct");
            }
        }

        #endregion CCA TEST Methods HI_015_005812

        #region CCA Test Methods HI_015_005813

        [TestMethod()]
        public void Process_HI_015_005813_Test()
        {
            string mrn = "TEST-CCA_64";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_64);
            Helpers.DeletePatientMasterIhiByMrn(mrn);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (!string.IsNullOrEmpty(patient.Ihi))
            {
                Assert.Fail("Incorrect test data");
            }
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_015_005813);
            Helpers.IhiLookupAuditsOccured(lastAudit, 1);
            lastAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiNumber != "8003608833345770")
            {
                Assert.Fail("IHI Audit is not correct");
            }
            if (!lastAudit.Request.Contains("medicareCardNumber"))
            {
                Assert.Fail("Medicare card was not used in the request message");
            }
            if (!lastAudit.Request.Contains("medicareIRN"))
            {
                Assert.Fail("Medicare IRN was not used in the request message");
            }
        }

        #endregion CCA Test Methods HI_015_005813

        #region CCA Test Methods HI_015_005814

        [TestMethod()]
        public void Process_HI_015_005814_Test()
        {
            string mrn = "TEST-CCA_31";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_31);
            Helpers.DeletePatientMasterIhiByMrn(mrn);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (!string.IsNullOrEmpty(patient.Ihi))
            {
                Assert.Fail("Incorrect test data");
            }
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_015_005814);
            Helpers.IhiLookupAuditsOccured(lastAudit, 1);
            lastAudit = Helpers.GetLastIhiLookupAudit();
            if (lastAudit.IhiNumber != "8003602346555520")
            {
                Assert.Fail("IHI Audit is not correct");
            }
            if (!lastAudit.Request.Contains("dvaFileNumber"))
            {
                Assert.Fail("DVA card was not used in the request message");
            }
        }

        #endregion CCA Test Methods HI_015_005814

        #region CCA TestMethods HI_015_005818

        [TestMethod()]
        public void Process_HI_015_005818a_Test()
        {
            string mrn;
            PatientMaster patient;
            SetHi015005818Data(out mrn, out patient);

            ProcessMessage(HL7MessageResource.HI_015_005818a);
            patient = Helpers.GetPatientByMrn(mrn);
            if (patient.CurrentName.FamilyName != "MARCELLEBAD")
            {
                Assert.Fail("Famnily name was not updated");
            }
            if (!string.IsNullOrEmpty(patient.Ihi))
            {
                Assert.Fail("IHI should be NULL");
            }
        }

        [TestMethod()]
        public void Process_HI_015_005818b_Test()
        {
            string mrn;
            PatientMaster patient;
            SetHi015005818Data(out mrn, out patient);

            ProcessMessage(HL7MessageResource.HI_015_005818b);
            patient = Helpers.GetPatientByMrn(mrn);
            if (patient.CurrentName.FamilyName != "MARCELLE")
            {
                Assert.Fail("Famnily name was not updated");
            }
            if (patient.Ihi != "8003603456797233")
            {
                Assert.Fail("IHI has not been updated");
            }
        }

        /// <summary>
        /// Sets the hi015005818 data.
        /// </summary>
        /// <param name="mrn">The MRN.</param>
        /// <param name="patient">The patient.</param>
        private void SetHi015005818Data(out string mrn, out PatientMaster patient)
        {
            mrn = "TEST-CCA_B3";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_B3);
            Helpers.DeletePatientMasterIhiByMrn(mrn);
            patient = Helpers.GetPatientByMrn(mrn);
            if (!string.IsNullOrEmpty(patient.Ihi))
            {
                Assert.Fail("Incorrect test data");
            }
        }

        #endregion CCA TestMethods HI_015_005818

        #region Private Methods

        /// <summary>
        /// Gets the user.
        /// </summary>
        /// <returns></returns>
        private static UserDetails GetUser()
        {
            UserDetails user = new UserDetails();

            //user.Login = "HAD/SANDEENM";
            //user.HpiI = "8003619900000081";
            //user.Name = "MILO SANDEEN";
            //user.HpiO = "8003624900000594"; // Each service should call PopulateUser to ensure the HPI-O is inserted from the HospitalCode.
            user.Role = UserRole.AuthorisedEmployee;
            return user;
        }

        /// <summary>
        /// Sets the hi015005808 data.
        /// </summary>
        private void SetHi015005808Data()
        {
            string mrn = "TEST-CCA_02";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_015_005808);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.DateOfBirth != new DateTime(1952, 11, 23))
            {
                Assert.Fail("Test data is not correct");
            }
        }

        /// <summary>
        /// Sets the hi015005847 data.
        /// </summary>
        /// <param name="mrn">The MRN.</param>
        /// <returns></returns>
        private PatientMaster SetHi015005847Data(string mrn)
        {
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_05);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            InsertCca05Ihi();
            patient = Helpers.GetPatientByMrn(mrn);
            if (patient.Ihi != "8003601240025663" || patient.IhiRecordStatusId != (int)IhiRecordStatus.Verified || patient.IhiStatusId != (int)IhiStatus.Active)
            {
                Assert.Fail("Test data is not correct");
            }
            PatientMasterIhiHistory history = Helpers.GetLastIhiHistoryByMrn(mrn);
            if (history.IhiHistoryId.HasValue)
            {
                Assert.Fail("IHI history records exist");
            }
            ProcessMessage(HL7MessageResource.HI_015_005847);
            patient = Helpers.GetPatientByMrn(mrn);
            return patient;
        }

        /// <summary>
        /// Sets the hi015005820 data.
        /// </summary>
        /// <param name="mrn">The MRN.</param>
        private void SetHi015005820Data(string mrn)
        {
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_015_005820);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (!string.IsNullOrEmpty(patient.Ihi))
            {
                Assert.Fail("Invalid Test Data");
            }
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            ProcessMessage(HL7MessageResource.TEST_CCA_23);
            Helpers.IhiLookupAlertsOccured(lastAlert, 0);
        }

        /// <summary>
        /// Sets the hi010008028 data.
        /// </summary>
        /// <param name="mrn">The MRN.</param>
        private void SetHi010008028Data(string mrn)
        {
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_010_008028);
            InsertCcaB1Ihi();
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_010_008028);
            Helpers.IhiLookupAlertsOccured(lastAlert, 0);
            Helpers.IhiLookupAuditsOccured(lastAudit, 1);
            IhiLookupAlert alert = Helpers.GetLastIhiLookupAlert();
        }

        /// <summary>
        /// Sets the hi010008526 data.
        /// </summary>
        /// <param name="mrnA">The MRN A.</param>
        /// <param name="mrnB">The MRN B.</param>
        private void SetHi010008526Data(string mrnA, string mrnB)
        {
            Helpers.DeletePatientByMrn(mrnA);
            Helpers.DeletePatientByMrn(mrnB);
            ProcessMessage(HL7MessageResource.TEST_CCA_B1);
            ProcessMessage(HL7MessageResource.TEST_CCA_B3);
            Helpers.DeletePatientMasterIhiByMrn(mrnA);
            Helpers.DeletePatientMasterIhiByMrn(mrnB);
            PatientMaster patient = Helpers.GetPatientByMrn(mrnA);
            if (!string.IsNullOrEmpty(patient.Ihi) || patient.IhiStatusId != 0 || patient.IhiStatusId != 0)
            {
                Assert.Fail("Test data has incorrect data");
            }
            patient = Helpers.GetPatientByMrn(mrnB);
            if (!string.IsNullOrEmpty(patient.Ihi) || patient.IhiStatusId != 0 || patient.IhiStatusId != 0)
            {
                Assert.Fail("Test data has incorrect data");
            }
        }

        /// <summary>
        /// Sets the hi015005873 data.
        /// </summary>
        /// <returns></returns>
        private IhiLookupAlert SetHi015005873Data()
        {
            string mrn = "TEST-CCA_B1";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_B1);
            InsertCcaB1Ihi();
            IhiLookupAlert alert = Helpers.GetLastIhiLookupAlert();
            ProcessMessage(HL7MessageResource.HI_015_005873, false);
            Helpers.IhiLookupAlertsOccured(alert, 4);
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            return lastAlert;
        }

        /// <summary>
        /// Sets the hi015005872 data.
        /// </summary>
        /// <returns></returns>
        private PatientMaster SetHi015005872Data()
        {
            string mrn = "TEST-CCA_23";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_23);
            PatientMaster originalPatient = Helpers.GetPatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.HI_015_005872);
            PatientMaster updated = Helpers.GetPatientByMrn(mrn);
            if (updated.IhiLastValidated <= originalPatient.IhiLastValidated)
            {
                Assert.Fail("IHI was not revalidated");
            }
            if (updated.CurrentName.FamilyName.ToUpper() != "JIMMY")
            {
                Assert.Fail("Patient name not updated");
            }
            return updated;
        }

        /// <summary>
        /// Sets the hi015005843 data.
        /// </summary>
        /// <param name="mrn">The MRN.</param>
        /// <returns></returns>
        private PatientMaster SetHi015005843Data(string mrn)
        {
            string ihi = "8003602345688553";
            Helpers.DeletePatientByMrn(mrn);
            ProcessMessage(HL7MessageResource.TEST_CCA_01);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            PatientMasterIhi ihiDetail = new PatientMasterIhi();
            ihiDetail.PatientMasterId = patient.PatientMasterId;
            ihiDetail.Ihi = ihi;
            ihiDetail.RegisteredFamilyName = "Test";
            ihiDetail.RegisteredGivenName = "Testing";
            ihiDetail.RegisteredSexId = 2;
            ihiDetail.IhiRecordStatusId = -1;
            ihiDetail.IhiStatusId = -1;
            ihiDetail.DateLastValidated = DateTime.Now.AddDays(-3);
            Helpers.PatientMasterIhiInsert(ihiDetail);
            ProcessMessage(HL7MessageResource.TEST_CCA_01);
            return patient;
        }

        /// <summary>
        /// Sets the hi015005875 data.
        /// </summary>
        private void SetHi015005875Data()
        {
            string mrn = "TEST-CCA_B3";
            string mrnB = "TEST-CCA_B3B";
            Helpers.DeletePatientByMrn(mrn);
            Helpers.DeletePatientByMrn(mrnB);
            ProcessMessage(HL7MessageResource.TEST_CCA_B3);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            Helpers.DeletePatientMasterIhiByMrn(mrn);
            PatientMaster patientB3 = Helpers.GetPatientByMrn(mrn);
            if (!string.IsNullOrEmpty(patientB3.Ihi))
            {
                Assert.Fail("TEST-CCA_B3 IHI is not empty");
            }
            IhiLookupAlert lastAlert = Helpers.GetLastIhiLookupAlert();
            IhiLookupAudit lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_015_005875a);
            Helpers.IhiLookupAlertsOccured(lastAlert, 0);
            Helpers.IhiLookupAuditsOccured(lastAudit, 0);
            PatientMaster patientB3B = Helpers.GetPatientByMrn(mrnB);
            if (patientB3.Id == patientB3B.Id)
            {
                Assert.Fail("Patient TEST-CCA_B3B has been created under TEST-CCA_B3");
            }
            lastAlert = Helpers.GetLastIhiLookupAlert();
            lastAudit = Helpers.GetLastIhiLookupAudit();
            ProcessMessage(HL7MessageResource.HI_015_005875b);
            Helpers.IhiLookupAlertsOccured(lastAlert, 1);
            Helpers.IhiLookupAuditsOccured(lastAudit, 1);
        }

        /// <summary>
        /// Tests the date time validation.
        /// </summary>
        /// <param name="message">The message.</param>
        private void TestDateTimeValidation(string message, bool shouldFail = true)
        {
            try
            {
                int expectedErrors = (shouldFail == true) ? 1 : 0;

                SystemErrorLog errorLog = Helpers.GetLastErrorMessage();
                HL7Message newPatientMessage = HL7Message.Parse(message);
                HL7GenericPasMessage pasMessage = newPatientMessage as HL7GenericPasMessage;
                MessageLoader pasLoader = new MessageLoader();
                HL7MessageLog log = new HL7MessageLog();
                UserDetails user = GetUser();
                pasLoader.Process(pasMessage, pasMessage.ToString(), user, out log);
                if (shouldFail && log.QueueStatusId == (int)QueueStatus.Success)
                {
                    Assert.Fail("The Date time validation did not fail");
                }
                if (!shouldFail && log.QueueStatusId != (int)QueueStatus.Success)
                {
                    Assert.Fail("Date Validation failed a valid date");
                }
                Helpers.AnySystemErrorsOccur(errorLog, expectedErrors);
            }
            catch (Exception ex)
            {
                Assert.Fail(string.Format("A System Error Occured - {0}", ex.Message));
            }
        }

        /// <summary>
        /// Tests the medicare validation.
        /// </summary>
        /// <param name="message">The message.</param>
        /// <param name="mrn">The MRN.</param>
        /// <param name="expectedMedicare">The expected medicare.</param>
        /// <param name="expectedStatus">if set to <c>true</c> [expected status].</param>
        /// <param name="resettingData">if set to <c>true</c> [resetting data].</param>
        private void TestMedicareValidation(string message, string mrn, string expectedMedicare = null, string expectedIrn = null, bool expectedStatus = false, bool resettingData = true)
        {
            try
            {
                if (resettingData)
                {
                    Helpers.DeletePatientByMrn(mrn);
                }
                SystemErrorLog errorLog = Helpers.GetLastErrorMessage();
                IhiLookupAlert alert = Helpers.GetLastIhiLookupAlert();

                HL7Message newPatientMessage = HL7Message.Parse(message);
                HL7GenericPasMessage pasMessage = newPatientMessage as HL7GenericPasMessage;
                MessageLoader pasLoader = new MessageLoader();
                HL7MessageLog log = new HL7MessageLog();
                UserDetails user = GetUser();
                pasLoader.Process(pasMessage, pasMessage.ToString(), user, out log);
                if (resettingData)
                {
                    return;
                }

                Helpers.AnySystemErrorsOccur(errorLog, 0);
                int numberAlerts = (expectedStatus == true) ? 0 : 1;
                Helpers.IhiLookupAlertsOccured(alert, numberAlerts);
                PatientMaster patient = Helpers.GetPatientByMrn(mrn);
                if (patient.MedicareNumber != expectedMedicare)
                {
                    Assert.Fail("Medicare number was not stored");
                }
                if (patient.MedicareIrn != expectedIrn)
                {
                    Assert.Fail("IRN number was not stored");
                }
                if (patient.IsMedicareNumberValid != expectedStatus)
                {
                    Assert.Fail("Medicare status is not correct");
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(string.Format("A System Error Occured - {0}", ex.Message));
            }
        }

        /// <summary>
        /// Tests the name current or legal/registered.
        /// </summary>
        /// <param name="message">The message.</param>
        /// <param name="mrn">The MRN.</param>
        /// <param name="lastName">The last name.</param>
        /// <param name="firstName">The first name.</param>
        /// <param name="currentName">if set to <c>true</c> [current name].</param>
        private void TestName(string message, string mrn, string lastName, string firstName, string legalLastName, string LegalFirstName)
        {
            try
            {
                SystemErrorLog errorLog = Helpers.GetLastErrorMessage();
                HL7Message newPatientMessage = HL7Message.Parse(message);
                HL7GenericPasMessage pasMessage = newPatientMessage as HL7GenericPasMessage;
                MessageLoader pasLoader = new MessageLoader();
                HL7MessageLog log = new HL7MessageLog();
                UserDetails user = GetUser();
                pasLoader.Process(pasMessage, pasMessage.ToString(), user, out log);
                Helpers.AnySystemErrorsOccur(errorLog);
                PatientMaster patient = Helpers.GetPatientByMrn(mrn);
                if (patient.CurrentName.FamilyName.ToUpper() != lastName.ToUpper())
                {
                    Assert.Fail("Current family name is not correct");
                }
                if (patient.CurrentName.GivenNames.ToUpper() != firstName.ToUpper())
                {
                    Assert.Fail("Current given name is not correct");
                }
                if (patient.LegalName.FamilyName.ToUpper() != legalLastName.ToUpper())
                {
                    Assert.Fail("Legal family name is not correct");
                }
                if (patient.LegalName.GivenNames.ToUpper() != LegalFirstName.ToUpper())
                {
                    Assert.Fail("Legal given name is not correct");
                }
            }
            catch (Exception ex)
            {
                Assert.Fail(string.Format("A System Error Occured - {0}", ex.Message));
            }
        }

        /// <summary>
        /// Processes the message.
        /// </summary>
        /// <param name="message">The message.</param>
        private void ProcessMessage(string message, bool failOnSystemErrors = true)
        {
            SystemErrorLog errorLog = Helpers.GetLastErrorMessage();
            HL7Message newPatientMessage = HL7Message.Parse(message);
            HL7GenericPasMessage pasMessage = newPatientMessage as HL7GenericPasMessage;
            MessageLoader pasLoader = new MessageLoader();
            HL7MessageLog log = new HL7MessageLog();
            UserDetails user = GetUser();
            pasLoader.Process(pasMessage, pasMessage.ToString(), user, out log);
            if (failOnSystemErrors)
            {
                Helpers.AnySystemErrorsOccur(errorLog);
            }
        }

        /// <summary>
        /// Inserts the ihi for patient Test-CCA-01.
        /// </summary>
        /// <param name="patientMasterId">The patient master id.</param>
        private void InsertCca01Ihi()
        {
            PatientMasterIhi ihi = new PatientMasterIhi();
            PatientMaster patient = Helpers.GetPatientByMrn("TEST-CCA_01");
            ihi.PatientMasterId = patient.PatientMasterId;
            ihi.Ihi = "8003602345688553";
            ihi.IhiRecordStatusId = 1;
            ihi.IhiStatusId = 1;
            ihi.RegisteredGivenName = "Test";
            ihi.RegisteredFamilyName = "Test 2";
            ihi.RegisteredSexId = 2;

            //ihi.PcehrAdvertised = false;
            ihi.DateLastValidated = DateTime.Now.AddDays(-2);

            //ihi.AccessCodeRequiredId = 1;
            Helpers.PatientMasterIhiInsert(ihi);
        }

        /// <summary>
        /// Inserts the cca b1 ihi.
        /// </summary>
        private void InsertCcaB1Ihi(string givenName = "Harriette George", string lastName = "DWAYNE")
        {
            PatientMasterIhi ihi = new PatientMasterIhi();
            PatientMaster patient = Helpers.GetPatientByMrn("TEST-CCA_B1");
            ihi.PatientMasterId = patient.PatientMasterId;

            ihi.Ihi = "8003601011850042";
            ihi.IhiRecordStatusId = 1;
            ihi.IhiStatusId = 1;
            ihi.RegisteredGivenName = givenName;
            ihi.RegisteredFamilyName = lastName;
            ihi.RegisteredSexId = 2;

            //ihi.PcehrAdvertised = false;
            ihi.DateLastValidated = DateTime.Now.AddDays(-2);

            //ihi.AccessCodeRequiredId = 1;
            Helpers.PatientMasterIhiInsert(ihi);
        }

        private void InsertCca05Ihi()
        {
            PatientMasterIhi ihi = new PatientMasterIhi();
            PatientMaster patient = Helpers.GetPatientByMrn("TEST-CCA_05");
            ihi.PatientMasterId = patient.PatientMasterId;
            ihi.Ihi = "8003601240025663";
            ihi.IhiRecordStatusId = 1;
            ihi.IhiStatusId = 1;
            ihi.RegisteredGivenName = "Arthur";
            ihi.RegisteredFamilyName = "Morse";
            ihi.RegisteredSexId = 1;

            //ihi.PcehrAdvertised = false;
            ihi.DateLastValidated = DateTime.Now.AddDays(-2);

            //ihi.AccessCodeRequiredId = 1;
            Helpers.PatientMasterIhiInsert(ihi);
        }

        private void InsertCca04Ihi(IhiStatus status)
        {
            PatientMasterIhi ihi = new PatientMasterIhi();
            PatientMaster patient = Helpers.GetPatientByMrn("TEST-CCA_04");
            ihi.PatientMasterId = patient.PatientMasterId;
            ihi.Ihi = "8003601240025184";
            ihi.IhiRecordStatusId = 1;
            ihi.IhiStatusId = (int)status;
            ihi.RegisteredGivenName = "Matthew";
            ihi.RegisteredFamilyName = "Joel";
            ihi.RegisteredSexId = 1;

            //ihi.PcehrAdvertised = false;
            ihi.DateLastValidated = DateTime.Now.AddDays(-2);

            //ihi.AccessCodeRequiredId = 1;
            Helpers.PatientMasterIhiInsert(ihi);
        }

        private void InsertCca02Ihi(IhiStatus status)
        {
            PatientMasterIhi ihi = new PatientMasterIhi();
            PatientMaster patient = Helpers.GetPatientByMrn("TEST-CCA_02");
            ihi.PatientMasterId = patient.PatientMasterId;
            ihi.Ihi = "8003601240025168";
            ihi.IhiRecordStatusId = 1;
            ihi.IhiStatusId = (int)status;
            ihi.RegisteredGivenName = "JACKOB";
            ihi.RegisteredFamilyName = "SIMONS";
            ihi.RegisteredSexId = 1;

            //ihi.PcehrAdvertised = false;
            ihi.DateLastValidated = DateTime.Now.AddDays(-2);

            //ihi.AccessCodeRequiredId = 1;
            Helpers.PatientMasterIhiInsert(ihi);
        }

        private void InsertCca03Ihi(IhiStatus status)
        {
            PatientMasterIhi ihi = new PatientMasterIhi();
            PatientMaster patient = Helpers.GetPatientByMrn("TEST-CCA_03");
            ihi.PatientMasterId = patient.PatientMasterId;
            ihi.Ihi = "8003601240025176";
            ihi.IhiRecordStatusId = 1;
            ihi.IhiStatusId = (int)status;
            ihi.RegisteredGivenName = "MELLISA";
            ihi.RegisteredFamilyName = "CRUISE";
            ihi.RegisteredSexId = 1;

            //ihi.PcehrAdvertised = false;
            ihi.DateLastValidated = DateTime.Now.AddDays(-2);

            //ihi.AccessCodeRequiredId = 1;
            Helpers.PatientMasterIhiInsert(ihi);
        }

        private void InsertCca11Ihi(IhiStatus status)
        {
            PatientMasterIhi ihi = new PatientMasterIhi();
            PatientMaster patient = Helpers.GetPatientByMrn("TEST-CCA_11");
            ihi.PatientMasterId = patient.PatientMasterId;
            ihi.Ihi = "8003606789133208";
            ihi.IhiRecordStatusId = 1;
            ihi.IhiStatusId = (int)status;
            ihi.RegisteredGivenName = "DILBERT";
            ihi.RegisteredFamilyName = "DOBSON";
            ihi.RegisteredSexId = 1;

            //ihi.PcehrAdvertised = false;
            ihi.DateLastValidated = DateTime.Now.AddDays(-2);

            //ihi.AccessCodeRequiredId = 1;
            Helpers.PatientMasterIhiInsert(ihi);
        }

        private void InsertCca06Ihi()
        {
            PatientMasterIhi ihi = new PatientMasterIhi();
            PatientMaster patient = Helpers.GetPatientByMrn("TEST-CCA_06");
            ihi.PatientMasterId = patient.PatientMasterId;
            ihi.Ihi = "8003601240025663";
            ihi.IhiRecordStatusId = 1;
            ihi.IhiStatusId = (int)IhiStatus.Active;
            ihi.RegisteredGivenName = "AUTHAR";
            ihi.RegisteredFamilyName = "MORSE";
            ihi.RegisteredSexId = 1;

            //ihi.PcehrAdvertised = false;
            ihi.DateLastValidated = DateTime.Now.AddDays(-2);

            //ihi.AccessCodeRequiredId = 1;
            Helpers.PatientMasterIhiInsert(ihi);
        }

        /// <summary>
        /// Sets the data H i010005820.
        /// </summary>
        /// <returns></returns>
        private PatientMaster SetDataHI010005820()
        {
            string mrn = "TEST-CCA_23";
            Helpers.DeletePatientByMrn(mrn);
            PatientMaster patient = Helpers.GetPatientByMrn(mrn);
            if (patient.PatientMasterId != null)
            {
                Assert.Fail("Patient exists");
            }
            SystemErrorLog lastLog = Helpers.GetLastErrorMessage();
            ProcessMessage(HL7MessageResource.HI_010_005820);
            Helpers.AnySystemErrorsOccur(lastLog, 0);
            patient = Helpers.GetPatientByMrn(mrn);
            if (patient.PatientMasterId == null)
            {
                Assert.Fail("Patient was not stored correctly");
            }
            return patient;
        }

        #endregion Private Methods
    }
}