﻿using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading;
using System.Xml;
using System.Xml.Schema;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.NpdrCcaNoc.NoticeOfConnection
{
    /// <summary>
    /// Conformance Test Specification: NPDR Notice of Connection Test Plan
    ///                                 Version 1.1 — 15 July 2013
    /// Operation:  removeDocument
    /// </summary>
    [TestClass]
    public class NOC_RemoveDocument : CcaTest
    {
        /// <summary>
        /// Test No:         NOC_DC_012
        /// Objective/Input:
        /// Perform a Remove or deregister document operation to remove a prescribe document from the NPDR.
        ///
        /// Expected Result:
        /// Document should be successfully removed from both the NPDR and the PCEHR.
        /// 
        /// NOTE: If this fails with "There is a VsDebuggerCausalityData element in the SOAP header" then
        /// you must run:
        ///    vsdiag_regwcf -u
        /// which is installed in "C:\Program Files\Microsoft Visual Studio 11.0\Common7\IDE".
        /// </summary>
        [TestMethod]
        [TestCategory("NPDR_NOC")]
        public void NPDR_NOC_12()
        {
            TestSuccessfulRemove(SampleDocumentType.PcehrPrescriptionRecord);
        }

        /// <summary>
        /// Test No:         NOC_DC_013
        /// Objective/Input:
        /// Perform a Remove or deregister document operation to remove a prescribe* document from the NPDR
        /// 
        /// * Assumed this is intended to read 'dispense' as test 12 is the same.
        ///
        /// Expected Result:
        /// Document should be successfully removed from both the NPDR and the PCEHR.
        /// </summary>
        [TestMethod]
        [TestCategory("NPDR_NOC")]
        public void NPDR_NOC_13()
        {
            TestSuccessfulRemove(SampleDocumentType.PcehrDispenseRecord);
        }
    
        /// <summary>
        /// Code behind tests 12 and 13.
        /// </summary>
        /// <param name="docType"></param>
        private void TestSuccessfulRemove(SampleDocumentType docType)
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document = patient.CreateNewDocument(docType);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document);

            // Add 10 seconds wait before removing
            Thread.Sleep(10000);

            byte[] auditInformation = UTF8Encoding.UTF8.GetBytes(TestContext.TestName);
            HipsResponse removeResponse = ProxyHelper.PcehrProxy.Remove(identifier, patient.TargetEpisode.AdmissionDate, document.GetSetId(), RemovalReason.Withdrawn, user, auditInformation);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, removeResponse, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.RemoveDocument, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            // Confirm that the payload is well formed and complies with the RemoveDocument Request XML Schema
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");
            nsmgr.AddNamespace("r", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/RemoveDocument/1.0");
            XmlNode payload = doc.SelectSingleNode("/s:Envelope/s:Body/r:removeDocument", nsmgr);
            doc.Schemas.Add(nsmgr.LookupNamespace("s"), "soap-envelope.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("c"), "PCEHR_CommonTypes.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("r"), "PCEHR_RemoveDocument.xsd");
            try
            {
                doc.Validate(null);
                LogAssert.IsTrue(doc.SchemaInfo.Validity == XmlSchemaValidity.Valid,
                    "SOAP request validates against SOAP envelope schema",
                    "SOAP request is invalid");

                doc.Validate(null, payload);
                LogAssert.IsTrue(payload.SchemaInfo.Validity == XmlSchemaValidity.Valid,
                    "removeDocument payload validates against Remove Document schema.",
                    "removeDocument payload is invalid");
            }
            catch (XmlException ex)
            {
                LogAssert.Fail(ex.Message);
            }
            catch (XmlSchemaValidationException ex)
            {
                LogAssert.Fail(ex.Message);
            }

            NpdrHeaderTests headerTests = new NpdrHeaderTests(this);
            headerTests.CheckHeader(doc, DialogueResource.HipsServiceRemove);

            XmlNode reasonForRemovalNode = doc.SelectSingleNode("/s:Envelope/s:Body/r:removeDocument/r:reasonForRemoval", nsmgr);
            LogAssert.AreEqual("Withdrawn", reasonForRemovalNode.InnerText, "The reasonForRemoval field");
        }

        /// <summary>
        /// Test No:         NOC_DC_014
        /// Objective/Input:
        /// Perform a Remove or deregister document operation of a document already removed from the NPDR
        ///
        /// Expected Result:
        /// Document remove request should fail as the document requested has already been removed
        /// </summary>
        [TestMethod]
        [TestCategory("NPDR_NOC")]
        public void NPDR_NOC_14()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.PcehrPrescriptionRecord);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document);

            // Remove once
            byte[] auditInformation = UTF8Encoding.UTF8.GetBytes("NPDR_NOC_14-a");
            HipsResponse removeResponse = ProxyHelper.PcehrProxy.Remove(identifier, patient.TargetEpisode.AdmissionDate, document.GetSetId(), RemovalReason.Withdrawn, user, auditInformation);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, removeResponse, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout, expectFailure: false);

            // Remove again
            byte[] auditInformation2 = UTF8Encoding.UTF8.GetBytes("NPDR_NOC_14-b");
            HipsResponse removeResponse2 = ProxyHelper.PcehrProxy.Remove(identifier, patient.TargetEpisode.AdmissionDate, document.GetSetId(), RemovalReason.Withdrawn, user, auditInformation2);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, removeResponse2, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout, expectFailure: true);

            List<PcehrMessageQueue> queueOps = patient.GetPcehrMessageQueueItems();
            LogAssert.AreEqual(
                "PcehrServiceError, Code: urn:oasis:names:tc:ebxml-regrep:ResponseStatusType:Error, Description: MEDVIEW_REPOSITORY_015, Details: No CDA document package matching the document unique ID could be found.",
                queueOps[0].Details, DialogueResource.QueueOperationStatusDetail);
        }

        /// <summary>
        /// Test No:        NOC_DC_015
        /// Objective/Input:
        /// Perform a removeDocument operation by a user certificate which is not the same as the user certificate used to upload document being removed.
        ///
        /// Expected Result:
        /// Document submission should fail NPDR certificate validation
        /// </summary>
        [TestMethod]
        [TestCategory("NPDR_NOC")]
        public void NPDR_NOC_15()
        {            
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            try
            {
                HpiOHelper.SwitchToHpiO(patient, this, 1);
                UserDetails user = patient.GetTestUser();
                PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
                CdaDocument document = patient.CreateNewDocument(SampleDocumentType.PcehrPrescriptionRecord);
                QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document);

                Log(@"Info: The hospital will now be changed in the database from the first HPI-O to the second HPI-O.");
                HpiOHelper.SwitchToHpiO(patient, this, 2);
                
                // Add 30 seconds wait before removing
                Thread.Sleep(30000);

                byte[] auditInformation = UTF8Encoding.UTF8.GetBytes("NPDR_NOC_15");
                HipsResponse removeResponse = ProxyHelper.PcehrProxy.Remove(identifier, patient.TargetEpisode.AdmissionDate, document.GetSetId(), RemovalReason.Withdrawn, user, auditInformation);
                LogAssert.ExpectResponse(HipsResponseIndicator.OK, removeResponse, DialogueResource.HipsServiceRemove);
                QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout, expectFailure: true);

                List<PcehrMessageQueue> queueOps = patient.GetPcehrMessageQueueItems();
                LogAssert.AreEqual(
                    "PcehrServiceError, Code: PCEHR_ERROR_3002, Description: Document metadata failed validation",
                    queueOps[0].Details, DialogueResource.QueueOperationStatusDetail);
            }
            finally
            {
                // Ensure HPI-O is changed back to original even on failure
                HpiOHelper.SwitchToHpiO(patient, this, 1);
            }
        }

        /// <summary>
        /// Test No:        NOC_DC_016
        /// Objective/Input:
        /// Performa a removeDocument operation by a user certificate that is not valid in the NPDR
        ///
        /// Expected Result:
        /// Document submission should fail NPDR certificate validation
        /// </summary>
        [TestMethod]
        [TestCategory("NPDR_NOC")]
        public void NPDR_NOC_16()
        {
            try
            {
                patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
                UserDetails user = patient.GetTestUser();
                PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
                CdaDocument document = patient.CreateNewDocument(SampleDocumentType.PcehrPrescriptionRecord);
                QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document);

                Log(@"Info: The hospital will be now changed in the database from the first HPI-O to the third HPI-O (revoked certificate).");
                HpiOHelper.SwitchToHpiO(patient, this, 3);

                byte[] auditInformation = UTF8Encoding.UTF8.GetBytes("NPDR_NOC_16");
                HipsResponse removeResponse = ProxyHelper.PcehrProxy.Remove(identifier, patient.TargetEpisode.AdmissionDate, document.GetSetId(), RemovalReason.Withdrawn, user, auditInformation);
                LogAssert.ExpectResponse(HipsResponseIndicator.OK, removeResponse, DialogueResource.HipsServiceRemove);
                QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout, expectFailure: true);

                List<PcehrMessageQueue> queueOps = patient.GetPcehrMessageQueueItems();
                LogAssert.IsTrue(queueOps[0].Details.Contains("The HTTP request was forbidden"),
                    "Queue operation status detail contains 'The HTTP request was forbidden'", 
                    string.Format("Queue operation status detail does not contain 'The HTTP request was forbidden', but instead {0}", queueOps[0].Details));
            }
            finally
            {
                HpiOHelper.SwitchToHpiO(patient, this, 1);
            }
        }
    }
}