﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text.RegularExpressions;
using System.Xml;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.CdaPackaging.eSignature
{
    /// <summary>
    /// Conformance Test Specification: CDA Packaging
    ///                                 Version 1.4 — 12 September 2012
    /// Profile Name:	eSignature
    /// Group:          Payload
    /// </summary>
    [TestClass]
    public class Payload : CcaTest
    {
        private CdaDocument cdaDocument;
        private CdaSignature cdaSignature;

        /// <summary>
        /// Shared helper method to ensure that a document has been uploaded
        /// and get the CDA Signature.
        /// The document will be reused between the tests in this class.
        /// </summary>
        private void UploadDocument()
        {
            SharedPackage shared = SharedPackage.GetSharedInstance(this, SampleDocumentType.DischargeSummary);
            this.cdaDocument = shared.CdaDocument;
            this.cdaSignature = shared.CdaSignature;
        }

        /// <summary>
        /// Verify that the sp:signedPayloadData element in the Signed Payload
        /// contains ONLY one s:eSignature element.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_026()
        {
            UploadDocument();
            XmlNodeList eSignatureElements = cdaSignature.GetESignatureElements();
            LogAssert.AreEqual(1, eSignatureElements.Count, DialogueResource.NumberOfESignatureElements);
        }

        /// <summary>
        /// a.  Verify that the ds:Manifest element in eSignature contains one ds:Reference element.
        /// b.  Verify that URI attribute of the ds:Reference element is set to part identifier of the root part of the CDA XML document.
        /// c.  Verify that ds:Reference element uses SHA-1 digest algorithm on its unmodified byte stream.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_027()
        {
            UploadDocument();

            XmlNodeList referenceElements = cdaSignature.GetReferenceElementsInManifest();

            // A - Verify that the ds:Manifest element in eSignature contains one ds:Reference element.
            LogAssert.AreEqual(1, referenceElements.Count, DialogueResource.NumberOfReferenceElementsInManifest);

            // B - Verify that URI attribute of the ds:Reference element is set to part identifier of the root part of the CDA XML document.
            LogAssert.AreEqual("CDA_ROOT.XML", cdaSignature.GetReferenceUri(referenceElements[0]), DialogueResource.UriAttributeOfReferenceElement);

            // C - Verify that ds:Reference element uses SHA-1 digest algorithm on its unmodified byte stream.
            string algorithm = cdaSignature.GetDigestMethodAlgorithm(referenceElements[0]);
            LogAssert.AreEqual("http://www.w3.org/2000/09/xmldsig#sha1", algorithm, DialogueResource.DigestMethodAlgorithm);
        }

        /// <summary>
        /// a.  Verify that the ds:Manifest element in eSignature contains ONLY one ds:Reference element.
        /// b.  Verify that URI attribute of the ds:Reference element is set to part identifier of the signatory part of the CDA XML document.
        /// c.  Verify that ds:Reference element uses SHA-1 digest algorithm on its unmodified byte stream.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_028()
        {
            UploadDocument();

            // A - Verify that the ds:Manifest element in eSignature contains ONLY one ds:Reference element.
            XmlNodeList referenceElements = cdaSignature.GetReferenceElementsInSignedInfo();
            LogAssert.AreEqual(1, referenceElements.Count, DialogueResource.NumberOfReferenceElementsInSignedInfo);

            // B - Verify that URI attribute of the ds:Reference element is set to part identifier of the signatory part of the CDA XML document.
            string signedPayloadDataId = cdaSignature.GetSignedPayloadDataId();
            string partIdentifier = string.Format("#{0}", signedPayloadDataId);
            string referenceUri = referenceElements[0].Attributes["URI"].Value;
            LogAssert.AreEqual(partIdentifier, referenceUri, DialogueResource.UriAttributeOfReferenceElement);

            // C - Verify that ds:Reference element uses SHA-1 digest algorithm on its unmodified byte stream.
            string algorithm = cdaSignature.GetDigestMethodAlgorithm(referenceElements[0]);
            LogAssert.AreEqual("http://www.w3.org/2000/09/xmldsig#sha1", algorithm, DialogueResource.DigestMethodAlgorithm);
        }

        /// <summary>
        /// Verify that the person that approved the eSignature can be identified
        /// from the values contained in the s:approver element of the eSignature.
        ///
        /// For HIPS, this checks the document author and signature approver have
        /// the same person ID, family name and given names.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_029()
        {
            UploadDocument();
            string documentAuthorPersonId = cdaDocument.GetAuthorPersonEntityIdentifier();
            string signatureApproverPersonId = new Uri(cdaSignature.GetApproverPersonId()).Segments.Last();
            LogAssert.AreEqual(documentAuthorPersonId, signatureApproverPersonId, DialogueResource.DocumentAuthorPersonId, DialogueResource.SignatureApproverPersonId);

            string documentAuthorFamilyName = cdaDocument.GetAuthorFamilyName();
            string signatureApproverFamilyName = cdaSignature.GetApproverFamilyName();
            LogAssert.AreEqual(documentAuthorFamilyName, signatureApproverFamilyName, DialogueResource.DocumentAuthorFamilyName, DialogueResource.SignatureApproverFamilyName);

            List<string> documentAuthorGivenNames = cdaDocument.GetAuthorGivenNames();
            List<string> signatureApproverGivenNames = cdaSignature.GetApproverGivenNames();
            LogAssert.AreEqual(documentAuthorGivenNames.Count, signatureApproverGivenNames.Count,
                DialogueResource.NumberOfDocumentAuthorGivenNames,
                DialogueResource.NumberOfSignatureApproverGivenNames);
            for (int i = 0; i < documentAuthorGivenNames.Count; i++)
            {
                LogAssert.AreEqual(documentAuthorGivenNames[i], signatureApproverGivenNames[i],
                    string.Format(DialogueResource.DocumentAuthorGivenName, i + 1),
                    string.Format(DialogueResource.SignatureApproverGivenName, i + 1));
            }
        }

        /// <summary>
        /// Verify that the s:signingTime element of the eSignature contains a valid time.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_030()
        {
            UploadDocument();
            string signingTime = cdaSignature.GetSigningTime();
            bool parsed;
            try
            {
                string[] formats = { "yyyy-MM-ddTHH:mm:ss.fffffffZ", "yyyy-MM-ddTHH:mm:ss.fffffffzzz" };
                DateTime.ParseExact(signingTime, formats, CultureInfo.InvariantCulture.DateTimeFormat, DateTimeStyles.None);
                parsed = true;
            }
            catch (FormatException)
            {
                parsed = false;
            }
            LogAssert.IsTrue(parsed,
                string.Format(DialogueResource.SigningTimeIsValid, signingTime),
                string.Format(DialogueResource.SigningTimeIsInvalid, signingTime));
        }

        /// <summary>
        /// Verify that the s:signingTime element in the eSignature includes an explicit timezone.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_031()
        {
            UploadDocument();
            string signingTime = cdaSignature.GetSigningTime();
            if (signingTime.EndsWith("Z"))
            {
                LogAssert.IsTrue(signingTime.EndsWith("Z"),
                    string.Format(DialogueResource.SigningTimeIsUTC, signingTime),
                    string.Format(DialogueResource.SigningTimeIsNotUTC, signingTime));
            }
            else
            {
                Regex timezone = new Regex(@"[+-]\d\d:\d\d");
                Match match = timezone.Match(signingTime);
                LogAssert.IsTrue(match.Success && signingTime.EndsWith(match.Value),
                    string.Format(DialogueResource.SigningTimeHasLocalTimeZone, signingTime, match.Value),
                    string.Format(DialogueResource.SigningTimeHasNoTimeZone, signingTime));
            }
        }
    }
}