﻿using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_002_1
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 27 September 2012
    /// Use Case #:	    UC.CIS.002.1
    /// Use Case Name:	Gain access to PCEHR (with Access Code)
    /// Test Case ID:   PCEHR_CIS_019100
    /// Objective:
    /// To check that, when accessing the PCEHR System the Clinical Information System only
    /// uses an individual healthcare identifier (IHI) if it has been validated or obtained
    /// during a configurable period, with the period determined by the local healthcare
    /// provider’s policy.
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_019100 : CcaTest
    {
        /// <summary>
        /// Attempt to access the PCEHR with a valid IHI that has not been validated within
        /// the configurable period and has one or more unresolved alerts against it and
        /// ensure that:
        /// a) the CIS prevents access to the PCEHR with that IHI with the unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_002_1")]
        public void CIS_0021_019100_A()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: false, hasUnresolvedAlerts: true, testDataId: "CCA ID IHI_38");
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithCode(patient.TargetPatientIdentifier, patient.TestPatientRecordAccessCode, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to access the PCEHR with an invalid IHI that has not been validated within
        /// the configurable period and has one or more unresolved alerts against it and ensure
        /// that:
        /// b) the CIS prevents access to the PCEHR with that IHI with the unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_002_1")]
        public void CIS_0021_019100_B()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: true);
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithCode(patient.TargetPatientIdentifier, patient.TestPatientRecordAccessCode, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to access the PCEHR with an invalid IHI that has not been validated within
        /// the configurable period and has no unresolved alerts against it and ensure that:
        ///
        /// c) the CIS validates* the IHI against the HI Service directly OR via another
        ///    software system (other than the PCEHR System) that is connected to the
        ///    HI Service.
        /// d) ensure the CIS prevents access to the PCEHR with that invalid IHI.
        ///
        /// * “validate” in this context means to send an IHI and demographic data to the HI
        ///    Service for verification or to return an IHI for the provided demographic data
        ///    for comparison with the CIS.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_002_1")]
        public void CIS_0021_019100_CD()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false);
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithCode(patient.TargetPatientIdentifier, patient.TestPatientRecordAccessCode, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidIhi, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Check that HIPS attempted to validate the IHI
            IhiLookupAudit ihiAudit = patient.GetLastIhiLookupAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, ihiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to Access the PCEHR with a valid IHI that has been validated within the
        /// configurable period and has one or more unresolved alerts against it and ensure
        /// that:
        /// e) the CIS prevents access to the PCEHR with that IHI with the unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_002_1")]
        public void CIS_0021_019100_E()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: true, testDataId: "CCA ID IHI_16");
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithCode(patient.TargetPatientIdentifier, patient.TestPatientRecordAccessCode, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to access the PCEHR with an invalid IHI that has been validated within the
        /// configurable period and has one or more unresolved alerts against it and ensure
        /// that:
        /// f) the CIS prevents access to the PCEHR with that IHI and the unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_002_1")]
        public void CIS_0021_019100_F()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: true, hasUnresolvedAlerts: true);
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithCode(patient.TargetPatientIdentifier, patient.TestPatientRecordAccessCode, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to access the PCEHR with an invalid IHI that has been validated within the
        /// configurable period and has no unresolved alerts against it and ensure that:
        /// g) the CIS prevents access to the PCEHR with that invalid IHI.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_002_1")]
        public void CIS_0021_019100_G()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: true, hasUnresolvedAlerts: false);
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithCode(patient.TargetPatientIdentifier, patient.TestPatientRecordAccessCode, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidIhi, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to access the PCEHR with a valid IHI that has not been validated within the
        /// configurable period and has no unresolved alerts against it and ensure that:
        /// h) the CIS validates* the IHI against the HI Service directly OR via another software
        ///    system (other than the PCEHR System) that is connected to the HI Service.
        /// i) ensure the CIS can access the PCEHR with the valid IHI.
        ///
        /// * “validate” in this context means to send an IHI and demographic data to the HI
        ///    Service for verification or to return an IHI for the provided demographic data
        ///    for comparison with the CIS.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_002_1")]
        public void CIS_0021_019100_HI()
        {
            // Changing 37 to 87
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: false, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithCode(patient.TargetPatientIdentifier, patient.TestPatientRecordAccessCode, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Check that HIPS validated the IHI
            IhiLookupAudit ihiAudit = patient.GetLastIhiLookupAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, ihiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS accessed the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to access the PCEHR with a valid IHI that has been validated within the
        /// configurable period and has no unresolved alerts against it and ensure that:
        /// j) the CIS can access the PCEHR with the valid IHI
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_002_1")]
        public void CIS_0021_019100_J()
        {
            // Changed from IHI 25 to 93 because 25 does not have a record access code
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_93");
            GainPcehrAccessResponse response = ProxyHelper.PcehrProxy.GainAccessWithCode(patient.TargetPatientIdentifier, patient.TestPatientRecordAccessCode, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, response.HipsResponse, DialogueResource.HipsServiceGainAccessWithCode);

            // Check that HIPS did not attempt to validate the IHI
            IhiLookupAudit ihiAudit = patient.GetLastIhiLookupAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, ihiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS accessed the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }
    }
}