﻿using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_201
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 27 September 2012
    /// Use Case #:	    UC.CIS.201
    /// Use Case Name:	Upload a Clinical Document
    /// Test Case ID:   PCEHR_CIS_019100
    /// Objective:
    /// To check that, when accessing the PCEHR System the Clinical Information System only
    /// uses an individual healthcare identifier (IHI) if it has been validated or obtained
    /// during a configurable period, with the period determined by the local healthcare
    /// provider’s policy.
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_019100 : CcaTest
    {
        /// <summary>
        /// Attempt to upload a clinical document into a healthcare recipient's
        /// PCEHR with a valid IHI that has not been validated within the
        /// configurable period and has one or more unresolved alerts against
        /// it and ensure that:
        /// a) the CIS prevents access to the PCEHR with that IHI with the
        ///    unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_201")]
        public void CIS_201_019100_A()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: false, hasUnresolvedAlerts: true, testDataId: "CCA ID IHI_33");
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, uploadResponse, DialogueResource.HipsServiceUploadDocument);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to upload a clinical document into a healthcare recipient's
        /// PCEHR with an invalid IHI that has not been validated within
        /// the configurable period and has one or more unresolved alerts against it and ensure
        /// that:
        /// b) the CIS prevents access to the PCEHR with that IHI with the unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_201")]
        public void CIS_201_019100_B()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: true);
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, uploadResponse, DialogueResource.HipsServiceUploadDocument);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to upload a clinical document into a healthcare recipient's
        /// PCEHR with an invalid IHI that has not been validated within
        /// the configurable period and has no unresolved alerts against it and ensure that:
        ///
        /// c) the CIS validates* the IHI against the HI Service directly OR via another
        ///    software system (other than the PCEHR System) that is connected to the
        ///    HI Service.
        /// d) ensure the CIS prevents access to the PCEHR with that invalid IHI.
        ///
        /// * “validate” in this context means to send an IHI and demographic data to the HI
        ///    Service for verification or to return an IHI for the provided demographic data
        ///    for comparison with the CIS.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_201")]
        public void CIS_201_019100_CD()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false);
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode()); 
            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidIhi, uploadResponse, DialogueResource.HipsServiceUploadDocument);

            // Check that HIPS did attempt to validate the IHI for this patient
            IhiLookupAudit hiAudit = patient.GetLastIhiLookupAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, hiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to upload a clinical document into a healthcare recipient's
        /// PCEHR with a valid IHI that has been validated within the
        /// configurable period and has one or more unresolved alerts against it and ensure
        /// that:
        /// e) the CIS prevents access to the PCEHR with that IHI with the unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_201")]
        public void CIS_201_019100_E()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: true, testDataId: "CCA ID IHI_16");
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode()); 
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, uploadResponse, DialogueResource.HipsServiceUploadDocument);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to upload a clinical document into a healthcare recipient's
        /// PCEHR with an invalid IHI that has been validated within the
        /// configurable period and has one or more unresolved alerts against it and ensure
        /// that:
        /// f) the CIS prevents access to the PCEHR with that IHI and the unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_201")]
        public void CIS_201_019100_F()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: true, hasUnresolvedAlerts: true);
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, uploadResponse, DialogueResource.HipsServiceUploadDocument);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to upload a clinical document into a healthcare recipient's
        /// PCEHR with an invalid IHI that has been validated within the
        /// configurable period and has no unresolved alerts against it and ensure that:
        /// g) the CIS prevents access to the PCEHR with that invalid IHI.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_201")]
        public void CIS_201_019100_G()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: true, hasUnresolvedAlerts: false);
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidIhi, uploadResponse, DialogueResource.HipsServiceUploadDocument);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to upload a clinical document into a healthcare recipient's
        /// PCEHR with a valid IHI that has not been validated within the
        /// configurable period and has no unresolved alerts against it and
        /// ensure that:
        /// h) the CIS validates* the IHI against the HI Service directly OR via another software
        ///    system (other than the PCEHR System) that is connected to the HI Service.
        /// i) ensure the CIS can access the PCEHR with the valid IHI.
        ///
        /// * “validate” in this context means to send an IHI and demographic data to the HI
        ///    Service for verification or to return an IHI for the provided demographic data
        ///    for comparison with the CIS.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_201")]
        public void CIS_201_019100_HI()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: false, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_57");
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, uploadResponse, DialogueResource.HipsServiceUploadDocument);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout);

            // Check that HIPS did attempt to validate the IHI for this patient
            IhiLookupAudit hiAudit = patient.GetLastIhiLookupAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, hiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS accessed the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to upload a clinical document into a healthcare recipient's
        /// PCEHR with a valid IHI that has been validated within the
        /// configurable period and has no unresolved alerts against it and
        /// ensure that:
        /// j) the CIS can access the PCEHR with the valid IHI
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_201")]
        public void CIS_201_019100_J()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_25");
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, uploadResponse, DialogueResource.HipsServiceUploadDocument);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout);

            // Check that HIPS did not attempt to validate the IHI for this patient
            IhiLookupAudit hiAudit = patient.GetLastIhiLookupAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, hiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS accessed the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }
    }
}