﻿using System.Linq;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_202
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems
    ///                                 Connecting to the PCEHR System
    ///                                 Version 1.6 — 27 September 2012
    /// Use Case #:	    UC.CIS.202
    /// Use Case Name:	Supersede a Clinical Document
    /// Test Case ID:   PCEHR_CIS_017842b
    /// Objective:
    /// To check that the CIS provides a mechanism to identify which clinical
    /// documents have been uploaded to the PCEHR System.
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_017842b : CcaTest
    {
        /// <summary>
        /// Create an amended version of the clinical document created and
        /// uploaded in PCEHR_CIS_017842a in use case UC.CIS.201 using the
        /// software, and supersede the uploaded version on the PCEHR System
        /// with the amended version:
        /// a. Verify that the software has a mechanism to record that a
        ///    document has been uploaded to the PCEHR.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_202")]
        public void CIS_202_017842b_A()
        {
            // Changed 86 to 85 because there is no 86 in the spreadsheet
            Log(DialogueResource.ChangedTestData86To85);
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_85");
            Episode episode = patient.TargetEpisode;
            CdaDocument cdaDocument1 = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, cdaDocument1);

            CdaDocument cdaDocument2 = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, cdaDocument2);

            // Demonstrate criteria A, that the software has a mechanism to record that a document has been uploaded to the PCEHR.
            // We will use the HIPS service 'GetOperationStatus' to confirm that the document was uploaded.
            OperationStatus statusResponse = ProxyHelper.PcehrProxy.GetOperationStatus(patient.TargetPatientIdentifier, episode.AdmissionDate, patient.GetTestUser());

            // Check that the document set is what we expected.
            LogAssert.AreEqual(1, statusResponse.UploadedDocuments.Count, DialogueResource.UploadedDocumentsCount);
            LogAssert.AreEqual((int)HIPS.PcehrDataStore.Schemas.Enumerators.ClinicalDocumentStatus.Active, statusResponse.UploadedDocuments[0].ClinicalDocumentStatusId, DialogueResource.UploadedDocumentStatus);
            LogAssert.AreEqual(episode.EpisodeId.Value, statusResponse.UploadedDocuments[0].EpisodeId, DialogueResource.UploadedDocumentEpisodeId);
            LogAssert.AreEqual(cdaDocument2.GetSetId(), statusResponse.UploadedDocuments[0].SourceSystemSetId, DialogueResource.UploadedDocumentSetId);

            // Check the versions are what we expected. First order them by version ID as there is no guarantee on the order returned.
            LogAssert.AreEqual(2, statusResponse.UploadedDocumentVersions.Count, DialogueResource.UploadedDocumentVersionsCount);
            IOrderedEnumerable<ClinicalDocumentVersion> sorted = statusResponse.UploadedDocumentVersions.OrderBy(a => a.ClinicalDocumentVersionId);
            LogAssert.AreEqual(cdaDocument1.GetDocId(), sorted.First().SourceSystemDocumentId, DialogueResource.FirstUploadedVersionDocumentId);
            LogAssert.AreEqual(cdaDocument2.GetDocId(), sorted.Last().SourceSystemDocumentId, DialogueResource.SecondUploadedVersionDocumentId);
        }
    }
}