﻿using System.Text;
using HIPS.CommonSchemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_202
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 27 September 2012
    /// Use Case #:	    UC.CIS.202
    /// Use Case Name:	Supersede a Clinical Document
    /// Test Case ID:   PCEHR_CIS_019042
    /// Objective:
    /// To check that the CIS provides the ability to support withdrawal of
    /// consent to upload clinical documents to the PCEHR, and to check that
    /// they are prevented from being uploaded where consent has been withdrawn.
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_019042 : CcaTest
    {
        /// <summary>
        /// Withdraw consent either before or after creating a clinical document
        /// for a healthcare recipient:
        /// a. Verify the software provides the ability to withdraw consent to
        ///    upload the document to the PCEHR System.
        /// b. Verify that the document is not uploaded to the PCEHR system.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_202")]
        public void CIS_202_019042_AB()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_85");

            // Withdraw consent - this will prevent any newer versions from being uploaded
            byte[] auditInformation = Encoding.UTF8.GetBytes("CIS_202_019042_AB");
            HipsResponse consentResponse = ProxyHelper.ConsentProxy.RecordConsent(true, patient.TargetPatientIdentifier, patient.TargetEpisode.AdmissionDate, patient.GetTestUser(), auditInformation);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, consentResponse, DialogueResource.HipsServiceRecordConsent);

            // Create and upload the initial version of the document, check that the response is that the upload cannot proceed because consent was withdrawn.
            CdaDocument cdaDocument1 = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                cdaDocument1.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(), 
                new Attachment[0], patient.TargetEpisode.AdmissionDate, cdaDocument1.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.ConsentWithdrawn, uploadResponse, DialogueResource.HipsServiceUploadDocument);

            // Call the HIPS service to upload a new version, check that the response is that the upload cannot proceed because consent was withdrawn.
            CdaDocument cdaDocument2 = patient.CreateAmendedDocument(cdaDocument1);
            HipsResponse supersedeResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                cdaDocument2.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, cdaDocument2.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.ConsentWithdrawn, supersedeResponse, DialogueResource.HipsServiceUploadDocument);

            // Call the HIPS service to verify that neither was uploaded.
            OperationStatus statusResponse = ProxyHelper.PcehrProxy.GetOperationStatus(patient.TargetPatientIdentifier, patient.TargetEpisode.AdmissionDate, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, statusResponse.Response, DialogueResource.HipsServiceGetOperationStatus);
            LogAssert.AreEqual(0, statusResponse.UploadedDocuments.Count, DialogueResource.UploadedDocumentsCount);
            LogAssert.AreEqual(0, statusResponse.UploadedDocumentVersions.Count, DialogueResource.UploadedDocumentVersionsCount);
        }
    }
}