﻿using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_202
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 27 September 2012
    /// Use Case #:	    UC.CIS.202
    /// Use Case Name:	Supersede a clinical document
    /// Test Case ID:   PCEHR_CIS_019100
    /// Objective:
    /// To check that, when accessing the PCEHR System the Clinical Information System only
    /// uses an individual healthcare identifier (IHI) if it has been validated or obtained
    /// during a configurable period, with the period determined by the local healthcare
    /// provider’s policy.
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_019100 : CcaTest
    {
        /// <summary>
        /// Attempt to supersede a clinical document in a healthcare recipient's
        /// PCEHR with a valid IHI that has not been validated within the
        /// configurable period and has one or more unresolved alerts against
        /// it and ensure that:
        /// a) the CIS prevents access to the PCEHR with that IHI with the
        ///    unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_202")]
        public void CIS_202_019100_A()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: false, hasUnresolvedAlerts: true, testDataId: "CCA ID IHI_51");
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, uploadResponse, DialogueResource.HipsServiceUploadDocument1);
            CdaDocument document2 = patient.CreateAmendedDocument(document);
            HipsResponse supersedeResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document2.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document2.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, supersedeResponse, DialogueResource.HipsServiceUploadDocument2);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to supersede a clinical document in a healthcare recipient's
        /// PCEHR with an invalid IHI that has not been validated within
        /// the configurable period and has one or more unresolved alerts against it and ensure
        /// that:
        /// b) the CIS prevents access to the PCEHR with that IHI with the unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_202")]
        public void CIS_202_019100_B()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: true);
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, uploadResponse, DialogueResource.HipsServiceUploadDocument1);
            CdaDocument document2 = patient.CreateAmendedDocument(document);
            HipsResponse supersedeResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document2.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document2.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, supersedeResponse, DialogueResource.HipsServiceUploadDocument2);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to supersede a clinical document in a healthcare recipient's
        /// PCEHR with an invalid IHI that has not been validated within
        /// the configurable period and has no unresolved alerts against it and ensure that:
        ///
        /// c) the CIS validates* the IHI against the HI Service directly OR via another
        ///    software system (other than the PCEHR System) that is connected to the
        ///    HI Service.
        /// d) ensure the CIS prevents access to the PCEHR with that invalid IHI.
        ///
        /// * “validate” in this context means to send an IHI and demographic data to the HI
        ///    Service for verification or to return an IHI for the provided demographic data
        ///    for comparison with the CIS.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_202")]
        public void CIS_202_019100_CD()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: false, hasUnresolvedAlerts: false);

            // Upload the initial version of the document, using the valid IHI
            patient.MakeIhiValid();
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, uploadResponse, DialogueResource.HipsServiceUploadDocument1);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout);

            // Check that HIPS did access the PCEHR (for the upload of the initial version)
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);

            // Upload the amended version of the document, using the invalid IHI
            patient.MakeIhiInvalid();
            CdaDocument document2 = patient.CreateAmendedDocument(document);
            HipsResponse supersedeResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document2.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document2.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidIhi, supersedeResponse, DialogueResource.HipsServiceUploadDocument2);

            // Check that HIPS did attempt to validate the IHI for this patient (the one with the invalid IHI)
            IhiLookupAudit hiAudit = patient.GetLastIhiLookupAudit();
            patient.TargetPatient.PatientMasterId = null;
            LogAssert.AreEqual(patient.TargetPatientMasterId, hiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS did not access the PCEHR (for the supersede operation) so the PCEHR Audit ID is the same as before.
            PcehrAudit pcehrAudit2 = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(pcehrAudit.PcehrAuditId, pcehrAudit2.PcehrAuditId, "PCEHR Audit ID");
        }

        /// <summary>
        /// Attempt to supersede a clinical document in a healthcare recipient's
        /// PCEHR with a valid IHI that has been validated within the
        /// configurable period and has one or more unresolved alerts against it and ensure
        /// that:
        /// e) the CIS prevents access to the PCEHR with that IHI with the unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_202")]
        public void CIS_202_019100_E()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: false, hasUnresolvedAlerts: true, testDataId: "CCA ID IHI_16");
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, uploadResponse, DialogueResource.HipsServiceUploadDocument1);
            
            CdaDocument document2 = patient.CreateAmendedDocument(document);
            HipsResponse supersedeResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document2.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document2.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, supersedeResponse, DialogueResource.HipsServiceUploadDocument2);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to supersede a clinical document in a healthcare recipient's
        /// PCEHR with an invalid IHI that has been validated within the
        /// configurable period and has one or more unresolved alerts against it and ensure
        /// that:
        /// f) the CIS prevents access to the PCEHR with that IHI and the unresolved alert.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_202")]
        public void CIS_202_019100_F()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: true, hasUnresolvedAlerts: true);
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, uploadResponse, DialogueResource.HipsServiceUploadDocument1);
            
            CdaDocument document2 = patient.CreateAmendedDocument(document);
            HipsResponse supersedeResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document2.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document2.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, supersedeResponse, DialogueResource.HipsServiceUploadDocument2);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to supersede a clinical document in a healthcare recipient's
        /// PCEHR with an invalid IHI that has been validated within the
        /// configurable period and has no unresolved alerts against it and ensure that:
        /// g) the CIS prevents access to the PCEHR with that invalid IHI.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_202")]
        public void CIS_202_019100_G()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: true, hasUnresolvedAlerts: false);
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidIhi, uploadResponse, DialogueResource.HipsServiceUploadDocument1);
            
            CdaDocument document2 = patient.CreateAmendedDocument(document);
            HipsResponse supersedeResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document2.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document2.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidIhi, supersedeResponse, DialogueResource.HipsServiceUploadDocument2);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to supersede a clinical document in a healthcare recipient's
        /// PCEHR with a valid IHI that has not been validated within the
        /// configurable period and has no unresolved alerts against it and
        /// ensure that:
        /// h) the CIS validates* the IHI against the HI Service directly OR via another software
        ///    system (other than the PCEHR System) that is connected to the HI Service.
        /// i) ensure the CIS can access the PCEHR with the valid IHI.
        ///
        /// * “validate” in this context means to send an IHI and demographic data to the HI
        ///    Service for verification or to return an IHI for the provided demographic data
        ///    for comparison with the CIS.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_202")]
        public void CIS_202_019100_HI()
        {
            Log(DialogueResource.CIS_202_019100_HI_1);

            // Upload the initial version of the document with an IHI that is not stale.
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_25");
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, uploadResponse, DialogueResource.HipsServiceUploadDocument1);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout);

            // Check that HIPS did not validate the IHI
            IhiLookupAudit hiAudit = patient.GetLastIhiLookupAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, hiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS did access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);

            // Make the IHI stale and attempt to supersede the document.
            Log(DialogueResource.CIS_202_019100_HI_2);
            patient.MakeIhiStale();

            CdaDocument document2 = patient.CreateAmendedDocument(document);
            HipsResponse supersedeResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document2.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document2.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, supersedeResponse, DialogueResource.HipsServiceUploadDocument2);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout);

            // Check that HIPS did validate the IHI
            IhiLookupAudit hiAudit2 = patient.GetLastIhiLookupAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, hiAudit2.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS did access the PCEHR
            PcehrAudit pcehrAudit2 = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, pcehrAudit2.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// Attempt to supersede a clinical document in a healthcare recipient's
        /// PCEHR with a valid IHI that has been validated within the
        /// configurable period and has no unresolved alerts against it and
        /// ensure that:
        /// j) the CIS can access the PCEHR with the valid IHI
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_202")]
        public void CIS_202_019100_J()
        {
            // Upload the initial version of the document with an IHI that is not stale.
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_34");
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            HipsResponse uploadResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, uploadResponse, DialogueResource.HipsServiceUploadDocument1);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout);

            // Check that HIPS did not validate the IHI
            IhiLookupAudit hiAudit = patient.GetLastIhiLookupAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, hiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS did access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);

            // Attempt to supersede the document.
            CdaDocument document2 = patient.CreateAmendedDocument(document);
            HipsResponse supersedeResponse = ProxyHelper.PcehrProxy.UploadOrSupersedeDocument(
                document2.GetBytes(), patient.TargetPatientIdentifier, patient.GetTestUser(),
                new Attachment[0], patient.TargetEpisode.AdmissionDate, document2.GetFormatCode());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, supersedeResponse, DialogueResource.HipsServiceUploadDocument2);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.UploadWaitTimeout);

            // Check that HIPS did not validate the IHI
            IhiLookupAudit hiAudit2 = patient.GetLastIhiLookupAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, hiAudit2.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS did access the PCEHR
            PcehrAudit pcehrAudit2 = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, pcehrAudit2.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }
    }
}