﻿using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;
using HIPS.PcehrSchemas;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using System.Xml;
using System.IO;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_501
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 7 March 2013
    /// Use Case #:	    UC.CIS.501
    /// Use Case Name:	Assisted Registration of an adult
    /// Test Case ID:   PCEHR_CIS_020208c
    /// Objective:
    /// To assist a consumer with registering for an eHealth record. 
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_020208c : CcaTest
    {
        /// <summary>
        /// 1. Prepare registration details and complete all necessary fields, without recording the consumer's consent to upload AODR information.
        ///     a. Verify that the CIS does not default to 'consent given' for the uploading of AODR information.
        ///     b. Attempt to submit the registration request.
        ///     c. Verify the CIS requires the user to record the consumer's consent to upload AODR information, and does not send the registration request to the eHealth record system.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501_Bypass_Ihi_Retrieve")]
        public void CIS_501_020208c_1_abc()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 13", preloadPatient: true);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven)
            };
            
            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse =  ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("Must provide a response for AODR documents consent.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);

        }

        /// <summary>
        /// 1. Prepare registration details and complete all necessary fields, without recording the consumer's consent to upload AODR information.
        ///     d. Select AODR 'consent given' or something equivalent.
        ///     e. Submit the registration request.
        ///     f. Verify the registration request contains 'ConsentGiven' for AODR.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501_Bypass_Ihi_Retrieve")]
        public void CIS_501_020208c_1_def()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 13", preloadPatient: false);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven)
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");

            VerifyDocumentConsentInformation(pcehrAudit, "AODR", "ConsentGiven");
        }

        /// <summary>
        /// 2. Repeat (1), selecting AODR 'consent not given' (or something equivalent) and verify 'ConsentNotGiven' is included in the registration request.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501_Bypass_Ihi_Retrieve")]
        public void CIS_501_020208c_2()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 13", preloadPatient: false);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven)
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");

            VerifyDocumentConsentInformation(pcehrAudit, "AODR", "ConsentNotGiven");
        }

        /// <summary>
        /// 3. Prepare registration details and complete all necessary fields, without recording the consumer's consent to upload ACIR  information.
        ///     a. Verify that the CIS does not default to 'consent given' for the uploading of ACIR information.
        ///     b. Attempt to submit the registration request.
        ///     c. Verify the CIS requires the user to record the consumer's consent to upload ACIR  information, and does not send the registration request to eHealth record system.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501_Bypass_Ihi_Retrieve")]
        public void CIS_501_020208c_3_abc()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 13");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven)
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("Must provide a response for ACIR documents consent.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);

        }

        /// <summary>
        /// 1. Prepare registration details and complete all necessary fields, without recording the consumer's consent to upload AODR information.
        ///     d. Select ACIR 'consent given' or something equivalent.
        ///     e. Submit the registration request.
        ///     f. Verify the registration request contains 'ConsentGiven' for ACIR.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501_Bypass_Ihi_Retrieve")]
        public void CIS_501_020208c_3_def()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 13", preloadPatient: false);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven)
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");

            VerifyDocumentConsentInformation(pcehrAudit, "ACIR", "ConsentGiven");
        }

        /// <summary>
        /// 4. Repeat (3), selecting ACIR 'consent not given'  (or something equivalent) and verify 'ConsentNotGiven' is included in the registration request.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501_Bypass_Ihi_Retrieve")]
        public void CIS_501_020208c_4()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 08", preloadPatient: false);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven)
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");

            VerifyDocumentConsentInformation(pcehrAudit, "ACIR", "ConsentNotGiven");
        }

        private void VerifyDocumentConsentInformation(PcehrAudit audit, string type, string consent)
        {
            // Confirm that the payload is well formed and complies with the gainPCEHRAccess Request XML Schema
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/RegisterPCEHR/2.0");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");

            // Check that the patient ID was provided in the request
            XmlNode documentNode = doc.SelectSingleNode(String.Format("/s:Envelope/s:Body/p:registerPCEHR/p:assertions/p:documentConsent/p:document[p:type='{0}']/p:status", type), nsmgr);
            LogAssert.AreEqual(consent, documentNode.InnerText, String.Format("{0} consent value", type));

        }
    }
}
