﻿using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;
using HIPS.PcehrSchemas;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using System.Xml;
using System.IO;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_501
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 7 March 2013
    /// Use Case #:	    UC.CIS.501
    /// Use Case Name:	Assisted Registration of an adult
    /// Test Case ID:   PCEHR_CIS_020219
    /// Objective:
    /// To assist a consumer with registering for an eHealth record. 
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_020219 : CcaTest
    {
        /// <summary>
        /// 1. Prepare registration details and complete all necessary fields, without recording consent to upload future and past MBS information.
        ///     a. Verify that the CIS does not default to 'consent given' to upload future MBS information.
        ///     b. Attempt to submit the registration request.
        ///     c. Verify the CIS requires the user to record the consumer's consent to upload future MBS information, and the registration request is not sent to the eHealth record system.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020219_1()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 18");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };
            
            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse =  ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("Must provide a response for MBS documents consent.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);

        }

        /// <summary>
        /// 2. Record 'consent given' (or equivalent) for the consumer's consent to upload future MBS information, without recording consent to upload past MBS information.
        ///     a. Verify that the CIS does not default to 'consent given' to upload past MBS information.
        ///     b. Attempt to submit the registration request.
        ///     c. Verify the CIS requires the user to record the consumer's consent to upload past MBS information, and the registration request is not sent to eHealth record system.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020219_2_abc()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 18");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven)
            };
            
            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse =  ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.ValidationError, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);
            LogAssert.AreEqual("Must provide a response for MBS Past Assimilation documents consent.", registrationResponse.HipsResponse.HipsErrorMessage, DialogueResource.HipsServiceRegisterPatient);

        }

        /// <summary>
        /// 2. Record 'consent given' (or equivalent) for the consumer's consent to upload future MBS information, without recording consent to upload past MBS information.
        ///     d. Record 'consent given' (or equivalent) for the consumer's consent to upload past MBS information.
        ///     e. Verify the registration request contains the correct information.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020219_2_de()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 18", preloadPatient: false);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven)
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse,  DialogueResource.HipsServiceRegisterPatient);

            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");
            
            VerifyPastMBSInformation(pcehrAudit, "ConsentGiven");
        }

        /// <summary>
        /// 2. Record 'consent given' (or equivalent) for the consumer's consent to upload future MBS information, without recording consent to upload past MBS information.
        ///     Repeat (2), with step (d) containing a record of consent equivalent to 'consent not given'.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020219_2_de_repeat()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 18", preloadPatient: false);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven)
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, registrationResponse.HipsResponse,  DialogueResource.HipsServiceRegisterPatient);

            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");

            VerifyPastMBSInformation(pcehrAudit, "ConsentNotGiven");
        }

        /// <summary>
        /// 3. Prepare registration details and complete all necessary fields, including 'consent not given' (or equivalent) for uploading future MBS information, without recording consent to upload past MBS information.
        ///     a. Attempt to submit the registration request.
        ///     b. Verify the CIS either requires the user to record 'consent to not given' (or equivalent) to upload past MBS information or that the CIS automatically records 'consent to not given' (or equivalent).
        ///     d. Verify the registration request contains the correct information.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020219_3()
        {
            // Patient with incorrect demographics
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 18", preloadPatient: false);

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven)
            };

            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, registrationResponse.HipsResponse,  DialogueResource.HipsServiceRegisterPatient);

            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");

            VerifyPastMBSInformation(pcehrAudit, "ConsentNotGiven");
        }

        private void VerifyPastMBSInformation(PcehrAudit audit, string consent)
        {
            // Confirm that the payload is well formed and complies with the gainPCEHRAccess Request XML Schema
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/RegisterPCEHR/2.0");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");

            // Check that the patient ID was provided in the request
            XmlNode mbsPastNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:assertions/p:documentConsent/p:document[p:type='MBSPastAssimilation']/p:status", nsmgr);
            LogAssert.AreEqual(consent, mbsPastNode.InnerText, "MBS consent value");

        }
    }
}
