﻿using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;
using HIPS.PcehrSchemas;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using System.Xml;
using System.IO;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_501
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 7 March 2013
    /// Use Case #:	    UC.CIS.501
    /// Use Case Name:	Assisted Registration of an adult
    /// Test Case ID:   PCEHR_CIS_020240
    /// Objective:
    /// To assist a consumer with registering for an eHealth record. 
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_020240 : CcaTest
    {
        /// <summary>
        /// Prepare registration details that include a phone number and an IVC delivery method set to 'SMS', 
        /// and complete all necessary fields for a successful registration of a consumer's eHealth record. 
        /// Ensure that all phone numbers in the consumer's local health record differ from the phone number in the registration details.
        ///     a. Submit the registration request to the eHealth record system
        ///     b. Once the registration request is completed, verify the CIS did not automatically update any of the communication 
        ///     details in the consumer's local health record with the phone number supplied in the registration request. 
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020240_1()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 06");

            // Get initial contact list
            List<Contact> contacts = patient.TargetPatient.Contacts;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };
            
            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.sms;
            channel.phoneNumber = "0401950881";

            Log("Checking communication details before registration:");
            LogAssert.DoesNotContain(patient.GetCurrentContacts(), channel.phoneNumber);

            PcehrRegistrationResponse registrationResponse =  ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            Log("Checking communication details after registration:");
            LogAssert.DoesNotContain(patient.GetCurrentContacts(), channel.phoneNumber);
        }

        /// <summary>
        /// Prepare registration details that include an email address and an IVC delivery method set to 'Email', and complete 
        /// all necessary fields for a successful registration of a consumer's eHealth record. Ensure that all email addresses 
        /// in the consumer's local health record  differ from the email address in the registration details.
        ///     a. Submit the registration request to the eHealth record system
        ///     b. Once the registration request is completed, verify that the CIS did not automatically update any of the 
        ///     communication details in the consumer's local health record with the email address supplied in the registration request. 
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020240_2()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 07");

            // Get the initial contact list
            List<Contact> contacts = patient.TargetPatient.Contacts;

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };
            
            // Create correspondence channel
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.email;
            channel.email = "philip.stalley-gordon@chamonix.com.au";

            Log("Checking communication details before registration:");
            LogAssert.DoesNotContain(patient.GetCurrentContacts(), channel.email);

            PcehrRegistrationResponse registrationResponse =  ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            Log("Checking communication details after registration:");
            LogAssert.DoesNotContain(patient.GetCurrentContacts(), channel.email);
        }


    }
}
