﻿using System.IO;
using System.Linq;
using System.Xml;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.DocumentConsumption
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  For All View Service Operations
    /// </summary>
    [TestClass]
    public class NOC_AllViewServiceOperations : CcaTest
    {
        /// <summary>
        /// Test No:         17
        /// Objective/Input:
        ///
        /// Ensure the client system includes PCEHR Common Header Elements and Transmission Signature containing a signed attestation of elements contained within the SOAP message on all SOAP Requests.
        ///
        /// - For this test case, the client system uses valid HPI-O certificates provided in the test data package. Service Invoker and Service Provider SHALL establish a secure communication channel when interacting with each other.
        /// - Implementation confirms to web services base profile and TLS security profile.
        /// - Unique message ID  is used in all transactions.
        /// - IHI number is specified in all transactions, is valid, matches the individual who owns the PCEHR and is numerical.
        /// - ID is specified in all transactions, has no leading or trailing spaces. If ID is HPII, ensure it is valid and numerical.
        /// - If role and/or alternate organisation is supplied, no leading or trailing spaces used
        /// - User name, vendor, product name, production version, platform, organisation ID, organisation name,  has no leading or trailing space, it is not blank.
        /// - Organisation name is set to the name of the organisation.
        /// - Product name, product version, product type and product platform are set to the appropriate values as per client system.
        ///
        /// Expected Result:
        ///
        /// The requests are transmitted successfully to the PCEHR System. Response is PCEHR_SUCCESS (SUCCESS).
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_017()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            PcehrHeaderTests headerTests = new PcehrHeaderTests(this);
            headerTests.CheckHeader(GetDocumentListActiveRequest(user, identifier), DialogueResource.HipsServiceGetDocumentListActive);
            headerTests.CheckHeader(GetChangeHistoryViewRequest(user, identifier), DialogueResource.HipsServiceGetChangeHistoryView);
        }

        private XmlDocument GetDocumentListActiveRequest(UserDetails user, PatientIdentifierBase identifier)
        {
            identifier.AlternateOrganisationName = "eHealth National Reform ";
            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentListActive(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListActive);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GetDocumentList, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            return doc;
        }

        private XmlDocument GetChangeHistoryViewRequest(UserDetails user, PatientIdentifierBase identifier)
        {
            // This is Test Document ID 106 in the Conformance Test Data for CIS connecting to PCEHR v1.8
            string documentUniqueId = CdaDocument.ConvertToOid("ebf96c6f-3252-47b0-aa89-b72740d5672d");

            // Get the Document Entry UUID for this Document ID.
            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentListActive(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListActive);
            DocumentMetaDataItem listItem = listResponse.DocumentList.FirstOrDefault(a => a.DocumentUniqueId == documentUniqueId);
            string documentEntryUuid = listItem.DocumentEntryUuid;

            // Get the list of historical versions and verify that there are 4 versions and all are event summaries
            ChangeHistoryViewResponse<PatientIdentifierBase> historyResponse = ProxyHelper.PcehrProxy.GetChangeHistoryView(identifier, user, documentEntryUuid);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, historyResponse.HipsResponse, DialogueResource.HipsServiceGetChangeHistoryView);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GetChangeHistoryView, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            return doc;
        }
    }
}