﻿using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.DocumentConsumption
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  getChangeHistoryView
    /// </summary>
    [TestClass]
    public class NOC_GetChangeHistoryView : CcaTest
    {
        /// <summary>
        /// Test No:         6
        /// Objective/Input: Ensure the DocumentID is used for the document unique
        ///                  identifier which is requested for the document change
        ///                  history view operation.
        /// Expected Result: The request is transmitted successfully. Confirmation
        ///                  that the payload is well formed and complies with the
        ///                  getChangeHistoryView Request XML Schema. The document
        ///                  id is provided.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_006()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            GainPcehrAccessResponse accessResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, accessResponse.HipsResponse, DialogueResource.HipsServiceGainAccessEmergency);

            // This is Test Document ID 106 in the Conformance Test Data for CIS connecting to PCEHR v1.8
            string documentUniqueId = CdaDocument.ConvertToOid("ebf96c6f-3252-47b0-aa89-b72740d5672d");

            // Get the Document Entry UUID for this Document ID.
            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentListActive(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListActive);
            DocumentMetaDataItem listItem = listResponse.DocumentList.FirstOrDefault(a => a.DocumentUniqueId == documentUniqueId);
            string documentEntryUuid = listItem.DocumentEntryUuid;

            // Get the list of historical versions and verify that there are 4 versions and all are event summaries
            ChangeHistoryViewResponse<PatientIdentifierBase> historyResponse = ProxyHelper.PcehrProxy.GetChangeHistoryView(identifier, user, documentEntryUuid);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, historyResponse.HipsResponse, DialogueResource.HipsServiceGetChangeHistoryView);
            LogAssert.AreEqual(4, historyResponse.DocumentList.Count, DialogueResource.DocumentListCount);
            foreach (DocumentMetaDataItem historyItem in historyResponse.DocumentList)
            {
                LogAssert.AreEqual("Event Summary", historyItem.DocumentClassName, DialogueResource.DocumentClassName);
            }

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GetChangeHistoryView, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            // Confirm that the payload is well formed and complies with the
            /// getChangeHistoryView Request XML Schema.
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/GetChangeHistoryView/1.0");
            XmlNode payload = doc.SelectSingleNode("/s:Envelope/s:Body/p:getChangeHistoryView", nsmgr);
            doc.Schemas.Add(nsmgr.LookupNamespace("s"), "soap-envelope.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("p"), "PCEHR_GetChangeHistoryView.xsd");
            try
            {
                doc.Validate(null, payload);
                LogAssert.IsTrue(true, "Validates against getChangeHistoryView schema", "no");
            }
            catch (XmlException ex)
            {
                LogAssert.Fail(ex.Message);
            }

            // Check that the document ID was provided in the request
            XmlNode documentIdNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:getChangeHistoryView/documentID", nsmgr);
            LogAssert.AreEqual(documentEntryUuid, documentIdNode.InnerText, "Document ID Node value");
        }

        /// <summary>
        /// Test No:         7
        /// Objective/Input: Verify that the client system can handle a
        ///                  Functional Error (PCEHR_ERROR_3002 - Document
        ///                  Metadata Failed Validation) when a get change
        ///                  history view operation is performed for a document
        ///                  that has been deleted.
        /// Expected Result: The request is transmitted successfully and a
        ///                  ( PCEHR_ERROR_3002 - Document Metadata Failed
        ///                  Validation ) is received. The client system is
        ///                  able to handle the error and inform the user
        ///                  appropriately.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_007()
        {
            // Create and upload the initial version of the document
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false);
            CdaDocument cdaDocument = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, cdaDocument);

            // Perform an operation to remove the document from the PCEHR System.
            byte[] auditInformation = Encoding.UTF8.GetBytes("PCEHR_NOC_7");
            HipsResponse response = ProxyHelper.PcehrProxy.Remove(patient.TargetPatientIdentifier, patient.TargetEpisode.AdmissionDate, cdaDocument.GetSetId(), RemovalReason.Withdrawn, patient.GetTestUser(), auditInformation);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, response, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout);

            // Gain emergency access to the PCEHR
            GainPcehrAccessResponse accessResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, accessResponse.HipsResponse, DialogueResource.HipsServiceGainAccessEmergency);

            DocumentQuery query = new DocumentQuery() { DocumentUniqueId = cdaDocument.GetDocId() };
            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentList(patient.TargetPatientIdentifier, patient.GetTestUser(), query);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListWithQuery);

            // Perform a get change history view operation for the document that has just been deleted
            string documentId = listResponse.DocumentList[0].DocumentEntryUuid;
            ChangeHistoryViewResponse<PatientIdentifierBase> historyResponse = ProxyHelper.PcehrProxy.GetChangeHistoryView(patient.TargetPatientIdentifier, patient.GetTestUser(), documentId);
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, historyResponse.HipsResponse, DialogueResource.HipsServiceGetChangeHistoryView);
            LogAssert.AreEqual("PCEHR_ERROR_3002", historyResponse.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
            LogAssert.AreEqual("Document metadata failed validation", historyResponse.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);
        }

        /// <summary>
        /// Test No:         8
        /// Objective/Input: To show the ability of the client system to handle
        ///                  a Standard Error (PCEHR_ERROR_0505 - Invalid HPI-O)
        ///                  when the user performs a  get change history view
        ///                  operation. The client system is configured with an
        ///                  invalid HPI-O.
        /// Expected Result: The request is transmitted successfully to the PCEHR
        ///                  System. A SOAP Fault with Standard Error
        ///                  PCEHR_ERROR_0505 - Invalid HPI-O is received. Client
        ///                  system is able to handle and inform the user
        ///                  regarding the error.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_008()
        {
            // Create the patient record
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false);

            HpiOHelper.ChangeHpiO(patient, this, makeValid: false);

            string documentId = string.Format("urn:uuid:{0}", Guid.NewGuid());
            ChangeHistoryViewResponse<PatientIdentifierBase> historyResponse = ProxyHelper.PcehrProxy.GetChangeHistoryView(patient.TargetPatientIdentifier, patient.GetTestUser(), documentId);
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, historyResponse.HipsResponse, DialogueResource.HipsServiceGetChangeHistoryView);
            LogAssert.AreEqual("PCEHR_ERROR_0505 - Invalid HPI-O", historyResponse.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);

            HpiOHelper.ChangeHpiO(patient, this, makeValid: true);
        }

        /// <summary>
        /// Test No:         9
        /// Objective/Input: Perform a getChangeHistoryView operation  with a
        ///                  Document unique ID which does not exist in the
        ///                  PCEHR System. IHI exists and is active in the
        ///                  PCEHR system. The client system uses Document ID
        ///                  of document which does not exist in the PCEHR
        ///                  system.
        /// Expected Result: The request is transmitted successfully to the
        ///                  PCEHR System. Client system is able to handle
        ///                  status Response Type Success and Empty Registry
        ///                  Object List.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_009()
        {
            // Create the patient record
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false);

            // Submit request for non-existent document
            string documentId = string.Format("urn:uuid:{0}", Guid.NewGuid());

            ChangeHistoryViewResponse<PatientIdentifierBase> historyResponse = ProxyHelper.PcehrProxy.GetChangeHistoryView(patient.TargetPatientIdentifier, patient.GetTestUser(), documentId);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, historyResponse.HipsResponse, DialogueResource.HipsServiceGetChangeHistoryView);
            LogAssert.AreEqual(0, historyResponse.DocumentList.Count, DialogueResource.NumberOfDocumentsInList);
        }
    }
}