﻿using System.IO;
using System.Xml;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.DocumentConsumption
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  getDocumentList
    /// </summary>
    [TestClass]
    public class NOC_GetDocumentList : CcaTest
    {
        /// <summary>
        /// Test No:         14
        /// Objective/Input:
        ///
        /// Invoke a Find Document query. Ensure Patient Id that needs to be used to find documents is in the correct format as shown in the following example:
        ///
        /// {0}^^^&1.2.36.1.2001.1003.0&ISO,
        ///
        /// where {0} – is the 16 digit HI number
        ///
        /// The query id parameter for the Find Document query is set to urn:uuid:14d4debf-8f97-4251-9a74-a90016b0af0d
        ///
        /// $XDSDocumentEntryStatus is set to urn:oasis:names:tc:ebxml-regrep:StatusType:Approved
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System.
        /// Confirmation that the payload is well formed and complies with the B2B DocumentRegistry interface.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_014()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentListActive(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListActive);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GetDocumentList, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            // Confirm that the payload is well formed and complies with the
            /// B2B DocumentRegistry interface.
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("q", "urn:oasis:names:tc:ebxml-regrep:xsd:query:3.0");
            nsmgr.AddNamespace("r", "urn:oasis:names:tc:ebxml-regrep:xsd:rim:3.0");
            XmlNode payload = doc.SelectSingleNode("/s:Envelope/s:Body/q:AdhocQueryRequest", nsmgr);
            doc.Schemas.Add(nsmgr.LookupNamespace("s"), "soap-envelope.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("q"), "query.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("r"), "rim.xsd");
            try
            {
                doc.Validate(null, payload);
                LogAssert.IsTrue(true, "Validates against SOAP, Query and RIM schemas", "no");
            }
            catch (XmlException ex)
            {
                LogAssert.Fail(ex.Message);
            }

            // Check that the patient ID was provided in the request
            XmlNode patientIdNode = doc.SelectSingleNode("/s:Envelope/s:Body/q:AdhocQueryRequest/r:AdhocQuery/r:Slot[@name=\"$XDSDocumentEntryPatientId\"]/r:ValueList/r:Value", nsmgr);
            string expected = string.Format("'{0}^^^&1.2.36.1.2001.1003.0&ISO'", patient.IhiInformation.Ihi);
            LogAssert.AreEqual(expected, patientIdNode.InnerText, "Patient ID Node value");

            // Check that the query id parameter for the Find Document query is set to urn:uuid:14d4debf-8f97-4251-9a74-a90016b0af0d
            XmlNode adhocQueryNode = doc.SelectSingleNode("/s:Envelope/s:Body/q:AdhocQueryRequest/r:AdhocQuery", nsmgr);
            LogAssert.AreEqual("urn:uuid:14d4debf-8f97-4251-9a74-a90016b0af0d", adhocQueryNode.Attributes["id"].Value, "The AdhocQuery id attribute value");

            // Check that $XDSDocumentEntryStatus is set to urn:oasis:names:tc:ebxml-regrep:StatusType:Approved
            XmlNode docStatusNode = doc.SelectSingleNode("/s:Envelope/s:Body/q:AdhocQueryRequest/r:AdhocQuery/r:Slot[@name=\"$XDSDocumentEntryStatus\"]/r:ValueList/r:Value", nsmgr);
            LogAssert.AreEqual("('urn:oasis:names:tc:ebxml-regrep:StatusType:Approved')", docStatusNode.InnerText, "The document status ID node value");
        }

        /// <summary>
        /// Test No:         15
        /// Objective/Input:
        ///
        /// To show the ability of the client system to handle an XDSRegistry error (PCEHR_ERROR_0505 - Invalid HPI-O)  when the get Document List operation is performed.
        /// The client system is configured with an invalid HPI-O.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. Client system is able to handle and inform the user regarding the XDSRegistryError (PCEHR_ERROR_0505 - Invalid HPI-O).
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_015()
        {
            // Create the patient record
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false);

            HpiOHelper.ChangeHpiO(patient, this, makeValid: false);

            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentListActive(patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, listResponse.HipsResponse, DialogueResource.HipsServiceGetChangeHistoryView);
            LogAssert.AreEqual("PCEHR_ERROR_0505", listResponse.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
            LogAssert.AreEqual("Invalid HPI-O", listResponse.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);

            HpiOHelper.ChangeHpiO(patient, this, makeValid: true);
        }

        /// <summary>
        /// Test No:         16
        /// Objective/Input:
        ///
        /// Perform a DocumentRegistry_RegistryStoredQuery B2B operation with an invalid Query id for the Adhoc Query.
        /// Pass an incorrect Query id in the request.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. The client system is able to handle the status Response Type Failure, and XDSRegistryError
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_016()
        {
            LogAssert.IsTrue(true, "The application does not provide any way to set the query id.", "no");
        }

        /// <summary>
        /// Test No:         109
        /// Objective/Input:
        /// Perform a getDocumentList operation for an IHI that does not have
        /// any documents persisted in the PCEHR System. IHI exists and is
        /// active in the PCEHR system. The client system uses IHI which does
        /// not have any documents in the PCEHR system.
        ///
        /// Expected Result:
        ///
        /// The request is transmitted successfully to the PCEHR System. Client
        /// system is able to handle status Response Type Success and Empty
        /// Registry Object List.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_109()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 2");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentListActive(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListActive);
            LogAssert.AreEqual(0, listResponse.DocumentList.Count, DialogueResource.NumberOfDocumentsInList);
        }

        /// <summary>
        /// Test No:         110
        /// Objective/Input:
        /// To show the ability of the client system to handle a functional
        /// error (PCEHR_ERROR_3002 - Document metadata failed validation).
        /// For this test case, the client system uses test data provided
        /// in the test data package.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. Client
        /// System is able to handle PCEHR_ERROR_3002 - Document metadata
        /// failed validation and inform the user appropriately.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_110()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 2");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            DocumentQuery query = new DocumentQuery();
            query.DocumentUniqueId = "abc";
            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentList(identifier, user, query);
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListWithQuery);
            LogAssert.AreEqual("PCEHR_ERROR_3002", listResponse.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
            LogAssert.AreEqual("Document metadata failed validation", listResponse.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);
        }
    }
}