﻿// -----------------------------------------------------------------------
// <copyright file="NOC_GetView.cs" company="NEHTA">
// Developed by Chamonix under contract to NEHTA.
// </copyright>
// -----------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Xml;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  getView
    /// </summary>
    [TestClass]
    public class NOC_GetView : CcaTest
    {
        /// <summary>
        /// Test No:         126
        ///
        /// Objective/Input:
        /// Perform a getView operation with valid input elements for a Prescription and Dispense View. 
        /// 
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. The client system is able
        /// to handle the response Status code PCEHR_SUCCESS ( response Status description SUCCESS)
        /// 
        /// A. The individual IHI Number is provided. 
        /// B. The version number of the corresponding view is provided.
        /// C. The fromDate and toDate are provided.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_126()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");

            PrescriptionAndDispenseViewRequest padvRequest = new PrescriptionAndDispenseViewRequest
            {
                FromDate = DateTime.Parse("1/1/2000"),
                ToDate = DateTime.Now
            };
            ViewResponse viewResponse = ProxyHelper.PcehrProxy.GetView(patient.TargetPatientIdentifier, patient.GetTestUser(), padvRequest);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, viewResponse.HipsResponse, DialogueResource.HipsServiceGetView);

            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GetView, audit.ServiceName, DialogueResource.PcehrAuditServiceName);
            XmlDocument soapRequest = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                soapRequest.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("soap", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("addr", "http://www.w3.org/2005/08/addressing");
            nsmgr.AddNamespace("core", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");
            nsmgr.AddNamespace("view", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/GetView/1.0");
            nsmgr.AddNamespace("padv", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/PrescriptionAndDispenseView/1.0");

            // A. The individual IHI Number is provided. 
            PcehrHeaderTests headerTests = new PcehrHeaderTests(this);
            headerTests.CheckHeader(soapRequest, DialogueResource.HipsServiceGetView);

            // B. The version number of the corresponding view is provided.
            string xPath = "/soap:Envelope/soap:Body/view:getView/view:view/padv:versionNumber";
            Log("Info: Searching payload for version number using XPath {0}", xPath);
            XmlNode node = soapRequest.SelectSingleNode(xPath, nsmgr);
            LogAssert.AreEqual("1.0", node.InnerText, "Version Number value");

            // C. The fromDate and toDate are provided.
            xPath = "/soap:Envelope/soap:Body/view:getView/view:view/padv:fromDate";
            Log("Info: Searching payload for from date using XPath {0}", xPath);
            node = soapRequest.SelectSingleNode(xPath, nsmgr);
            string expectedFromDate = padvRequest.FromDate.ToString("yyyy-MM-dd");
            LogAssert.AreEqual(expectedFromDate, node.InnerText, "fromDate value");

            xPath = "/soap:Envelope/soap:Body/view:getView/view:view/padv:toDate";
            Log("Info: Searching payload for toDate using XPath {0}", xPath);
            node = soapRequest.SelectSingleNode(xPath, nsmgr);
            string expectedToDate = padvRequest.ToDate.ToString("yyyy-MM-dd");
            LogAssert.AreEqual(expectedToDate, node.InnerText, "toDate value");
        }

        /// <summary>
        /// Test No:         129
        ///
        /// Objective/Input:
        /// Perform a getView operation for an IHI such that there is no data to display in the requested view. 
        /// This test case is performed by using the IHI provided in the test data package.
        /// 
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. Client system is able to handle status
        /// Response Type Success and Empty View.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_129()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_50");

            PrescriptionAndDispenseViewRequest padvRequest = new PrescriptionAndDispenseViewRequest
            {
                FromDate = DateTime.Parse("1/1/2000"),
                ToDate = DateTime.Parse("2/1/2000")
            };
            ViewResponse viewResponse = ProxyHelper.PcehrProxy.GetView(patient.TargetPatientIdentifier, patient.GetTestUser(), padvRequest);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, viewResponse.HipsResponse, DialogueResource.HipsServiceGetView);

            CdaDocument viewDoc = new CdaDocument(viewResponse.Document);
            IList<string> links = viewDoc.GetAllLinks();
            LogAssert.AreEqual(0, links.Count, "Number of document links in view result");
        }

        /// <summary>
        /// Test No:         130
        ///
        /// Objective/Input:
        /// Verify the client system ability to handle functional error. Ensure
        /// an incorrect start date or end date is passed in the request to the
        /// PCEHR System
        /// 
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System.  The
        /// PCEHR System SHALL return a failure response indicating the
        /// following error: 
        /// PCEHR_ERROR_0138 - Invalid start date
        /// or
        /// PCEHR_ERROR_0139 - Invalid end date
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_130()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_50");

            PrescriptionAndDispenseViewRequest padvRequest = new PrescriptionAndDispenseViewRequest
            {
                FromDate = DateTime.Parse("1/1/0001"),
                ToDate = DateTime.Parse("31/12/9999")
            };
            ViewResponse viewResponse = ProxyHelper.PcehrProxy.GetView(patient.TargetPatientIdentifier, patient.GetTestUser(), padvRequest);
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, viewResponse.HipsResponse, DialogueResource.HipsServiceGetView);
            LogAssert.AreEqual("PCEHR_ERROR_0139", viewResponse.HipsResponse.ResponseCode, "PCEHR Response Code");
            LogAssert.AreEqual("Invalid end date", viewResponse.HipsResponse.ResponseCodeDescription, "PCEHR Response Description");
        }

        /// <summary>
        /// Test No:         131
        ///
        /// Objective/Input:
        /// Perform a getView operation with invalid version Number.
        /// Pass an incorrect invalid document type in the request to the PCEHR System.
        /// 
        /// Expected Result:
        /// The request should not be transmitted to the PCEHR System.
        /// Client CIS or CSP system must inform the user that the document type is invalid.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_131()
        {
            Log(@"HIPS contains a class to represent each supported version of
each supported view. Within the class is a fixed value for the version number.
Since the version number is built into HIPS, not given to HIPS as input, it
does not need any run-time validation. The currently-supported version of the
Prescription and Dispense View is ""1.0"". Clients cannot change the version
number because it is compiled into the application. If any other views are
supported in future, or a new version of the Prescription and Dispense View,
a new class will be added to HIPS to represent the data structure of the
parameters for that version.");
        }

        /// <summary>
        /// Test No:         132
        ///
        /// Objective/Input:
        /// To show the ability of the client system to handle  Standard Error
        /// (PCEHR_ERROR_0505 – Invalid HPI-O) for remove document* operation.
        /// The client system is configured with an invalid HPI-O.
        /// 
        /// * Assuming that getView is intended here.
        /// 
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. Client
        /// system is able to handle and inform the user regarding the
        /// (PCEHR_ERROR_0505 - Invalid HPI-O). 
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_132()
        {
            Log(@"Info: The HPI-O will be changed in the database from the valid HPI-O to an invalid HPI-O.");

            try
            {
                patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_50");
                HpiOHelper.ChangeHpiO(patient, this, makeValid: false);

                PrescriptionAndDispenseViewRequest padvRequest = new PrescriptionAndDispenseViewRequest
                {
                    FromDate = DateTime.Parse("1/1/2001"),
                    ToDate = DateTime.Parse("2/2/2002")
                };
                ViewResponse viewResponse = ProxyHelper.PcehrProxy.GetView(patient.TargetPatientIdentifier, patient.GetTestUser(), padvRequest);
                LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, viewResponse.HipsResponse, DialogueResource.HipsServiceGetView);
                LogAssert.AreEqual("PCEHR_ERROR_0505", viewResponse.HipsResponse.ResponseCode, "PCEHR Response Code");
                LogAssert.AreEqual("Invalid HPI-O", viewResponse.HipsResponse.ResponseCodeDescription, "PCEHR Response Description");
            }
            finally
            {
                HpiOHelper.ChangeHpiO(patient, this, makeValid: true);
            }
        }

        /// <summary>
        /// Test No:         133
        ///
        /// Objective/Input:
        /// Verify that when IHI number is not provided in the input message,
        /// an error message appears and the request is not transmitted to
        /// PCEHR System. 
        /// 
        /// Expected Result:
        /// The request should not be transmitted to the PCEHR System.
        /// Client CIS or CSP system must inform the user that the IHI number
        /// MUST be supplied.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_133()
        {
            patient = CcaPatient.GetPatientWithoutIhi("CCA ID IHI_50");
            PrescriptionAndDispenseViewRequest padvRequest = new PrescriptionAndDispenseViewRequest
            {
                FromDate = DateTime.Parse("1/1/2001"),
                ToDate = DateTime.Parse("2/2/2002")
            };
            ViewResponse viewResponse = ProxyHelper.PcehrProxy.GetView(patient.TargetPatientIdentifier, patient.GetTestUser(), padvRequest);
            LogAssert.ExpectResponse(HipsResponseIndicator.InvalidIhi, viewResponse.HipsResponse, DialogueResource.HipsServiceGetView);
            LogAssert.AreEqual("No IHI was found for this patient and so the PCEHR cannot be accessed.", viewResponse.HipsResponse.HipsErrorMessage, "HIPS Error Message");
        }
    }
}