﻿// -----------------------------------------------------------------------
// <copyright file="NOC_RemoveDocument.cs" company="NEHTA">
// Developed by Chamonix for NEHTA.
// </copyright>
// -----------------------------------------------------------------------

using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Xml;
using System.Xml.Schema;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nehta.VendorLibrary.PCEHR;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.DocumentProduction
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  removeDocument
    /// </summary>
    [TestClass]
    public class NOC_RemoveDocument : CcaTest
    {
        /// <summary>
        /// Runs PCEHR_NOC_046 to test removal for Discharge Summary.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_046_DS()
        {
            PCEHR_NOC_046(SampleDocumentType.DischargeSummary);
        }

        /// <summary>
        /// Runs PCEHR_NOC_046 to test removal for Shared Health Summary.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_046_SHS()
        {
            /* This code grabs a shared health summary from the PCEHR - commented out as we want to use the customised one.
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            DocumentQuery query = new DocumentQuery { DocumentClassCode = new List<ClassCodes>() { ClassCodes.SharedHealthSummary } };
            DocumentListResponse<PatientIdentifierBase> list = ProxyHelper.PcehrProxy.GetDocumentList(patient.TargetPatientIdentifier, patient.GetTestUser(), query);
            DocumentRequest request = new DocumentRequest { DocumentUniqueId = list.DocumentList[0].DocumentUniqueId, RepositoryUniqueId = list.DocumentList[0].RepositoryUniqueId };
            DocumentResponse doc = ProxyHelper.PcehrProxy.RetrieveDocument(patient.TargetPatientIdentifier, patient.GetTestUser(), request);
            File.WriteAllBytes(@"C:\Work\NPDR\Testing\CcaNoc\SharedHealthSummary.xml", doc.Document);
            */
            PCEHR_NOC_046(SampleDocumentType.SharedHealthSummary);
        }

        /// <summary>
        /// Runs PCEHR_NOC_046 to test removal for Event Summary.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_046_ES()
        {
            /* This code grabs an event summary from the PCEHR - commented out as we want to use the customised one.
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            DocumentQuery query = new DocumentQuery { DocumentClassCode = new List<ClassCodes>() { ClassCodes.EventSummary } };
            DocumentListResponse<PatientIdentifierBase> list = ProxyHelper.PcehrProxy.GetDocumentList(patient.TargetPatientIdentifier, patient.GetTestUser(), query);
            DocumentRequest request = new DocumentRequest { DocumentUniqueId = list.DocumentList[0].DocumentUniqueId, RepositoryUniqueId = list.DocumentList[0].RepositoryUniqueId };
            DocumentResponse doc = ProxyHelper.PcehrProxy.RetrieveDocument(patient.TargetPatientIdentifier, patient.GetTestUser(), request);
            File.WriteAllBytes(@"C:\Work\NPDR\Testing\CcaNoc\EventSummary.xml", doc.Document);
            */

            PCEHR_NOC_046(SampleDocumentType.EventSummary);
        }

        /// <summary>
        /// Runs PCEHR_NOC_046 to test removal for Specialist Letter.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_046_SL()
        {
            /* This code grabs a specialist letter from the PCEHR - commented out as we want to use the customised one.
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            DocumentQuery query = new DocumentQuery { DocumentClassCode = new List<ClassCodes>() { ClassCodes.SpecialistLetter } };
            DocumentListResponse<PatientIdentifierBase> list = ProxyHelper.PcehrProxy.GetDocumentList(patient.TargetPatientIdentifier, patient.GetTestUser(), query);
            DocumentRequest request = new DocumentRequest { DocumentUniqueId = list.DocumentList[0].DocumentUniqueId, RepositoryUniqueId = list.DocumentList[0].RepositoryUniqueId };
            DocumentResponse doc = ProxyHelper.PcehrProxy.RetrieveDocument(patient.TargetPatientIdentifier, patient.GetTestUser(), request);
            File.WriteAllBytes(@"C:\Work\NPDR\Testing\CcaNoc\SpecialistLetter.xml", doc.Document);
            */
            PCEHR_NOC_046(SampleDocumentType.SpecialistLetter);
        }

        /// <summary>
        /// Runs PCEHR_NOC_046 to test removal for PCEHR Prescription Record.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_046_PR()
        {
            PCEHR_NOC_046(SampleDocumentType.PcehrPrescriptionRecord);
        }

        /// <summary>
        /// Runs PCEHR_NOC_046 to test removal for PCEHR Dispense Record.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_046_DR()
        {
            PCEHR_NOC_046(SampleDocumentType.PcehrDispenseRecord);
        }

        /// <summary>
        /// Test No:         46
        /// Objective/Input:
        ///
        /// Perform a Remove or deregister document operation to remove a document from the PCEHR system.
        /// This test case is performed by using the document id provided in the test data package.
        ///
        /// Expected Result:
        ///
        /// A. The removeDocument request is sent successfully by using the XDSDocumentEntry.uniqueId of the document to be removed.
        /// B. The operation request includes the full Document Exchange Common Header.
        /// C. The reasonForRemoval field is set to either 'IncorrectIdentity' or 'Withdrawn'.
        /// </summary>
        private void PCEHR_NOC_046(SampleDocumentType docType)
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document = patient.CreateNewDocument(docType);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document);

            byte[] auditInformation = UTF8Encoding.UTF8.GetBytes("PCEHR_NOC_46");
            HipsResponse removeResponse = ProxyHelper.PcehrProxy.Remove(identifier, patient.TargetEpisode.AdmissionDate, document.GetSetId(), RemovalReason.Withdrawn, user, auditInformation);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, removeResponse, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout);

            // Get the Audit
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.RemoveDocument, audit.ServiceName, DialogueResource.PcehrAuditServiceName);

            // Confirm that the payload is well formed and complies with the RemoveDocument Request XML Schema
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");
            nsmgr.AddNamespace("r", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/RemoveDocument/1.0");
            XmlNode payload = doc.SelectSingleNode("/s:Envelope/s:Body/r:removeDocument", nsmgr);
            doc.Schemas.Add(nsmgr.LookupNamespace("s"), "soap-envelope.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("c"), "PCEHR_CommonTypes.xsd");
            doc.Schemas.Add(nsmgr.LookupNamespace("r"), "PCEHR_RemoveDocument.xsd");
            try
            {
                doc.Validate(null);
                LogAssert.IsTrue(doc.SchemaInfo.Validity == XmlSchemaValidity.Valid,
                    "SOAP request validates against SOAP envelope schema",
                    "SOAP request is invalid");

                doc.Validate(null, payload);
                LogAssert.IsTrue(payload.SchemaInfo.Validity == XmlSchemaValidity.Valid,
                    "removeDocument payload validates against Remove Document schema.",
                    "removeDocument payload is invalid");
            }
            catch (XmlException ex)
            {
                LogAssert.Fail(ex.Message);
            }
            catch (XmlSchemaValidationException ex)
            {
                LogAssert.Fail(ex.Message);
            }

            PcehrHeaderTests headerTests = new PcehrHeaderTests(this);
            headerTests.CheckHeader(doc, DialogueResource.HipsServiceRemove);

            XmlNode reasonForRemovalNode = doc.SelectSingleNode("/s:Envelope/s:Body/r:removeDocument/r:reasonForRemoval", nsmgr);
            LogAssert.AreEqual("Withdrawn", reasonForRemovalNode.InnerText, "The reasonForRemoval field");
        }

        /// <summary>
        /// Test No:         47
        /// Objective/Input:
        /// To show the ability of the client system to handle functional error
        /// PCEHR_ERROR_2501 (Document not found) and inform the user correctly
        /// when a document is not found when performing a removeDocument
        /// operation. This test case is performed by using the document id
        /// provided in the test data package.
        ///
        /// Expected Result:
        /// The request should be transmitted successfully, PCEHR system
        /// response with a Functional Error PCEHR_ERROR_2501 (Document not
        /// found). Client system is able to process that error response and
        /// inform the client system user appropriately.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_047()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document);

            // Remove once
            byte[] auditInformation = UTF8Encoding.UTF8.GetBytes("PCEHR_NOC_47-a");
            HipsResponse removeResponse = ProxyHelper.PcehrProxy.Remove(identifier, patient.TargetEpisode.AdmissionDate, document.GetSetId(), RemovalReason.Withdrawn, user, auditInformation);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, removeResponse, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout, expectFailure: false);

            // Remove again
            byte[] auditInformation2 = UTF8Encoding.UTF8.GetBytes("PCEHR_NOC_47-b");
            HipsResponse removeResponse2 = ProxyHelper.PcehrProxy.Remove(identifier, patient.TargetEpisode.AdmissionDate, document.GetSetId(), RemovalReason.Withdrawn, user, auditInformation2);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, removeResponse2, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout, expectFailure: true);

            List<PcehrMessageQueue> queueOps = patient.GetPcehrMessageQueueItems();
            LogAssert.AreEqual(
                "PcehrServiceError, Code: PCEHR_ERROR_2501, Description: Document not found",
                queueOps[0].Details, DialogueResource.QueueOperationStatusDetail);
        }

        /// <summary>
        /// Test No:         48
        /// Objective/Input:
        /// To show the ability of the client system to handle Standard Error
        /// (PCEHR_ERROR_0505 – Invalid HPI-O) for remove document operation.
        /// The client system is configured with an invalid HPI-O.
        ///
        /// Expected Result:
        /// The request should be transmitted successfully to the PCEHR System
        /// and a SOAP Fault containing Standard Error PCEHR_ERROR_0505-Invalid
        /// HPI-O) is returned. Client system is able to process that error
        /// response and inform the client system user appropriately.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_048()
        {
            Log(@"Info: The HPI-O will be changed in the database from the valid HPI-O to an invalid HPI-O.");
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document);

            HpiOHelper.ChangeHpiO(patient, this, makeValid: false);

            byte[] auditInformation = UTF8Encoding.UTF8.GetBytes("PCEHR_NOC_48");
            HipsResponse removeResponse = ProxyHelper.PcehrProxy.Remove(identifier, patient.TargetEpisode.AdmissionDate, document.GetSetId(), RemovalReason.Withdrawn, user, auditInformation);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, removeResponse, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout, expectFailure: true);

            List<PcehrMessageQueue> queueOps = patient.GetPcehrMessageQueueItems();
            LogAssert.AreEqual(
                "PcehrServiceError, Code: badParam, Description: PCEHR_ERROR_0505 - Invalid HPI-O",
                queueOps[0].Details, DialogueResource.QueueOperationStatusDetail);

            HpiOHelper.ChangeHpiO(patient, this, makeValid: true);
        }

        /// <summary>
        /// Test No:         49
        /// Objective/Input:
        /// Perform a removeDocument operation by a Provider Organisation which
        /// is not the same as the Author Organisation of the document being
        /// removed. DocumentID  exists in the PCEHR System. HPI-O in request
        /// does not match the Author Organisation of the document in the PCEHR
        /// system.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. The
        /// client system is able to handle the errorCode  badParam and message
        /// (%Actual Error Message%).
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_049()
        {
            Log(@"Info: The hospital will be changed in the database from the first HPI-O to the second HPI-O.");
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 17");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            CdaDocument document = patient.CreateNewDocument(SampleDocumentType.DischargeSummary);
            QueueHelper.UploadDocumentAndWaitUntilProcessed(patient, LogAssert, document);

            HpiOHelper.SwitchToHpiO(patient, this, 2);

            byte[] auditInformation = UTF8Encoding.UTF8.GetBytes("PCEHR_NOC_49");
            HipsResponse removeResponse = ProxyHelper.PcehrProxy.Remove(identifier, patient.TargetEpisode.AdmissionDate, document.GetSetId(), RemovalReason.Withdrawn, user, auditInformation);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, removeResponse, DialogueResource.HipsServiceRemove);
            QueueHelper.WaitForQueuedOperation(patient, LogAssert, DialogueResource.RemoveWaitTimeout, expectFailure: true);

            List<PcehrMessageQueue> queueOps = patient.GetPcehrMessageQueueItems();
            LogAssert.AreEqual(
                "PcehrServiceError, Code: PCEHR_ERROR_3002, Description: Document metadata failed validation",
                queueOps[0].Details, DialogueResource.QueueOperationStatusDetail);

            HpiOHelper.SwitchToHpiO(patient, this, 1);
        }
    }
}