﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.SqlClient;
using System.Data;
using System.ComponentModel;
using System.Configuration;
using System.Data.SqlTypes;
using System.Text.RegularExpressions;
using System.Globalization;
using System.IO;

namespace HipsMonitoring
{
    public class HipsInformation : INotifyPropertyChanged
    {
        #region Members

        private string connectionString;
        private SqlConnection myConnection;

        private int ihiAlertCount;
        private int pasMessageCount;
        private int pasErrorMessageCount;
        private int pcehrPendingCount;
        private int pcehrFailedCount;
        private int pcehrOldestPendingMinutes;

        /// <summary>
        /// The hours for rolling average
        /// </summary>
        private int hoursForRollingAverage;

        /// <summary>
        /// How often in minutes to perform a refresh
        /// </summary>
        private int ihiRefreshTime;

        /// <summary>
        /// How often in minutes to perform a refresh
        /// </summary>
        private int pasRefreshTime;

        /// <summary>
        /// How many hours to view pas message failed
        /// </summary>
        private int pasFailureHours;

        ///  /// <summary>
        /// How often in minutes to view the IHI connection failure
        /// </summary>
        private int ihiConnectionTime;
        private int pcehrFailedDays;

        ///  /// <summary>
        /// Percentage of IHI failure
        /// </summary>
        private decimal ihiConnectionFailurePercentage;
        private decimal pcehrConnectionFailurePercentage;
        
        private double refreshTime;
        private decimal currentAverage;
        private decimal currentPcehrAverage;
        private int totalPasMessageCount;
        private int totalPatients;
        private int totalIhisFound;
        private int cdaDocumentsUploaded;
        private int pcehrAdvertised;

        private int consentPatientsChecked;
        private DateTime? consentTimeChecked;

        private const string DISPLAY_ALL = "ALL";
        private string maxListDisplay = DISPLAY_ALL;

        #endregion

        #region Properties
        /// <summary>
        /// Gets the IHI alert counts
        /// </summary>
        public int IhiAlertCount
        {
            get 
            {
                return ihiAlertCount;
            }
        }

        /// <summary>
        /// Gets the Pas message counts
        /// </summary>
        public int PasMessageCount
        {
            get
            {
                return pasMessageCount;
            }
        }

        /// <summary>
        /// Gets the Pas Error Count
        /// </summary>
        public int PasErrorMessageCount
        {
            get
            {
                return pasErrorMessageCount;
            }
        }

        /// <summary>
        /// Gets the failed PCEHR Error count
        /// </summary>
        public int PcehrFailedCount
        {
            get
            {
                return pcehrFailedCount;
            }
        }

        //Gets the PCEHR Pending Count
        public int PcehrPendingCount
        {
            get
            {
                return pcehrPendingCount;
            }
        }

        //Gets the oldest pending PCEHR minute
        public int PcehrOldestPendingMinutes
        {
            get
            {
                return pcehrOldestPendingMinutes;
            }
        }

        /// <summary>
        /// Gets and sets the IHI refresh time
        /// </summary>
        public int IhiRefreshTime
        {
            get
            {
                return ihiRefreshTime;
            }
            set
            {
                this.ihiRefreshTime = value;
            }
        }
        
        /// <summary>
        /// Gets and Sets the IHI Connection Time to refresh
        /// </summary>
        public int IhiConnectionTime
        {
            get
            {
                return ihiConnectionTime;
            }
            set
            {
                this.ihiConnectionTime = value;
            }
        }

        /// <summary>
        /// Gets and sets the PCEHR failed days to display
        /// </summary>
        public int PcehrFailedDays
        {
            get
            {
                return pcehrFailedDays;
            }
            set
            {
                this.pcehrFailedDays = value;
            }
        }

        /// <summary>
        /// Gets the ihiConnection Failure Percentage
        /// </summary>
        public decimal IhiConnectionFailurePercentage
        {
            get
            {
                return ihiConnectionFailurePercentage;
            }
        }

        /// <summary>
        /// Gets the PCEHR connection failure percentage
        /// </summary>
        public decimal PcehrConnectionFailurePercentage
        {
            get
            {
                return pcehrConnectionFailurePercentage;
            }
        }

        /// <summary>
        /// Gets and sets the Pas Refresh time to refresh the display
        /// </summary>
        public int PasRefreshTime
        {
            get
            {
                return pasRefreshTime;
            }
            set
            {
                this.pasRefreshTime = value;
            }
        }


        /// <summary>
        /// Gets and sets the Pas hours in failure to display
        /// </summary>
        public int PasFailureHours
        {
            get
            {
                return pasFailureHours;
            }
            set
            {
                this.pasFailureHours = value;
            }
        }

        /// <summary>
        /// Gets the refresh time for the display 
        /// </summary>
    
        public double RefreshTime
        {
            get
            {
                return this.refreshTime;
            }
        }

        /// <summary>
        /// Gets and sets the rolling average display
        /// </summary>
        public int HoursForRollingAverage
        {
            get
            {
                return this.hoursForRollingAverage;
            }
            set
            {
                this.hoursForRollingAverage = value;
            }

        }

        /// <summary>
        /// Gets the current average for the IHI connection
        /// </summary>
        public decimal CurrentAverage
        {
            get
            {
                return this.currentAverage;
            }
        }

        /// <summary>
        /// Gets the current average for the PCEHR connection
        /// </summary>
        public decimal CurrentPcehrAverage
        {
            get
            {
                return this.currentPcehrAverage;
            }
        }

        /// <summary>
        /// Gets the total Pas message count
        /// </summary>
        public int TotalPasMessageCount
        {
            get
            {
                return this.totalPasMessageCount;
            }
        }

        /// <summary>
        /// Gets the total patient count
        /// </summary>
        public int TotalPatients
        {
            get
            {
                return this.totalPatients;
            }
        }

        /// <summary>
        /// Gets the total ihi count
        /// </summary>
        public int TotalIhisFound
        {
            get
            {
                return this.totalIhisFound;
            }
            set
            {
                totalIhisFound = value;
                RaisePropertyChanged("TotalIhisFound");
            }
        }


        /// <summary>
        /// Get the Number of documents uploaded
        /// </summary>
        public int CdaDocumentsUploaded
        {
            get
            {
                return this.cdaDocumentsUploaded;
            }
        }

        /// <summary>
        /// Gets the number of patients with PCEHR
        /// </summary>
        public int PcehrAdvertised
        {
            get
            {
                return this.pcehrAdvertised;
            }
        }

        /// <summary>
        /// Gets the number of patients retireved the last time consent was checked.
        /// </summary>
        public int ConsentPatientsChecked
        {
            get
            {
                return this.consentPatientsChecked;
            }
        }


        /// <summary>
        /// Gets the Percentage of the IHI for the total patients.
        /// </summary>
        public decimal IhiPercentage
        {
            get
            {
                return (totalIhisFound == 0) ? 0 : ((decimal)totalIhisFound / (decimal)totalPatients)*100;
            }
        }

        /// <summary>
        /// Gets the Percentage of the PCEHR for the total patients.
        /// </summary>
        public decimal PcehrPercentage
        {
            get
            {
                return (pcehrAdvertised == 0) ? 0 : ((decimal)pcehrAdvertised / (decimal)totalPatients)*100;
            }
        }

        /// <summary>
        /// Gets the date and time the consent service was last checked.
        /// </summary>
        public DateTime? ConsentTimeChecked
        {
            get
            {
                return this.consentTimeChecked;
            }
        }

        /// <summary>
        /// Gets and sets the maximum to display in the lists - defaults to all
        /// </summary>
        public string MaxListDisplay
        {
            get
            {
                return maxListDisplay;
            }
            set
            {
                this.maxListDisplay = value;
            }
        }

        #endregion

        #region Constructor
        /// <summary>
        /// Initializes a new instance of the <see cref="HipsInformation" /> class.
        /// </summary>
        public HipsInformation()
        {
            connectionString = ConfigurationManager.ConnectionStrings["MonitoringConnectionString"].ToString();
            myConnection = new SqlConnection(connectionString);
            //messageTable = new List<InfoMessages>();
            //Refresh();
        }
        #endregion

        #region Methods

        /// <summary>
        /// Does a complete refresh of all metrics
        /// </summary>
        public void Refresh()
        {
            DateTime start = DateTime.Now;
            GetIHIConnectionFailurePercenatage();
            GetPcehrConnectionFailurePercentage();
            GetMonitoringItemCounts();
            GetConsentHeartBeat();
            refreshTime = DateTime.Now.Subtract(start).Milliseconds;

        }

        
        /// <summary>
        /// Gets the failure % of IHI lookups
        /// </summary>
        /// <returns></returns>
        public decimal GetIHIConnectionFailurePercenatage()
        {
            try
            {
                myConnection.Open();
                SqlCommand myCommand = new SqlCommand("hips.MonitoringIhiConnectionFailure", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;
                myCommand.Parameters.Add(new SqlParameter("@Minutes", SqlDbType.Int)).Value = this.ihiConnectionTime;
                decimal count = Convert.ToDecimal(myCommand.ExecuteScalar());
                ihiConnectionFailurePercentage = count;

                return count;
            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }

        }

        public decimal GetPcehrConnectionFailurePercentage()
        {
            try
            {
                myConnection.Open();
                SqlCommand myCommand = new SqlCommand("hips.MonitoringPcehrConnectionFailure", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;
                myCommand.Parameters.Add(new SqlParameter("@Minutes", SqlDbType.Int)).Value = this.ihiConnectionTime;
                decimal count = Convert.ToDecimal(myCommand.ExecuteScalar());
                pcehrConnectionFailurePercentage = count;

                return count;
            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the TOP select 
        /// </summary>
        /// <returns></returns>
        private string GetSelect()
        {
            string sql = "SELECT ";
            if (!maxListDisplay.Equals(DISPLAY_ALL))
            {
                sql = sql + "TOP " + maxListDisplay + " ";
            }
            return sql;
        }
       
        /// <summary>
        /// Gets the HIPS errors from the SystemErrorLog
        /// </summary>
        public DataTable GetHIPSErrors()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringHIPSErrors ORDER BY DateCreated DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the PCEHR errors from the SystemErrorLog
        /// </summary>
        public DataTable GetPcehrErrors()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringPcehrErrors ORDER BY DateCreated DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the IHI Connection errors from the SystemErrorLog
        /// </summary>
        public DataTable GetIHIConnectionErrors()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringIHILookupConnectionErrors ORDER BY DateCreated DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the IHI Lookup Data errors from the SystemErrorLog
        /// </summary>
        public DataTable GetIHILookupInfo()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringIHILookupDataErrors ORDER BY DateCreated DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the Info Messages from the SystemErrorLog
        /// </summary>
        public DataTable GetInfoMessages()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringInfoMessages ORDER BY DateCreated DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the PCEHR Info Messages from the SystemErrorLog
        /// </summary>
        public DataTable GetPcehrInfoMessages()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringPcehrInfoMessages ORDER BY DateCreated DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the Background Processor Messages from the SystemErrorLog
        /// </summary>
        public DataTable GetBackGroundProcessorMessages()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringBackgroundProcessorMessages ORDER BY DateCreated DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the MSMQ Messages Messages from the SystemErrorLog
        /// </summary>
        public DataTable GetMsmqMessages()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringMsmqMessages ORDER BY DateCreated DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the PUMA Error Messages Messages from the SystemErrorLog
        /// </summary>
        public DataTable GetPumaErrorMessages()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringPumaErrorMessages ORDER BY DateCreated DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the Project CDA Documents
        /// </summary>
        public DataTable GetPcehrUploadedDocuments()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringUploadedDocuments ORDER BY 'Upload DateTime' DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }


        /// <summary>
        /// Gets the PCEHR Errors and Details for the last 7 days
        /// </summary>
        public DataTable GetPcehrUploadErrors()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                //this is not limited by the max display combo as it is ALL errors
                //dataAdapter = new SqlDataAdapter("SELECT * FROM hips.MonitoringPcehrUploadErrors WHERE DATEDIFF(day, DateCreated, SYSDATETIME())<=" + this.PcehrFailedDays + " ORDER BY PcehrMessageQueueId DESC", connectionString);
                dataAdapter = new SqlDataAdapter("SELECT * FROM hips.MonitoringPcehrUploadErrors ORDER BY PcehrMessageQueueId DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

        /// <summary>
        /// Gets the Merge Info errors from the SystemErrorLog
        /// </summary>
        public DataTable GetMergeInfo()
        {
            try
            {
                SqlDataAdapter dataAdapter = new SqlDataAdapter();
                myConnection.Open();
                dataAdapter = new SqlDataAdapter(GetSelect() + " * FROM hips.MonitoringMergeInfoMessages ORDER BY DateCreated DESC", connectionString);
                DataTable table = new DataTable();
                dataAdapter.Fill(table);
                return table;

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }
       
          /// <summary>
        /// Gets the monitoring item counts for 
        /// - CDA Documents uploaded
        /// - Number of PCEHRs advertised
        /// </summary>
        /// <returns></returns>
        public void GetMonitoringItemCounts()
        {
            try
            {
                myConnection.Open();
                SqlCommand myCommand = new SqlCommand("hips.MonitoringItemCounts", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;
                myCommand.Parameters.Add(new SqlParameter("@PcehrFailedDays", SqlDbType.Int)).Value = this.PcehrFailedDays;
                myCommand.Parameters.Add(new SqlParameter("@MonitorHours", SqlDbType.Int)).Value = this.hoursForRollingAverage;
                myCommand.Parameters.Add(new SqlParameter("@PasRefreshMinutes", SqlDbType.Int)).Value = this.pasRefreshTime;
                myCommand.Parameters.Add(new SqlParameter("@IhiRefreshMinutes", SqlDbType.Int)).Value = this.ihiRefreshTime;
                
                SqlDataReader reader = myCommand.ExecuteReader();

                if (reader.HasRows)
                {
                    reader.Read();

                    if (reader[reader.GetOrdinal("CDAUploaded")] == DBNull.Value)
                    {
                        this.cdaDocumentsUploaded = 0;
                    }
                    else
                    {
                        this.cdaDocumentsUploaded = reader.GetInt32((reader.GetOrdinal("CDAUploaded")));
                    }


                    if (reader[reader.GetOrdinal("PCEHRAdvertised")] == DBNull.Value)
                    {
                        this.pcehrAdvertised = 0;
                    }
                    else
                    {
                        this.pcehrAdvertised = reader.GetInt32((reader.GetOrdinal("PCEHRAdvertised")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringPcehrOldestPendingMinutes")] == DBNull.Value)
                    {
                        this.pcehrOldestPendingMinutes = 0;
                    }
                    else
                    {
                        this.pcehrOldestPendingMinutes = reader.GetInt32((reader.GetOrdinal("MonitoringPcehrOldestPendingMinutes")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringRecentPcehrFailedCount")] == DBNull.Value)
                    {
                        this.pcehrFailedCount = 0;
                    }
                    else
                    {
                        this.pcehrFailedCount = reader.GetInt32((reader.GetOrdinal("MonitoringRecentPcehrFailedCount")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringRecentPcehrPendingCount")] == DBNull.Value)
                    {
                        this.pcehrPendingCount = 0;
                    }
                    else
                    {
                        this.pcehrPendingCount = reader.GetInt32((reader.GetOrdinal("MonitoringRecentPcehrPendingCount")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringPasErrorCount")] == DBNull.Value)
                    {
                        this.pasErrorMessageCount = 0;
                    }
                    else
                    {
                        this.pasErrorMessageCount = reader.GetInt32((reader.GetOrdinal("MonitoringPasErrorCount")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringIhiAverageTime")] == DBNull.Value)
                    {
                        this.currentAverage = 0;
                    }
                    else
                    {
                        this.currentAverage = reader.GetDecimal((reader.GetOrdinal("MonitoringIhiAverageTime")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringPcehrAverageTime")] == DBNull.Value)
                    {
                        this.currentPcehrAverage = 0;
                    }
                    else
                    {
                        this.currentPcehrAverage = reader.GetDecimal((reader.GetOrdinal("MonitoringPcehrAverageTime")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringPasMessageCount")] == DBNull.Value)
                    {
                        this.pasMessageCount = 0;
                    }
                    else
                    {
                        this.pasMessageCount = reader.GetInt32((reader.GetOrdinal("MonitoringPasMessageCount")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringPatientCount")] == DBNull.Value)
                    {
                        this.totalPatients = 0;
                    }
                    else
                    {
                        this.totalPatients = reader.GetInt32((reader.GetOrdinal("MonitoringPatientCount")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringTotalMessages")] == DBNull.Value)
                    {
                        this.totalPasMessageCount = 0;
                    }
                    else
                    {
                        this.totalPasMessageCount = reader.GetInt32((reader.GetOrdinal("MonitoringTotalMessages")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringUniqueIhiCount")] == DBNull.Value)
                    {
                        TotalIhisFound = 0;
                    }
                    else
                    {
                        TotalIhisFound = reader.GetInt32((reader.GetOrdinal("MonitoringUniqueIhiCount")));
                    }


                    if (reader[reader.GetOrdinal("MonitoringIhiAlertCount")] == DBNull.Value)
                    {
                        ihiAlertCount = 0;
                    }
                    else
                    {
                        ihiAlertCount = reader.GetInt32((reader.GetOrdinal("MonitoringIhiAlertCount")));
                    }
                    
                    
                    

                }
                reader.Close();
            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
        }

       

        /// <summary>
        /// Gets the date and time of the last Consent request from the HIB
        /// </summary>
        public void GetConsentHeartBeat()
        {
            int numberOfPatients = 0;
            DateTime? dateLastChecked = DateTime.MinValue;

            try
            {
                myConnection.Open();
                SqlCommand myCommand = new SqlCommand("SELECT ExceptionMessage, DateCreated FROM hips.MonitoringConsentHeartBeat", myConnection);
                SqlDataReader reader = myCommand.ExecuteReader();

                if (reader.HasRows)
                {
                    reader.Read();

                    if (reader[reader.GetOrdinal("ExceptionMessage")] != DBNull.Value)
                    {
                        string exmessge =  reader.GetString((reader.GetOrdinal("ExceptionMessage")));
                        numberOfPatients = Int32.Parse(Regex.Match(exmessge, @"\d+").Value.ToString());
                    }
                    else 
                    {
                        numberOfPatients = 0;
                    }

                    if (reader[reader.GetOrdinal("DateCreated")] != DBNull.Value)
                    {
                        dateLastChecked = reader.GetDateTime((reader.GetOrdinal("DateCreated")));
                    }
                    else 
                    {
                        dateLastChecked = null;
                    }

                }
                else
                {
                    numberOfPatients = 0;
                    dateLastChecked = null; 
                }
                

            }
            finally
            {
                myConnection.Close();
                consentPatientsChecked = numberOfPatients;
                consentTimeChecked = dateLastChecked;
            }
        }

        /// <summary>
        /// Gets the Document (decoded base64) that was sent to the PCEHR in the the request 
        /// (/s:Envelope/s:Body/x:ProvideAndRegisterDocumentSetRequest/x:Document)
        /// </summary>
        public bool GetPcehrFailedUploadDocument(string fileName, int PcehrMessageQueueId)
        {
            bool successfulSave = false;
            try
            {
                myConnection.Open();
                SqlCommand myCommand = new SqlCommand("hips.MonitoringFailedUploadDocument", myConnection);
                myCommand.CommandType = CommandType.StoredProcedure;
                myCommand.Parameters.Add(new SqlParameter("@PcehrMessageQueueId", SqlDbType.Int)).Value = PcehrMessageQueueId;
                
                string encoded = Convert.ToString(myCommand.ExecuteScalar());
                byte[] decodedBytes = null;

                if (encoded != null || !encoded.Equals(""))
                {
                    //decode from base64
                    decodedBytes = Convert.FromBase64String(encoded);

                    try
                    {
                        System.IO.FileStream fileStream = new System.IO.FileStream(fileName, System.IO.FileMode.Create, System.IO.FileAccess.Write);
                        fileStream.Write(decodedBytes, 0, decodedBytes.Length);
                        fileStream.Close();
                        successfulSave = true;

                    }
                    catch (IOException e)
                    {
                        throw e;
                    }
                }

            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                myConnection.Close();
            }
            return successfulSave;
        }
      
        #endregion

        #region Private Methods

        public static DateTime GetFirstDateFromString(string input)
        {
            string pattern = @"[0-9]{2}/[0-9]{2}/[0-9]{4} [0-9]{2}:[0-9]{2}:[0-9]{2}";
            Match match = Regex.Match(input, pattern);
            DateTime result;
            if (DateTime.TryParseExact(match.Value, "dd/MM/yyyy HH:mm:ss", CultureInfo.CurrentCulture, DateTimeStyles.None, out result))
            {
                return result;
            }

            return DateTime.MinValue;
        }

        #endregion

        #region Implement INotifyPropertyChanged
        public event PropertyChangedEventHandler PropertyChanged;
        public void RaisePropertyChanged(string propertyName)
        {
            PropertyChangedEventHandler handler = PropertyChanged;
            if (handler != null) handler(this, new PropertyChangedEventArgs(propertyName));
        }
        #endregion
    }
}
