﻿#region Using Directives
using System;
using System.Transactions;
using System.Collections;
using PUMA.ServiceContracts.ServiceInterface;
using System.Collections.Generic;
using HIPS.CommonSchemas;
using System.Collections.ObjectModel;
using PUMA.PcehrDataStore.Schemas;
using PUMA.PcehrDataStore.DataAccess;
using System.Security.Permissions;
using HIPS.Client.Proxy;
using PUMA.AppServer.ServiceHost.Services;
using HIPS.CommonSchemas.PatientIdentifier;
using System.ServiceModel;
using HIPS.PcehrDataStore.Schemas;
#endregion

namespace PUMA.AppServer.ServiceHost
{
    /// <summary>
    /// 
    /// 
    /// 
    /// </summary>
    public class EpisodeService : BaseService, IEpisodeService
    {
        /// <summary>
        /// Gets the specified episode id.
        /// </summary>
        /// <param name="episodeId">The episode id.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "ManageConsentAccess")]
        public EpisodeTile Get(int episodeId)
        {
            EpisodeDl dataAccess = new EpisodeDl(Helpers.SecurityHelper.GetUserDetail());
            return dataAccess.Get(episodeId);
        }
        
        /// <summary>
        /// Gets the a list episodes with uploaded documents for a specific patient.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="startRow">The start row.</param>
        /// <param name="endRow">The end row.</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "RemoveDischargeAccess")]
        public ObservableCollection<EpisodeTile> GetEpisodesWithUploadedDocuments(int patientId, int startRow, int endRow)
        {
            EpisodeDl dataAccess = new EpisodeDl(Helpers.SecurityHelper.GetUserDetail());
            return dataAccess.GetEpisodesWithUploadedDocuments(patientId, startRow, endRow);
        }

        /// <summary>
        /// Gets the count of episodes with uploaded document for the selected patient.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "RemoveDischargeAccess")]
        public int GetEpisodesWithUploadedDocumentCount(int patientId)
        {
            EpisodeDl dataAccess = new EpisodeDl(Helpers.SecurityHelper.GetUserDetail());
            return dataAccess.GetEpisodesWithUploadedDocumentCount(patientId);
        }

        /// <summary>
        /// Gets a list of episodes for a specific patient.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="startRow">The start row.</param>
        /// <param name="endRow">The end row.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "ManageConsentAccess")]
        public ObservableCollection<EpisodeTile> GetEpisodes(int patientId, int startRow, int endRow)
        {
            EpisodeDl dataAccess = new EpisodeDl(Helpers.SecurityHelper.GetUserDetail());
            return dataAccess.GetEpisodes(patientId, startRow, endRow);
        }

        /// <summary>
        /// Gets a count of episodes for the selected patient.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "ManageConsentAccess")]
        public int GetEpisodesCount(int patientId)
        {
            EpisodeDl dataAccess = new EpisodeDl(Helpers.SecurityHelper.GetUserDetail());
            return dataAccess.GetEpisodesCount(patientId);
        }

        /// <summary>
        /// Records if the patient has withdrawn their consent to upload documents to the PCEHR for the current episode.
        /// </summary>
        /// <param name="pcehrWithdrawn">if set to <c>true</c> [pcehr withdrawn].</param>
        /// <param name="admissionDate">The admission date.</param>
        /// <param name="patientIdentifier">The patient identifier.</param>
        /// <param name="user">The user.</param>
        /// <param name="auditInformation">The audit information.</param>
        /// <returns></returns>
        /// <exception cref="System.Exception">Connection could not be established</exception>
        [PrincipalPermission(SecurityAction.Demand, Role = "ManageConsentAccess")]
        public bool RecordConsent(bool pcehrWithdrawn, DateTime admissionDate, PatientIdentifierBase patientIdentifier, byte[] auditInformation)
        {
            UserDetails user = Helpers.SecurityHelper.GetUserDetail();
            
            using (ConsentProxy proxy = new ConsentProxy(EndPointResource.ConsentEndPoint))
            {
                try
                {
                    HipsResponse response = proxy.RecordConsent(pcehrWithdrawn, patientIdentifier, admissionDate, user, auditInformation);
                    return response.Status == HipsResponseIndicator.OK;
                }
                catch (EndpointNotFoundException ex)
                {
                    proxy.Abort();
                    throw new Exception("Connection could not be established", ex);
                }
                catch (Exception e)
                {
                    proxy.Abort();
                    throw e;
                }
            }
        }

        /// <summary>
        /// Gets the current consent reason.
        /// </summary>
        /// <param name="patientIdentifier">The patient identifier.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "ManageConsentAccess")]
        public string GetCurrentConsentReason(int episodeId)
        {
            string result = string.Empty;

            ConsentAuditDl consentDl = new ConsentAuditDl();
            ConsentAudit consentAudit = consentDl.GetCurrentConsentAudit(episodeId);

            if (consentAudit.AuditInformation != null)
            {
                result = System.Text.Encoding.UTF8.GetString(consentAudit.AuditInformation);
            }

            return result;
        }
    }
}

