﻿#region Using Directives
using System;
using System.Transactions;
using System.Collections;
using PUMA.ServiceContracts.ServiceInterface;
using PUMA.PcehrDataStore.DataAccess;
using PUMA.PcehrDataStore.Schemas;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.CommonSchemas;
using HIPS.Client.Proxy;
using PUMA.AppServer.ServiceHost.Services;
using System.ServiceModel;
using HIPS.PcehrDataStore.Schemas;
using System.Security.Permissions;
#endregion

namespace PUMA.AppServer.ServiceHost
{
    /// <summary>
    /// This service provides access to the Patient Data access class
    /// </summary>
    public class PatientService : BaseService, IPatientService
    {

        /// <summary>
        /// Gets the specified patient id.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="includeFullDetails">if set to <c>true</c> [include full details].</param>
        /// <param name="patient">The patient.</param>
        /// <param name="lastUpdate">The last update.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "NoAccess")]
        public bool Get(int patientId, bool includeFullDetails, out Patient patient, DateTime? lastUpdate = null)
        {
            PatientDl dataAccess = new PatientDl();
            return dataAccess.Get(patientId, includeFullDetails, out patient, null);
        }

        /// <summary>
        /// Gets a list of patients that are in hospital (or left after a set number of days) who still need to have
        /// thier IHI found
        /// </summary>
        /// <param name="hospitalId">The hospital id.</param>
        /// <param name="daysToCheckAfterDischarge">The days to check after discharge.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "MissingIHIAccess")]
        public List<PatientsMissingIhiReport> PatientsWithoutIhi(int? hospitalId)
        {
            PatientDl dataAccess = new PatientDl();
            return dataAccess.PatientsWithoutIhi(hospitalId);
        }

        [PrincipalPermission(SecurityAction.Demand, Role = "NoAccess")]
        public ObservableCollection<PatientTile> PatientsWithIhi(int? hospitalId)
        {
            PatientDl dataAccess = new PatientDl();
            return dataAccess.PatientsWithIhi(hospitalId);
        }

        /// <summary>
        /// Gets a list of Patients that have overriden PAS data
        /// </summary>
        /// <param name="hospitalId">The hospital id.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "MissingIHIAccess")]
        public List<PatientsWithOverridesReport> PatientsWithOverridesReport(int? hospitalId)
        {
            PatientDl dataAccess = new PatientDl();
            return dataAccess.PatientsWithOverridesReport(hospitalId);
        }

        /// <summary>
        /// Updates the specified patient.
        /// </summary>
        /// <param name="patient">The patient.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "NoAccess")]
        public bool Update(Patient patient)
        {
            PatientDl dataAccess = new PatientDl();
            return dataAccess.Update(patient);
        }

        /// <summary>
        /// Records the disclosure of PCEHR.
        /// </summary>
        /// <param name="pcehrDisclosed">if set to <c>true</c> [pcehr disclosed].</param>
        /// <param name="patientIdentifier">The patient identifier.</param>
        /// <param name="user">The user.</param>
        /// <param name="auditInformation">The audit information.</param>
        /// <returns></returns>
        /// <exception cref="System.Exception">Connection could not be established</exception>
        [PrincipalPermission(SecurityAction.Demand, Role = "DisclosePcehrAccess")]
        public bool RecordDisclosure(bool pcehrDisclosed, PatientIdentifierBase patientIdentifier, byte[] auditInformation)
        {
            using (ConsentProxy proxy = new ConsentProxy(EndPointResource.ConsentEndPoint))
            {
                try
                {
                    HipsResponse response = proxy.RecordDisclosure(pcehrDisclosed, patientIdentifier, Helpers.SecurityHelper.GetUserDetail(), auditInformation);
                    return response.Status == HipsResponseIndicator.OK;
                }
                catch (EndpointNotFoundException ex)
                {
                    proxy.Abort();
                    throw new Exception("Connection could not be established", ex);
                }
                catch (Exception e)
                {
                    proxy.Abort();
                    throw e;
                }
            }
        }

        /// <summary>
        /// Gets the current disclosure reason.
        /// </summary>
        /// <param name="healthOrganisationProviderId">The health organisation provider id.</param>
        /// <param name="patientMasterId">The patient master id.</param>
        /// <returns></returns>
        [PrincipalPermission(SecurityAction.Demand, Role = "DisclosePcehrAccess")]
        public string GetCurrentDisclosureReason(int healthOrganisationProviderId, int patientMasterId)
        {
            string result = string.Empty;

            DisclosureAuditDl disclosureDl = new DisclosureAuditDl();
            DisclosureAudit disclosureAudit = disclosureDl.GetCurrentDisclosureAudit(healthOrganisationProviderId, patientMasterId);

            if (disclosureAudit.AuditInformation != null)
            {
                result = System.Text.Encoding.UTF8.GetString(disclosureAudit.AuditInformation);
            }

            return result;
        }
    }
}

