﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace PUMA.Helpers
{
    /// <summary>
    /// This class holds all the date helper functions
    /// </summary>
    static public class DateHelper
    {
        #region Private Members
        static private int[] monthDay;
        #endregion

        /// <summary>
        /// Gets the age to be displayed from two dates
        /// </summary>
        /// <param name="fromDate">From date.</param>
        /// <param name="toDate">To date.</param>
        /// <returns></returns>
        /// <exception cref="System.Exception">Start date is greated than end date</exception>
        static public string Age(DateTime fromDate, DateTime toDate)
        {
            monthDay = new int[12] { 31, -1, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
            if (fromDate > toDate)
            {
                throw new Exception("Start date is greated than end date");
            }
            int increment = 0;

            // The number of days old, excluding whole months or years.
            // For example, if a child is "1 year 2 months 3 days old" this is the number 3.
            int days = CalculateDays(fromDate, toDate, ref increment);
            
            // The total number of weeks old
            // For example, if a child is "1 year 2 months 3 days old" this is the number 61.
            int weeks = CalculateWeeks(fromDate, toDate);
            
            // The number of months old, excluding whole years
            // For example, if a child is "1 year 2 months 3 days old" this is the number 2 (rather than 14).
            int months = CalculateMonths(fromDate, toDate, ref increment);

            // The number of years old
            // For example, if a child is "1 year 2 months 3 days old" this is the number 1.
            int years = CalculateYears(fromDate, toDate, increment);

            // Adjust the number of months to include whole years.
            // For example, if a child is "1 year 2 months 3 days old" this is the number 14 (rather than 2).
            months += years * 12;

            string result = string.Empty;
            if (years > 2)
            {
                result = string.Format(DateResource.AgeInYears, years);
            }
            else if (months > 2)
            {
                result = string.Format(DateResource.AgeInMonths, months);
            }
            else if (weeks > 2)
            {
                result = string.Format(DateResource.AgeInWeeks, weeks);
            }
            else
            {
                result = (days > 1) ? string.Format(DateResource.AgeInDays, days) : string.Format(DateResource.AgeInDay, days);
            }
            return result;
        }

        /// <summary>
        /// Calculates the number of days.
        /// </summary>
        /// <param name="fromDate">From date.</param>
        /// <param name="toDate">To date.</param>
        /// <returns></returns>
        static private int CalculateDays(DateTime fromDate, DateTime toDate, ref int increment)
        {
            int days = 0;
            if (fromDate.Day > toDate.Day)
            {
                increment = monthDay[fromDate.Month - 1];
            }
            if (increment == -1)
            {
                if (DateTime.IsLeapYear(fromDate.Year))
                {
                    increment = 29;
                }
                else
                {
                    increment = 28;
                }
            }

            if (increment != 0)
            {
                days = (toDate.Day + increment) - fromDate.Day;
                increment = 1;
            }
            else
            {
                days = toDate.Day - fromDate.Day;
            }
            return days;
        }


        /// <summary>
        /// Calculates the weeks.
        /// </summary>
        /// <param name="fromDate">From date.</param>
        /// <param name="toDate">To date.</param>
        /// <returns></returns>
        static private int CalculateWeeks(DateTime fromDate, DateTime toDate)
        {
            TimeSpan span = toDate - fromDate;
            int days = (int)span.TotalDays;
            return days / 7;
        }
        /// <summary>
        /// Calculates the months.
        /// </summary>
        /// <param name="fromDate">From date.</param>
        /// <param name="toDate">To date.</param>
        /// <param name="increment">The increment.</param>
        /// <returns></returns>
        static private int CalculateMonths(DateTime fromDate, DateTime toDate, ref int increment)
        {
            int months = 0;
            if ((fromDate.Month + increment) > toDate.Month)
            {
                months = (toDate.Month + 12) - (fromDate.Month + increment);
                increment = 1;
            }
            else
            {
                months = (toDate.Month) - (fromDate.Month + increment);
                increment = 0;
            }
            return months;
        }
        /// <summary>
        /// Calculates the years.
        /// </summary>
        /// <param name="fromDate">From date.</param>
        /// <param name="toDate">To date.</param>
        /// <param name="increment">The increment.</param>
        /// <returns></returns>
        static private int CalculateYears(DateTime fromDate, DateTime toDate, int increment)
        {
            return  toDate.Year - (fromDate.Year + increment);
        }
    }
}
