﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.SqlClient;
using PUMA.PcehrDataStore.Schemas;
using PUMA.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.Common.DataStore.DataAccess;
using System.Collections.ObjectModel;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace PUMA.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the episode view
    /// </summary>
    public class EpisodeDl : BaseDl
    {
        #region Constructor
        public EpisodeDl(HIPS.CommonSchemas.UserDetails userDetails)
            : base(userDetails)
        {
        }
        #endregion

        #region Methods

        /// <summary>
        /// Gets the specified episode id.
        /// </summary>
        /// <param name="episodeId">The episode id.</param>
        /// <returns></returns>
        public EpisodeTile Get(int episodeId)
        {
            EpisodeTile result = new EpisodeTile();
            try
            {
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.EpisodeGet))
                {
                    AddIntParameter(ParametersResource.EpisodeId, episodeId, command);
                    PopulateBusinessObject<EpisodeTile>(command.ExecuteReader(), result);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageEpisodeGet, ex, User, LogMessage.PUMA_MESSAGE_1006);
            }
            return result;
        }

        /// <summary>
        /// Gets the episodes with uploaded documents.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="startRow">The start row.</param>
        /// <param name="endRow">The end row.</param>
        /// <returns></returns>
        public ObservableCollection<EpisodeTile> GetEpisodesWithUploadedDocuments(int patientId, int startRow, int endRow)
        {
            ObservableCollection<EpisodeTile> results = new ObservableCollection<EpisodeTile>();
            try
            {
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.EpisodesWithUploadedDocuments))
                {
                    AddIntParameter(ParametersResource.PatientId, patientId, command);
                    AddIntParameter(ParametersResource.RowNumberStart, startRow, command);
                    AddIntParameter(ParametersResource.RowNumberEnd, endRow, command);
                    results = GetPopulatedBusinessObservableCollection<EpisodeTile>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageEpisodesWithDocumentGet, ex, User, LogMessage.PUMA_MESSAGE_1007);
            }
            return results;
        }

        /// <summary>
        /// Gets the count of episodes with uploaded document for the selected patient.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <returns></returns>
        public int GetEpisodesWithUploadedDocumentCount(int patientId)
        {
            int result = 0;
            try
            {
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.EpisodesWithUploadedDocumentsCount))
                {
                    AddIntParameter(ParametersResource.PatientId, patientId, command);
                    result = (int)command.ExecuteScalar();
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageEpisodesWithDocumentCount, ex, User, LogMessage.PUMA_MESSAGE_1008);
            }
            return result;
        }

        /// <summary>
        /// Gets all episodes for a patient.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="startRow">The start row.</param>
        /// <param name="endRow">The end row.</param>
        /// <returns></returns>
        public ObservableCollection<EpisodeTile> GetEpisodes(int patientId, int startRow, int endRow)
        {
            ObservableCollection<EpisodeTile> results = new ObservableCollection<EpisodeTile>();
            try
            {
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.EpisodesGet))
                {
                    AddIntParameter(ParametersResource.PatientId, patientId, command);
                    AddIntParameter(ParametersResource.RowNumberStart, startRow, command);
                    AddIntParameter(ParametersResource.RowNumberEnd, endRow, command);
                    results = GetPopulatedBusinessObservableCollection<EpisodeTile>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageEpisodesGet, ex, User, LogMessage.PUMA_MESSAGE_1009);
            }
            return results;
        }

        /// <summary>
        /// Gets the count of episodes for the selected patient.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <returns></returns>
        public int GetEpisodesCount(int patientId)
        {
            int result = 0;
            try
            {
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.EpisodesCount))
                {
                    AddIntParameter(ParametersResource.PatientId, patientId, command);
                    result = (int)command.ExecuteScalar();
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageEpisodesCount, ex, User, LogMessage.PUMA_MESSAGE_1010);
            }
            return result;
        }


        /// <summary>
        /// Updates the specified episode id.
        /// </summary>
        /// <param name="episodeId">The episode id.</param>
        /// <param name="ihiInvestigationComplete">if set to <c>true</c> [ihi investigation complete].</param>
        /// <returns></returns>
        public bool Update(int episodeId, bool ihiInvestigationComplete)
        {
            bool result = true;
            try
            {
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.EpisodeUpdate))
                {
                    AddIntParameter(ParametersResource.EpisodeId, episodeId, command);
                    AddBoolParameter(ParametersResource.IhiInvestigationComplete, ihiInvestigationComplete, command);
                    AddStringParameter("UserModified", User.Name, command);
                    command.ExecuteNonQuery();
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                result = false;
                EventLogger.WriteLog(ConstantsResource.ErrorMessageMenuGet, ex, User, LogMessage.PUMA_MESSAGE_1011);
            }
            return result;

        }
        #endregion
    }
}
