﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using PUMA.PcehrDataStore.Schemas;
using System.Configuration;
using System.Data.SqlClient;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonSchemas;
using System.Collections.ObjectModel;
using PUMA.Helpers;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace PUMA.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the patient table
    /// </summary>
    public class PatientDl : BaseDl
    {
        #region Private Members
        EpisodeDl episodeAccess;
        #endregion

        #region Properties

        /// <summary>
        /// Gets the episode access.
        /// </summary>
        private EpisodeDl EpisodeAccess
        {
            get
            {
                if (episodeAccess == null)
                {
                    episodeAccess = new EpisodeDl(SecurityHelper.GetUserDetail());
                }
                return episodeAccess;
            }
        }
        #endregion

        #region Methods
        /// <summary>
        /// Gets the specified patient by id and if needed fills theall the associated data like Episodes.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="includeFullDetails">if set to <c>true</c> [include full details].</param>
        /// <param name="patient">The patient.</param>
        /// <param name="lastUpdate">The last update.</param>
        /// <returns></returns>
        public bool Get(int patientId, bool includeFullDetails, out Patient patient, DateTime? lastUpdate = null)
        {
            patient = new Patient();
            bool result = false;
            try
            {
                //TODO: Need to sort out the Last Update Read logic
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.PatientMasterGet))
                {
                    AddIntParameter(ParametersResource.PatientId, patientId, command);
                    AddDateParameter(ParametersResource.DateLastUpdated, lastUpdate, command);
                    result = PopulateBusinessObject<Patient>(command.ExecuteReader(), patient);
                    command.Connection.Close();
                    if (includeFullDetails)
                    {
                        //patient.EpisodeList = EpisodeAccess.GetAll(patientID, lastUpdate);
                        //patient.PathologyList = PathologyAccess.GetAll(patientID, lastUpdate);
                    }
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessagePatientMasterGetById, ex, new UserDetails(), LogMessage.PUMA_MESSAGE_1017);
            }
            return result;
        }



        /// <summary>
        /// Gets a list of patients that are in hospital (or left after a set number of days) who still need to have
        /// thier IHI found
        /// </summary>
        /// <param name="hospitalId">The hospital id.</param>
        /// <param name="daysToCheckAfterDischarge">The days to check after discharge.</param>
        /// <returns></returns>
        public List<PatientsMissingIhiReport> PatientsWithoutIhi(int? hospitalId)
        {
            int daysAfterDischarge;
            List<PatientsMissingIhiReport> items = new List<PatientsMissingIhiReport>();

            try
            {
                daysAfterDischarge = Int32.Parse(ConfigurationManager.AppSettings["DaysAfterDischarge"]);

                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.PatientsWithoutIhi))
                {
                    AddIntParameter(ParametersResource.HospitalId, hospitalId, command);
                    AddIntParameter(ParametersResource.NumberOfDaysAfterDischarge, daysAfterDischarge, command);
                    items = GetPopulatedBusinessList<PatientsMissingIhiReport>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorPatientMissingIhiReport, ex, new UserDetails(), LogMessage.PUMA_MESSAGE_1018);
            }
            return items;
        }

        public ObservableCollection<PatientTile> PatientsWithIhi(int? hospitalId)
        {
            ObservableCollection<PatientTile> items = new ObservableCollection<PatientTile>();
            try
            {
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.PatientTileGet))
                {
                    AddIntParameter(ParametersResource.HospitalId, hospitalId, command);
                    AddBoolParameter(ParametersResource.PatientsHaveIhi, true, command);
                    items = GetPopulatedBusinessObservableCollection<PatientTile>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorPatientMissingIhiReport, ex, new UserDetails(), LogMessage.PUMA_MESSAGE_1019);
            }
            return items;
        }

        /// <summary>
        /// Gets a list of patients with Overriden PAS Data
        /// </summary>
        /// <param name="hospitalId">The hospital id.</param>
        /// <returns></returns>
        public List<PatientsWithOverridesReport> PatientsWithOverridesReport(int? hospitalId)
        {
            List<PatientsWithOverridesReport> items = new List<PatientsWithOverridesReport>();
            try
            {
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.PatientsWithOverridesReport))
                {
                    AddIntParameter(ParametersResource.HospitalId, hospitalId, command);
                    items = GetPopulatedBusinessList<PatientsWithOverridesReport>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorPatientMissingIhiReport, ex, new UserDetails(), LogMessage.PUMA_MESSAGE_1020);
            }
            return items;
        }

         /// <summary>
        /// Updates the specified patient.
        /// </summary>
        /// <param name="patient">The patient.</param>
        /// <returns></returns>
        public bool Update(Patient patient)
        {
            bool result = false;

            using (SqlCommand command = GetSqlCommand("dbo.PatientMasterPatientGet"))
            {
                Patient foundPatient = new Patient();
                //command.Parameters.Add(new SqlParameter("@PatientMasterID", patient.PatientMasterID));
                result = PopulateBusinessObject<Patient>(command.ExecuteReader(), foundPatient);
                command.Connection.Close();
            }

            if (result)
            {
                using (SqlCommand command = GetSqlCommand("dbo.PatientMasterPatientUpdate"))
                {
                    result = base.Update<Patient>(patient, command);
                }
            }
            else
            {
                using (SqlCommand command = GetSqlCommand("dbo.PatientMasterPatientInsert"))
                {
                    result = base.Insert<Patient>(patient, command);
                }
            }
            return result;
        }       
        #endregion

       

    }
}
