﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using HIPS.Common.PcehrDataStore.DataAccess;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using PUMA.Operation;
using System.ServiceModel;
using System.Data;
using PUMA.Helpers;
using HIPS.CommonSchemas;
using System.DirectoryServices.AccountManagement;
using System.DirectoryServices;
using System.Configuration;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace PUMA.PcehrDataStore.DataAccess
{
    public class SecurityDl : BaseDl
    {
        #region Private Members

        private string ldapPathString;

        #endregion

        #region Constructor

        /// <summary>
        /// Initializes a new instance of the <see cref="SecurityDl"/> class.
        /// </summary>
        /// <exception cref="System.Exception">No LDAP path configured.</exception>
        public SecurityDl() : base()
        {
            ldapPathString = ConfigurationManager.AppSettings["LdapPath"];
            if (String.IsNullOrEmpty(ldapPathString))
            {
                throw new Exception(ConstantsResource.ErrorMessageLDAPPathNotConfigured);
            }
        }
        
        #endregion

        #region Methods

        /// <summary>
        /// Gets the user permission.
        /// </summary>
        /// <returns>A list of permissions for the user.</returns>
        public List<String> GetUserPermission()
        {
            UserDetails user = SecurityHelper.GetUserDetail();
            List<String> results = new List<String>();
            try
            {
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.UserFunctionalPermissionCacheGet))
                {
                    AddStringParameter(ParametersResource.Username, user.Login, command);
                    AddStringParameter(ParametersResource.IPAddress, PumaOperationContext.Current.IPAddress, command);
                    AddStringParameter(ParametersResource.MachineName, PumaOperationContext.Current.MachineName, command);
                    AddIntParameter(ParametersResource.HospitalId, PumaOperationContext.Current.HospitalId, command);

                    SqlDataReader reader = command.ExecuteReader();
                    while (reader.Read())
                    {
                        String permission = reader["FunctionalPermissionName"].ToString();
                        results.Add(permission);
                    }
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageHospitalGet, ex, user, LogMessage.PUMA_MESSAGE_1027);
            }
            return results;
        }

        /// <summary>
        /// Creates the user permission.
        /// </summary>
        public void CreateUserPermission(out string firstName, out string lastName, out string hpii)
        {
            firstName = string.Empty;
            lastName = string.Empty;
            hpii = string.Empty;

            UserDetails user = SecurityHelper.GetUserDetail();

            try
            {
                List<string> hadGroupList = GetHadGroups(out firstName, out lastName, out hpii);
                
                using (SqlCommand command = GetSqlCommand(StoredProceduresResource.UserFunctionalPermissionCacheInsert))
                {
                    AddStringParameter(ParametersResource.Username, user.Login, command);
                    AddStringParameter(ParametersResource.IPAddress, PumaOperationContext.Current.IPAddress, command);
                    AddStringParameter(ParametersResource.MachineName, PumaOperationContext.Current.MachineName, command);
                    AddStringParameter("@UserModified", SecurityHelper.GetUserDetail().Login, command);

                    DataTable hadGroups = new DataTable();
                    hadGroups.Columns.Add(new DataColumn("HadGroupName", typeof(string)));
                    foreach (string hadGroupName in hadGroupList)
                    {
                        hadGroups.Rows.Add(hadGroupName);
                    }

                    SqlParameter parameter;
                    parameter = command.Parameters.AddWithValue(ParametersResource.HadGroups, hadGroups);
                    parameter.SqlDbType = SqlDbType.Structured;
                    parameter.TypeName = StoredProceduresResource.HadGroupTVP;

                    command.ExecuteNonQuery();

                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageHospitalGet, ex, user, LogMessage.PUMA_MESSAGE_1028);
            }
        }

        /// <summary>
        /// Gets the had groups.
        /// </summary>
        /// <returns>A list of HadGroups</returns>
        public List<String> GetHadGroups(out string firstName, out string lastName, out string hpii)
        {
            List<String> results = new List<String>();
            UserDetails user = SecurityHelper.GetUserDetail();

            firstName = user.Login;
            lastName = "Local";
            hpii = string.Empty;

#if PUMA_LOCAL_SECURITY

            try
            {
                using (SqlCommand command = GetSqlCommand("puma.LocalSecurityGet"))
                {
                    AddStringParameter(ParametersResource.Username, user.Login, command);

                    SqlDataReader reader = command.ExecuteReader();
                    while (reader.Read())
                    {
                        String permission = reader["HadGroupName"].ToString();
                        results.Add(permission);
                    }
                    
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageLocalSecurityGet, ex, user, LogMessage.PUMA_MESSAGE_1029);
            }

#else
            try
            {
                results = SelectGroupsByUserName(user.Login, out firstName, out lastName, out hpii);
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageADSecurityGet, ex, user, LogMessage.PUMA_MESSAGE_1030);
            }
#endif

            return results;
        }

        #endregion

        #region Active Directory Methods

        /// <summary>
        /// Returns a list of all groups, including nested, provided user is a member of.
        /// </summary>
        /// <param name="userName">sAMAccountName of User</param>
        /// <returns>List of All Groups (including nested)</returns>
        public List<String> SelectGroupsByUserName(string userName, out string firstName, out string lastName, out string hpii)
        {
            List<String> groups = new List<String>();

            SearchResult result = PerformLDAPSingleItemQuery("(|(&(objectCategory=Person)(samAccountName=" + userName + ")))");

            if (result == null) throw new Exception(ConstantsResource.ErrorMessageUsernameNotInAD);

            string sid = GenerateADString(result);
            firstName = GetStringValue(result, "givenName");
            lastName = GetStringValue(result, "sn");
            hpii = GetStringValue(result, "hpii");

            groups = GetUserGroupList(sid);

            return groups;

        }

        /// <summary>
        /// Performs the LDAP single item query.
        /// </summary>
        /// <param name="query">The query.</param>
        /// <returns></returns>
        private SearchResult PerformLDAPSingleItemQuery(string query)
        {
            SearchResult Result = null;
            using (DirectorySearcher mDirectorySearcher = new DirectorySearcher(ldapPathString))
            {

                mDirectorySearcher.Filter = query;

                mDirectorySearcher.SearchScope = SearchScope.Subtree;

                AddDirectorySearcherProperties(mDirectorySearcher);

                Result = mDirectorySearcher.FindOne();
            }
            return Result;
        }

        /// <summary>
        /// Adds the directory searcher properties.
        /// </summary>
        /// <param name="directorySearcher">The directory searcher.</param>
        private static void AddDirectorySearcherProperties(DirectorySearcher directorySearcher)
        {
            directorySearcher.PropertiesToLoad.Add("objectSid");
            directorySearcher.PropertiesToLoad.Add("givenName");
            directorySearcher.PropertiesToLoad.Add("samAccountName");
            directorySearcher.PropertiesToLoad.Add("sn");
            directorySearcher.PropertiesToLoad.Add("mail");
            directorySearcher.PropertiesToLoad.Add("objectClass");
            directorySearcher.PropertiesToLoad.Add("hpii");
        }

        /// <summary>
        /// Gets the user group list.
        /// </summary>
        /// <param name="sid">The sid.</param>
        /// <returns></returns>
        internal List<String> GetUserGroupList(string sid)
        {
            List<String> groupNameList = new List<String>();
            List<String> sidList = new List<String>();

            using (DirectorySearcher directorySearcher = new DirectorySearcher(ldapPathString))
            {
                sidList = this.GetUserGroupSidList(sid);

                foreach (string groupSid in sidList)
                {
                    directorySearcher.Filter = "(objectsid=" + groupSid + ")";

                    SearchResult result = directorySearcher.FindOne();

                    if (result != null)
                    {
                        groupNameList.Add(GenerateADString(result));
                    }
                }
            }

            return groupNameList;
        }

        /// <summary>
        /// Generates the AD string.
        /// </summary>
        /// <param name="searchResult">The search result.</param>
        /// <returns></returns>
        protected string GenerateADString(SearchResult searchResult)
        {
            string result = string.Empty;
            System.Security.Principal.SecurityIdentifier ItemSID = new System.Security.Principal.SecurityIdentifier((byte[])searchResult.Properties["objectSid"][0], 0);

            string ObjectCategory = searchResult.GetDirectoryEntry().SchemaClassName;
            if (ObjectCategory.ToUpperInvariant().Trim() == "USER")
            {
                result = ItemSID.Value;
            }
            else if (ObjectCategory.ToUpperInvariant().Trim() == "GROUP")
            {
                result = GetStringValue(searchResult, "samAccountName");
            }

            return result;
        }

        /// <summary>
        /// Gets the string value.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <param name="property">The property.</param>
        /// <returns></returns>
        protected static string GetStringValue(SearchResult item, string property)
        {
            if (item.Properties.Contains(property))
            {
                if (item.Properties[property].Count > 0)
                {
                    return item.Properties[property][0].ToString();
                }
            }
            return string.Empty;
        }

        /// <summary>
        /// Gets the user group sid list.
        /// </summary>
        /// <param name="userSid">The user sid.</param>
        /// <returns></returns>
        internal List<string> GetUserGroupSidList(string userSid)
        {
            List<string> groupSidList = new List<string>();

            using (DirectorySearcher directorySearcher = new DirectorySearcher(ldapPathString))
            {
                directorySearcher.Filter = "(objectsid=" + userSid + ")";

                SearchResult sr = directorySearcher.FindOne();

                if (sr != null)
                {
                    using (DirectoryEntry user = new DirectoryEntry(sr.Path))
                    {
                        user.RefreshCache(new string[] { "tokenGroups" });

                        foreach (byte[] sid in user.Properties["tokenGroups"])
                        {
                            System.Security.Principal.SecurityIdentifier objectSid = new System.Security.Principal.SecurityIdentifier(sid, 0);
                            groupSidList.Add(objectSid.Value);
                        }
                    }
                }
            }

            return groupSidList;
        }

        #endregion
    }
}
