﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using PUMA.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.Base.Schemas;
using HIPS.Base.Schemas.Enumerators;

namespace PUMA.PcehrDataStore.Schemas
{
    /// <summary>
    /// This class represents a patient with all of their details
    /// </summary>
    [Serializable]
    public class OriginalPatient : BaseInformationSchema
    {
        #region Private Members
        private string ihi;
        private bool? pcehrExists;
        private bool isInHospital;
        private int dischargesOutstanding;
        private int possibleOutstandingPcehrUpload;
        private string mrn;
        private string registeredLastName;
        private string registeredFirstNames;
        private string currentLastName;
        private string currentFirstNames;
        private string medicareNumber;
        private string medicareNumberSequence;
        private string dvaNumber;
        private string addressLine1;
        private string addressLine2;
        private string placeName;
        private string postCode;
        private DateTime dateOfBirth;
        private DateTime? dateOfDeath;
        private string languageSpoken;
        private int countryID;
        private string countryCode;
        private string countryDescription;
        private int maritalStatusID;
        private string maritalStatusCode;
        private string maritalStatusDescription;
        private int nationalityId;
        private string nationalityCode;
        private string nationalityDescription;
        //private int indigenousStatusId;
        //private string indigenousStatusCode;
        //private string indigenousStatusDescription;
        private int religionID;
        private string religionCode;
        private string religionDescription;
        private int currentSexID;
        private string currentSexCode;
        private string currentSexDescription;
        private int registeredSexID;
        private string registeredSexCode;
        private string registeredSexDescription;
        private int registeredTitleID;
        private string registeredTitleCode;
        private string registeredTitleDescription;
        private int currentTitleID;
        private string currentTitleCode;
        private string currentTitleDescription;
        private int registeredSuffixID;
        private string registeredSuffixCode;
        private string registeredSuffixDescription;
        private int currentSuffixID;
        private string currentSuffixCode;
        private string currentSuffixDescription;
        private int stateID;
        private string stateCode;
        private string stateDescription;
        private string homePhoneNumber;
        private string workPhoneNumber;
        private string email;
        private string mobile;
        private int newRadiologyCount;
        private int newProcedureCount;
        private int newPathologyCount;
        private int newEpisodeCount;
        private int newDrgCount;
        private int newDiagnosisCount;
        private int numberOutstandingIndicators;
        #endregion
        /// <summary>
        /// Gets or sets the patient master id.
        /// </summary>
        /// <value>
        /// The patient master id.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int PatientMasterID { get; set; }
        /// <summary>
        /// Gets or sets the patient id.
        /// </summary>
        /// <value>
        /// The patient id.
        /// </value>
        public int? PatientID
        {
            get
            {
                return base.Id;
            }
            set
            {
                base.Id = value;
            }
        }
        /// <summary>
        /// Gets or sets the ihi.
        /// </summary>
        /// <value>
        /// The ihi.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string Ihi
        {
            get
            {
                //if (string.IsNullOrEmpty(ihi))
                //{
                //    return "Unknown";
                //}
                //if (!IsIhiValid)
                //{
                //    return "Requires Validation";
                //}
                return ihi;
            }
            set
            {
                ihi = value;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether [pcehr exists].
        /// </summary>
        /// <value>
        ///   <c>true</c> if [pcehr exists]; otherwise, <c>false</c>.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public bool? PcehrExists
        {
            get
            {
                return pcehrExists;
            }
            set
            {
                pcehrExists = value;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether this patient is in hospital.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is in hospital; otherwise, <c>false</c>.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public bool IsInHospital
        {
            get
            {
                return isInHospital;
            }
            set
            {
                isInHospital = value;
                base.OnPropertyChanged("IsInHospital");
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether this instance is discharge outstanding.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is discharge outstanding; otherwise, <c>false</c>.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public int DischargesOutstanding
        {
            get
            {
                return dischargesOutstanding;
            }
            set
            {
                dischargesOutstanding = value;
                base.OnPropertyChanged("DischargesOutstanding");
            }
        }

        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public int PossibleOutstandingPcehrUpload
        {
            get
            {
                return possibleOutstandingPcehrUpload;
            }
            set
            {
                possibleOutstandingPcehrUpload = value;
            }
        }
        
        /// <summary>
        /// Gets the pcehr exists description.
        /// </summary>
        public string PcehrExistsDescription
        {
            get
            {
                if (PcehrExists.HasValue == true)
                {
                    if (PcehrExists.Value)
                    {
                        return "PCEHR Exists";
                    }
                    else
                    {
                        return "No PCEHR";
                    }
                }
                else
                {
                    return "PCEHR status is unknown";
                }
            }
        }



        /// <summary>
        /// Gets a value indicating whether this instance is ihi valid.
        /// </summary>
        /// <value>
        /// 	<c>true</c> if this instance is ihi valid; otherwise, <c>false</c>.
        /// </value>
        public bool IsIhiValid
        {
            get
            {
                TimeSpan duration = DateTime.Now.Subtract(IhiDateLastValidated);
                if (duration.TotalHours > 24)
                {
                    return false;
                }
                return true;
            }
        }
        /// <summary>
        /// Gets or sets the ihi date last validated.
        /// </summary>
        /// <value>
        /// The ihi date last validated.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public DateTime IhiDateLastValidated { get; set; }
        /// <summary>
        /// Gets or sets the hospital id.
        /// </summary>
        /// <value>
        /// The hospital id.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public int HospitalID { get; set; }
        /// <summary>
        /// Gets or sets the hospital code.
        /// </summary>
        /// <value>
        /// The hospital code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string HospitalCode { get; set; }
        /// <summary>
        /// Gets or sets the hospital description.
        /// </summary>
        /// <value>
        /// The hospital description.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string HospitalDescription { get; set; }
        /// <summary>
        /// Gets or sets the MRN.
        /// </summary>
        /// <value>
        /// The MRN.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string Mrn 
        {
            get
            {
                return mrn;
            }
            set
            {
                mrn = value;
                base.OnPropertyChanged("Mrn");
                base.OnPropertyChanged("CurrentListName");
            }
        }
        /// <summary>
        /// Gets or sets the last name of the registered.
        /// </summary>
        /// <value>
        /// The last name of the registered.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string RegisteredLastName 
        {
            get
            {
                return registeredLastName;
            }
            set
            {
                registeredLastName = value;
                base.OnPropertyChanged("RegisteredLastName");
                base.OnPropertyChanged("FullRegisteredName");
            }
        }
        /// <summary>
        /// Gets or sets the registered first names.
        /// </summary>
        /// <value>
        /// The registered first names.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string RegisteredFirstNames
        {
            get
            {
                return registeredFirstNames;
            }
            set
            {
                registeredFirstNames = value;
                base.OnPropertyChanged("RegisteredFirstNames");
                base.OnPropertyChanged("FullRegisteredName");
            }
        }
        /// <summary>
        /// Gets or sets the last name of the current.
        /// </summary>
        /// <value>
        /// The last name of the current.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string CurrentLastName
        {
            get
            {
                return currentLastName;
            }
            set
            {
                currentLastName = value;
                base.OnPropertyChanged("CurrentLastName"); 
                base.OnPropertyChanged("FullCurrentName");
                base.OnPropertyChanged("CurrentListName");
            }
        }
        /// <summary>
        /// Gets or sets the current first names.
        /// </summary>
        /// <value>
        /// The current first names.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string CurrentFirstNames
        {
            get
            {
                return currentFirstNames;
            }
            set
            {
                currentFirstNames = value;
                base.OnPropertyChanged("CurrentFirstNames");
                base.OnPropertyChanged("FullCurrentName");
                base.OnPropertyChanged("CurrentListName");
            }
        }
        /// <summary>
        /// Gets or sets the medicare number.
        /// </summary>
        /// <value>
        /// The medicare number.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string MedicareNumber
        {
            get
            {
                return medicareNumber;
            }
            set
            {
                medicareNumber = value;
                base.OnPropertyChanged("MedicareNumber");
            }
        }
        /// <summary>
        /// Gets or sets the medicare number sequence.
        /// </summary>
        /// <value>
        /// The medicare number sequence.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string MedicareNumberSequence
        {
            get
            {
                return medicareNumberSequence;
            }
            set
            {
                medicareNumberSequence = value;
                base.OnPropertyChanged("MedicareNumberSequence");
            }
        }
        /// <summary>
        /// Gets or sets the dva number.
        /// </summary>
        /// <value>
        /// The dva number.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string DvaNumber
        {
            get
            {
                return dvaNumber;
            }
            set
            {
                dvaNumber = value;
                base.OnPropertyChanged("DvaNumber");
            }
        }

        /// <summary>
        /// Gets or sets the address line1.
        /// </summary>
        /// <value>
        /// The address line1.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string AddressLine1
        {
            get
            {
                return addressLine1;
            }
            set
            {
                addressLine1 = value;
                base.OnPropertyChanged("AddressLine1");
                base.OnPropertyChanged("Address");
            }
        }
        /// <summary>
        /// Gets or sets the address line2.
        /// </summary>
        /// <value>
        /// The address line2.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string AddressLine2
        {
            get
            {
                return addressLine2;
            }
            set
            {
                addressLine2 = value;
                base.OnPropertyChanged("AddressLine2");
                base.OnPropertyChanged("Address");
            }
        }
        /// <summary>
        /// Gets or sets the name of the place.
        /// </summary>
        /// <value>
        /// The name of the place.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string PlaceName
        {
            get
            {
                return placeName;
            }
            set
            {
                placeName = value;
                base.OnPropertyChanged("PlaceName");
                base.OnPropertyChanged("Address");
            }
        }
        /// <summary>
        /// Gets or sets the post code.
        /// </summary>
        /// <value>
        /// The post code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string PostCode
        {
            get
            {
                return postCode;
            }
            set
            {
                postCode = value;
                base.OnPropertyChanged("PostCode");
                base.OnPropertyChanged("Address");
            }
        }
        /// <summary>
        /// Gets or sets the date of birth.
        /// </summary>
        /// <value>
        /// The date of birth.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public DateTime DateOfBirth
        {
            get
            {
                return dateOfBirth;
            }
            set
            {
                dateOfBirth = value;
                base.OnPropertyChanged("DateOfBirth");
                base.OnPropertyChanged("Age");
                base.OnPropertyChanged("CurrentListName");
            }
        }
        /// <summary>
        /// Gets or sets the date of death.
        /// </summary>
        /// <value>
        /// The date of death.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public DateTime? DateOfDeath
        {
            get
            {
                return dateOfDeath;
            }
            set
            {
                dateOfDeath = value;
                base.OnPropertyChanged("dateOfDeath");
            }
        }
        /// <summary>
        /// Gets or sets the language spoken.
        /// </summary>
        /// <value>
        /// The language spoken.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string LanguageSpoken
        {
            get
            {
                return languageSpoken;
            }
            set
            {
                languageSpoken = value;
                base.OnPropertyChanged("LanguageSpoken");
            }
        }
        /// <summary>
        /// Gets or sets the country ID.
        /// </summary>
        /// <value>
        /// The country ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int CountryID
        {
            get
            {
                return countryID;
            }
            set
            {
                countryID = value;
                base.OnPropertyChanged("CountryID");
            }
        }
        /// <summary>
        /// Gets or sets the country code.
        /// </summary>
        /// <value>
        /// The country code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string CountryCode
        {
            get
            {
                return countryCode;
            }
            set
            {
                countryCode = value;
                base.OnPropertyChanged("CountryCode");
            }
        }
        /// <summary>
        /// Gets or sets the country.
        /// </summary>
        /// <value>
        /// The country.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string CountryDescription
        {
            get
            {
                return countryDescription;
            }
            set
            {
                countryDescription = value;
                base.OnPropertyChanged("CountryDescription");
            }
        }
        /// <summary>
        /// Gets or sets the marital status ID.
        /// </summary>
        /// <value>
        /// The marital status ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int MaritalStatusID
        {
            get
            {
                return maritalStatusID;
            }
            set
            {
                maritalStatusID = value;
                base.OnPropertyChanged("MaritalStatusID");
            }
        }
        /// <summary>
        /// Gets or sets the marital status code.
        /// </summary>
        /// <value>
        /// The marital status code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string MaritalStatusCode
        {
            get
            {
                return maritalStatusCode;
            }
            set
            {
                maritalStatusCode = value;
                base.OnPropertyChanged("MaritalStatusCode");
            }
        }
        /// <summary>
        /// Gets or sets the marital status.
        /// </summary>
        /// <value>
        /// The marital status.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string MaritalStatusDescription
        {
            get
            {
                return maritalStatusDescription;
            }
            set
            {
                maritalStatusDescription = value;
                base.OnPropertyChanged("MaritalStatusDescription");
            }
        }
        /// <summary>
        /// Gets or sets the nationality ID.
        /// </summary>
        /// <value>
        /// The nationality ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int NationalityId
        {
            get
            {
                return nationalityId;
            }
            set
            {
                nationalityId = value;
                base.OnPropertyChanged("NationalityID");
            }
        }
        /// <summary>
        /// Gets or sets the nationality code.
        /// </summary>
        /// <value>
        /// The nationality code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string NationalityCode
        {
            get
            {
                return nationalityCode;
            }
            set
            {
                nationalityCode = value;
                base.OnPropertyChanged("NationalityCode");
            }
        }
        /// <summary>
        /// Gets or sets the nationality.
        /// </summary>
        /// <value>
        /// The nationality.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string NationalityDescription
        {
            get
            {
                return nationalityDescription;
            }
            set
            {
                nationalityDescription = value;
                base.OnPropertyChanged("NationalityDescription");
            }
        }

        /// <summary>
        /// Gets or sets the IndigenousStatus code.
        /// </summary>
        /// <value>
        /// The IndigenousStatus code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string IndigenousStatusCode
        {
            get
            {
                return IndigenousStatusCode;
            }
            set
            {
                IndigenousStatusCode = value;
                base.OnPropertyChanged("IndigenousStatusCode");
            }
        }
        /// <summary>
        /// Gets or sets the IndigenousStatus.
        /// </summary>
        /// <value>
        /// The IndigenousStatus.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string IndigenousStatusDescription
        {
            get
            {
                return IndigenousStatusDescription;
            }
            set
            {
                IndigenousStatusDescription = value;
                base.OnPropertyChanged("IndigenousStatusDescription");
            }
        }
        /// <summary>
        /// Gets or sets the religion ID.
        /// </summary>
        /// <value>
        /// The religion ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int ReligionID
        {
            get
            {
                return religionID;
            }
            set
            {
                religionID = value;
                base.OnPropertyChanged("ReligionID");
            }
        }
        /// <summary>
        /// Gets or sets the religion code.
        /// </summary>
        /// <value>
        /// The religion code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string ReligionCode
        {
            get
            {
                return religionCode;
            }
            set
            {
                religionCode = value;
                base.OnPropertyChanged("ReligionCode");
            }
        }
        /// <summary>
        /// Gets or sets the religion.
        /// </summary>
        /// <value>
        /// The religion.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string ReligionDescription 
        {
            get
            {
                return religionDescription;
            }
            set
            {
                religionDescription = value;
                base.OnPropertyChanged("ReligionDescription");
            }
        }

        /// <summary>
        /// Gets or sets the current sex ID.
        /// </summary>
        /// <value>
        /// The current sex ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int CurrentSexID
        {
            get
            {
                return currentSexID;
            }
            set
            {
                currentSexID = value;
                base.OnPropertyChanged("CurrentSexID");
            }
        }
        /// <summary>
        /// Gets or sets the registered sex ID.
        /// </summary>
        /// <value>
        /// The registered sex ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int RegisteredSexID
        {
            get
            {
                return registeredSexID;
            }
            set
            {
                registeredSexID = value;
                base.OnPropertyChanged("RegisteredSexID");
            }
        }

        /// <summary>
        /// Gets or sets the current sex code.
        /// </summary>
        /// <value>
        /// The current sex code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string CurrentSexCode
        {
            get
            {
                return currentSexCode;
            }
            set
            {
                currentSexCode = value;
                base.OnPropertyChanged("CurrentSexCode");
            }
        }
        /// <summary>
        /// Gets or sets the registered sex code.
        /// </summary>
        /// <value>
        /// The registered sex code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string RegisteredSexCode
        {
            get
            {
                if (string.IsNullOrEmpty(registeredSexCode))
                {
                    registeredSexCode = currentSexCode;
                }
                return registeredSexCode;
            }
            set
            {
                registeredSexCode = value;
                base.OnPropertyChanged("RegisteredSexCode");
            }
        }

        /// <summary>
        /// Gets or sets the current sex description.
        /// </summary>
        /// <value>
        /// The current sex description.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string CurrentSexDescription
        {
            get
            {
                return currentSexDescription;
            }
            set
            {
                currentSexDescription = value;
                base.OnPropertyChanged("CurrentSexDescription");
            }
        }
        /// <summary>
        /// Gets or sets the registered sex description.
        /// </summary>
        /// <value>
        /// The registered sex description.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string RegisteredSexDescription
        {
            get
            {
                return registeredSexDescription;
            }
            set
            {
                registeredSexDescription = value;
                base.OnPropertyChanged("RegisteredSexDescription");
            }
        }

        /// <summary>
        /// Gets or sets the registered title ID.
        /// </summary>
        /// <value>
        /// The registered title ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int RegisteredTitleID
        {
            get
            {
                return registeredTitleID;
            }
            set
            {
                registeredTitleID = value;
                base.OnPropertyChanged("RegisteredTitleID");
                base.OnPropertyChanged("FullRegisteredName");
            }
        }
        /// <summary>
        /// Gets or sets the registered title code.
        /// </summary>
        /// <value>
        /// The registered title code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string RegisteredTitleCode
        {
            get
            {
                return registeredTitleCode;
            }
            set
            {
                registeredTitleCode = value;
                base.OnPropertyChanged("RegisteredTitleCode");
            }
        }
        /// <summary>
        /// Gets or sets the registered title description.
        /// </summary>
        /// <value>
        /// The registered title description.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string RegisteredTitleDescription
        {
            get
            {
                return registeredTitleDescription;
            }
            set
            {
                registeredTitleDescription = value;
                base.OnPropertyChanged("RegisteredTitleDescription");
            }
        }
        /// <summary>
        /// Gets or sets the current title ID.
        /// </summary>
        /// <value>
        /// The current title ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int CurrentTitleID
        {
            get
            {
                return currentTitleID;
            }
            set
            {
                currentTitleID = value;
                base.OnPropertyChanged("CurrentTitleID");
                base.OnPropertyChanged("FullCurrentName");
            }
        }

        /// <summary>
        /// Gets or sets the current title code.
        /// </summary>
        /// <value>
        /// The current title code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string CurrentTitleCode
        {
            get
            {
                return currentTitleCode;
            }
            set
            {
                currentTitleCode = value;
                base.OnPropertyChanged("CurrentTitleCode");
            }
        }
        /// <summary>
        /// Gets or sets the current title description.
        /// </summary>
        /// <value>
        /// The current title description.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string CurrentTitleDescription
        {
            get
            {
                return currentTitleDescription;
            }
            set
            {
                currentTitleDescription = value;
                base.OnPropertyChanged("CurrentTitleDescription");
            }
        }


        /// <summary>
        /// Gets or sets the registered suffix ID.
        /// </summary>
        /// <value>
        /// The registered suffix ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int RegisteredSuffixID
        {
            get
            {
                return registeredSuffixID;
            }
            set
            {
                registeredSuffixID = value;
                base.OnPropertyChanged("RegisteredSuffixID");
                base.OnPropertyChanged("FullRegisteredName");
            }
        }
        /// <summary>
        /// Gets or sets the registered suffix code.
        /// </summary>
        /// <value>
        /// The registered suffix code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string RegisteredSuffixCode 
        {
            get
            {
                return registeredSuffixCode;
            }
            set
            {
                registeredSuffixCode = value;
                base.OnPropertyChanged("RegisteredSuffixCode");
            }
        }

        /// <summary>
        /// Gets or sets the registered suffix description.
        /// </summary>
        /// <value>
        /// The registered suffix description.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string RegisteredSuffixDescription
        {
            get
            {
                return registeredSuffixDescription;
            }
            set
            {
                registeredSuffixDescription = value;
                base.OnPropertyChanged("RegisteredSuffixDescription");
            }
        }

        /// <summary>
        /// Gets or sets the current suffix ID.
        /// </summary>
        /// <value>
        /// The current suffix ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int CurrentSuffixID
        {
            get
            {
                return currentSuffixID;
            }
            set
            {
                currentSuffixID = value;
                base.OnPropertyChanged("CurrentSuffixID");
                base.OnPropertyChanged("FullCurrentName");
            }
        }

        /// <summary>
        /// Gets or sets the current suffix code.
        /// </summary>
        /// <value>
        /// The current suffix code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string CurrentSuffixCode
        {
            get
            {
                return currentSuffixCode;
            }
            set
            {
                currentSuffixCode = value;
                base.OnPropertyChanged("CurrentSuffixCode");
            }
        }
        /// <summary>
        /// Gets or sets the current suffix description.
        /// </summary>
        /// <value>
        /// The current suffix description.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string CurrentSuffixDescription
        {
            get
            {
                return currentSuffixDescription;
            }
            set
            {
                currentSuffixDescription = value;
                base.OnPropertyChanged("CurrentSuffixDescription");
            }
        }
        /// <summary>
        /// Gets or sets the diagnosis list.
        /// </summary>
        /// <value>
        /// The diagnosis list.
        /// </value>
        public List<PatientDiagnosis> DiagnosisList { get; set; }
        /// <summary>
        /// Gets or sets the state ID.
        /// </summary>
        /// <value>
        /// The state ID.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int StateID 
        {
            get
            {
                return stateID;
            }
            set
            {
                stateID = value;
                base.OnPropertyChanged("StateID");
                base.OnPropertyChanged("Address");
            }
        }
        /// <summary>
        /// Gets or sets the state code.
        /// </summary>
        /// <value>
        /// The state code.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string StateCode
        {
            get
            {
                return stateCode;
            }
            set
            {
                stateCode = value;
                base.OnPropertyChanged("StateCode");
            }
        }
        /// <summary>
        /// Gets or sets the state description.
        /// </summary>
        /// <value>
        /// The state description.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string StateDescription
        {
            get
            {
                return stateDescription;
            }
            set
            {
                stateDescription = value;
                base.OnPropertyChanged("StateDescription");
            }
        }
        /// <summary>
        /// Gets or sets the home phone number.
        /// </summary>
        /// <value>
        /// The home phone number.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string HomePhoneNumber
        {
            get
            {
                return homePhoneNumber;
            }
            set
            {
                homePhoneNumber = value;
                base.OnPropertyChanged("HomePhoneNumber");
            }
        }
        /// <summary>
        /// Gets or sets the work phone number.
        /// </summary>
        /// <value>
        /// The work phone number.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string WorkPhoneNumber
        {
            get
            {
                return workPhoneNumber;
            }
            set
            {
                workPhoneNumber = value;
                base.OnPropertyChanged("WorkPhoneNumber");
            }
        }
        /// <summary>
        /// Gets or sets the mobile.
        /// </summary>
        /// <value>
        /// The mobile.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string Mobile
        {
            get
            {
                return mobile;
            }
            set
            {
                mobile = value;
                base.OnPropertyChanged("Mobile");
            }
        }
        /// <summary>
        /// Gets or sets the email.
        /// </summary>
        /// <value>
        /// The email.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public string Email
        {
            get
            {
                return email;
            }
            set
            {
                email = value;
                base.OnPropertyChanged("Email");
            }
        }
        /// <summary>
        /// Gets or sets the DRG list.
        /// </summary>
        /// <value>
        /// The DRG list.
        /// </value>
        public List<Drg> DrgList { get; set; }
        /// <summary>
        /// Gets or sets the episode list.
        /// </summary>
        /// <value>
        /// The episode list.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public List<Episode> EpisodeList { get; set; }
        /// <summary>
        /// Gets or sets the pathology list.
        /// </summary>
        /// <value>
        /// The pathology list.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public List<Pathology> PathologyList { get; set; }
        /// <summary>
        /// Gets or sets the procedure list.
        /// </summary>
        /// <value>
        /// The procedure list.
        /// </value>
        public List<Procedure> ProcedureList { get; set; }
        /// <summary>
        /// Gets or sets the radiology list.
        /// </summary>
        /// <value>
        /// The radiology list.
        /// </value>
        public List<Radiology> RadiologyList { get; set; }
        /// <summary>
        /// Gets or sets the new diagnosis count.
        /// </summary>
        /// <value>
        /// The new diagnosis count.
        /// </value>
        public int NewDiagnosisCount
        {
            get
            {
                return newDiagnosisCount;
            }
            set
            {
                newDiagnosisCount = value;
                base.OnPropertyChanged("NewDiagnosisCount");
                SetNumberOutstandingIndicators();
            }
        }
        /// <summary>
        /// Gets or sets the new DRG count.
        /// </summary>
        /// <value>
        /// The new DRG count.
        /// </value>
        public int NewDrgCount
        {
            get
            {
                return newDrgCount;
            }
            set
            {
                newDrgCount = value;
                base.OnPropertyChanged("NewDrgCount");
                SetNumberOutstandingIndicators();
            }
        }
        /// <summary>
        /// Gets or sets the new episode count.
        /// </summary>
        /// <value>
        /// The new episode count.
        /// </value>
        public int NewEpisodeCount
        {
            get
            {
                return newEpisodeCount;
            }
            set
            {
                newEpisodeCount = value;
                base.OnPropertyChanged("NewEpisodeCount");
                SetNumberOutstandingIndicators();
            }
        }
        /// <summary>
        /// Gets or sets the new pathology count.
        /// </summary>
        /// <value>
        /// The new pathology count.
        /// </value>
        public int NewPathologyCount
        {
            get
            {
                return newPathologyCount;
            }
            set
            {
                newPathologyCount = value;
                base.OnPropertyChanged("NewPathologyCount");
                SetNumberOutstandingIndicators();
            }
        }
        /// <summary>
        /// Gets or sets the new procedure count.
        /// </summary>
        /// <value>
        /// The new procedure count.
        /// </value>
        public int NewProcedureCount
        {
            get
            {
                return newProcedureCount;
            }
            set
            {
                newProcedureCount = value;
                base.OnPropertyChanged("NewProcedureCount");
                SetNumberOutstandingIndicators();
            }
        }
        /// <summary>
        /// Gets or sets the new radiology count.
        /// </summary>
        /// <value>
        /// The new radiology count.
        /// </value>
        public int NewRadiologyCount
        {
            get
            {
                return newRadiologyCount;
            }
            set
            {
                newRadiologyCount = value;
                base.OnPropertyChanged("NewRadiologyCount");
                SetNumberOutstandingIndicators();
            }
        }


        /// <summary>
        /// Gets or sets the number outstanding indicators across all of them.
        /// </summary>
        /// <value>
        /// The number outstanding indicators.
        /// </value>
        public int NumberOutstandingIndicators
        {
            get
            {
                return numberOutstandingIndicators;
            }
            set
            {
                numberOutstandingIndicators = value;
                base.OnPropertyChanged("NumberOutstandingIndicators");
            }
        }

        /// <summary>
        /// Sets the number outstanding indicators - to force binding to work.
        /// </summary>
        private void SetNumberOutstandingIndicators()
        {
            NumberOutstandingIndicators = NewEpisodeCount + NewDiagnosisCount + NewDrgCount + NewPathologyCount + NewProcedureCount + NewRadiologyCount + DischargesOutstanding;
        }
        /// <summary>
        /// Gets or sets the last viewed.
        /// </summary>
        /// <value>
        /// The last viewed.
        /// </value>
        public DateTime LastViewed { get; set; }
        /// <summary>
        /// Gets or sets the last updated.
        /// </summary>
        /// <value>
        /// The last updated.
        /// </value>
        public DateTime? LastUpdated { get; set; }

        /// <summary>
        /// Gets the hospital name of this patient.
        /// </summary>
        /// <value>
        /// The name of the hospital.
        /// </value>
        public string HospitalName
        {
            get
            {
                return string.Format("{0} {1} {2} {3} {4}", this.PatientID, this.CurrentFirstNames.ToUpper(), this.CurrentLastName.ToUpper(), this.CurrentSexDescription, this.Age);
            }
        }

        /// <summary>
        /// Gets the description FullCurrentName and Address.
        /// </summary>
        public string Description
        {
            get
            {
                return string.Format("{0}{1}{2}", this.HospitalName, Environment.NewLine, this.Address);
            }
        }

        /// <summary>
        /// Gets the age for display purposes
        /// </summary>
        public string Age
        {
            get
            {
                DateTime now = DateTime.Now;
                TimeSpan age = now.Subtract(this.DateOfBirth);
                string result = string.Empty;
                if (age.Days <= 2)
                {
                    result = string.Format("{0} hours", age.Hours);
                }
                else if (age.Days <= 14)
                {
                    result = string.Format("{0} days", age.Days);
                }
                else if (age.Days < 63)
                {
                    result = string.Format("{0} weeks", (age.Days / 7));
                }
                else if (age.Days < 731)
                {
                    result = string.Format("{0} Months", Months(this.DateOfBirth, now));
                }
                else
                {
                    result = string.Format("{0} Years", Years(this.DateOfBirth, now));
                }
                return result;
            }
        }

        /// <summary>
        /// Gets the address.
        /// </summary>
        public string Address
        {
            get
            {
                string addressLine = string.IsNullOrEmpty(this.AddressLine1) ? string.Empty : this.AddressLine1;
                string placeName = string.IsNullOrEmpty(this.PlaceName) ? string.Empty : this.PlaceName;
                string postCode = string.IsNullOrEmpty(this.PostCode) ? string.Empty : this.PostCode;
                return string.Format("{0} {1} {2}", addressLine.ToUpper(), placeName.ToUpper(), postCode);
            }
        }


        /// <summary>
        /// Gets the name of the current list.
        /// </summary>
        /// <value>
        /// The name of the current list.
        /// </value>
        public string CurrentListName
        {
            get
            {
                return string.Format("{0} {1}{2}{3}{4}{5}", this.Mrn, this.Age, Environment.NewLine, this.CurrentLastName, Environment.NewLine, this.CurrentFirstNames);
            }
        }

        public string RegisteredListName
        {
            get
            {
                return string.Format("{0} {1}{2}{3}{4}{5}", this.Mrn, this.Age, Environment.NewLine, this.RegisteredLastName, Environment.NewLine, this.RegisteredFirstNames);
            }
        }


        /// <summary>
        /// Gets the full name of the current.
        /// </summary>
        /// <value>
        /// The full name of the current.
        /// </value>
        public string FullCurrentName
        {
            get
            {
                return string.Format("{0} {1}", this.CurrentFirstNames, this.CurrentLastName);
            }
        }

        /// <summary>
        /// Gets the full name of the registered.
        /// </summary>
        /// <value>
        /// The full name of the registered.
        /// </value>
        public string FullRegisteredName
        {
            get
            {
                return string.Format("{0} {1}", this.RegisteredFirstNames, this.RegisteredLastName);
            }
        }

        /// <summary>
        /// Gets the latest date by last viewed or last updated.
        /// </summary>
        public DateTime LatestDateViewedUpdated
        {
            get
            {
                if (LastUpdated.HasValue && LastUpdated.Value  > LastViewed)
                {
                    return LastUpdated.Value;
                }
                return LastViewed;
            }
        }


        #region Private Methods
        /// <summary>
        /// Returns the number of Months between two dates
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <returns></returns>
        private static int Months(DateTime startDate, DateTime endDate)
        {
            int yearMonths = 12 * (endDate.Year - startDate.Year);
            int monthsThisYear = endDate.Month - startDate.Month;
            int lessDaysMonths = 0;
            if ((endDate.Day - startDate.Day) < 0)
            {
                lessDaysMonths = 1;
            }

            return yearMonths + monthsThisYear - lessDaysMonths;
        }

        /// <summary>
        /// Calculates the number of years between two dates
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <returns></returns>
        private int Years(DateTime startDate, DateTime endDate)
        {
            int totalNumberOfYears = endDate.Year - startDate.Year;
            int lessMonthYears = 0;
            if ((endDate.Month - startDate.Month) < 0)
            {
                lessMonthYears = -1;
            }
            return totalNumberOfYears + lessMonthYears;
        }
        #endregion
    }
}
