﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using HIPS.Base.Schemas;
using HIPS.Base.Schemas.Enumerators;
using GalaSoft.MvvmLight;
using PUMA.Helpers;

namespace PUMA.PcehrDataStore.Schemas
{
    /// <summary>
    /// This class represents a Menu Item
    /// </summary>
    [Serializable]
    public class PatientTile : BaseSchema
    {
        #region Private Members
        private int hospitalId;
        private int episodeId;
        private int episodeCount;
        private string givenNames;
        private string familyName;
        private string mrn;
        private int patientMasterId;
        private int patientId;
        private int currentSexId;
        private DateTime dateOfBirth;
        private DateTime? dateOfDeath;
        private bool? userToSearchForIhi;
        private string ihi;
        private bool? pcehrAdvertised;
        private bool? pcehrDisclosed;
        private DateTime? admissionDate;
        private DateTime? dischargeDate;
        private string ward;
        private bool? consentWithdrawn;
        #endregion

        /// <summary>
        /// Gets or sets the hospital id.
        /// </summary>
        /// <value>
        /// The hospital id.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public int HospitalId
        {
            get
            {
                return this.hospitalId;
            }
            set
            {
                Set(() => HospitalId, ref this.hospitalId, value);
            }
        }


        /// <summary>
        /// Gets or sets the given names.
        /// </summary>
        /// <value>
        /// The given names.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string GivenNames
        {
            get
            {
                return this.givenNames;
            }
            set
            {
                Set(() => GivenNames, ref this.givenNames, value);
            }
        }
        /// <summary>
        /// Gets or sets the name of the family.
        /// </summary>
        /// <value>
        /// The name of the family.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string FamilyName
        {
            get
            {
                return this.familyName;
            }
            set
            {
                Set(() => FamilyName, ref this.familyName, value);
            }
        }
        /// <summary>
        /// Gets or sets the MRN.
        /// </summary>
        /// <value>
        /// The MRN.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string Mrn
        {
            get
            {
                return this.mrn;
            }
            set
            {
                Set(() => Mrn, ref this.mrn, value);
            }
        }
        /// <summary>
        /// Gets or sets the episode id.
        /// </summary>
        /// <value>
        /// The episode id.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public int EpisodeId
        {
            get
            {
                return this.episodeId;
            }
            set
            {
                Set(() => EpisodeId, ref this.episodeId, value);
            }
        }
        /// <summary>
        /// Gets or sets the episode count.
        /// </summary>
        /// <value>
        /// The episode count.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public int EpisodeCount
        {
            get
            {
                return this.episodeCount;
            }
            set
            {
                Set(() => EpisodeCount, ref this.episodeCount, value);
            }
        }
        /// <summary>
        /// Gets or sets the patient master id.
        /// </summary>
        /// <value>
        /// The patient master id.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public int PatientMasterId
        {
            get
            {
                return this.patientMasterId;
            }
            set
            {
                if (this.patientMasterId != value)
                {
                    this.patientMasterId = value;
                    RaisePropertyChanged(() => PatientMasterId);
                }
            }
        }
        /// <summary>
        /// Gets or sets the patient id.
        /// </summary>
        /// <value>
        /// The patient id.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public int PatientId
        {
            get
            {
                return this.patientId;
            }
            set
            {
                if (this.patientId != value)
                {
                    this.patientId = value;
                    RaisePropertyChanged(() => PatientId);
                }
            }
        }

        /// <summary>
        /// Gets or sets the current sex id.
        /// </summary>
        /// <value>
        /// The current sex id.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public int CurrentSexId
        {
            get
            {
                return this.currentSexId;
            }
            set
            {
                if (this.currentSexId != value)
                {
                    this.currentSexId = value;
                    RaisePropertyChanged(() => CurrentSexId);
                }
            }
        }

        /// <summary>
        /// Gets or sets the date of birth.
        /// </summary>
        /// <value>
        /// The date of birth.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public DateTime DateOfBirth
        {
            get
            {
                return this.dateOfBirth;
            }
            set
            {
                if (this.dateOfBirth != value)
                {
                    this.dateOfBirth = value;
                    RaisePropertyChanged(() => DateOfBirth);
                    RaisePropertyChanged(() => Age);
                }
            }
        }
        /// <summary>
        /// Gets the age as a string in Days, months or years.
        /// </summary>
        /// <value>
        /// The age.
        /// </value>
        public string Age
        {
            get
            {
                return (DateOfDeath.HasValue) ? DateHelper.Age(DateOfBirth, DateOfDeath.Value) : DateHelper.Age(DateOfBirth, DateTime.Now);
            }
        }
        /// <summary>
        /// Gets or sets the date of death.
        /// </summary>
        /// <value>
        /// The date of death.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public DateTime? DateOfDeath
        {
            get
            {
                return this.dateOfDeath;
            }
            set
            {
                if (this.dateOfDeath != value)
                {
                    this.dateOfDeath = value;
                    RaisePropertyChanged(() => DateOfDeath);
                }
            }
        }
        /// <summary>
        /// Gets or sets the user to search for ihi.
        /// </summary>
        /// <value>
        /// The user to search for ihi.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public bool? UserToSearchForIhi
        {
            get
            {
                return this.userToSearchForIhi;
            }
            set
            {
                if (this.userToSearchForIhi != value)
                {
                    this.userToSearchForIhi = value;
                    RaisePropertyChanged(() => UserToSearchForIhi);
                }
            }
        }
        /// <summary>
        /// Gets or sets the ihi.
        /// </summary>
        /// <value>
        /// The ihi.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string Ihi
        {
            get
            {
                return this.ihi;
            }
            set
            {
                if (this.ihi != value)
                {
                    this.ihi = value;
                    RaisePropertyChanged(() => Ihi);
                }
            }
        }
        /// <summary>
        /// Gets or sets the pcehr advertised.
        /// </summary>
        /// <value>
        /// The pcehr advertised.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public bool? PcehrAdvertised
        {
            get
            {
                return this.pcehrAdvertised;
            }
            set
            {
                if (this.pcehrAdvertised != value)
                {
                    this.pcehrAdvertised = value;
                    RaisePropertyChanged(() => PcehrAdvertised);
                }
            }
        }
        /// <summary>
        /// Gets or sets the the flag that indicates if the PCEHR has been disclosed.
        /// </summary>
        /// <value>
        /// The pcehr disclosed.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public bool? PcehrDisclosed
        {
            get
            {
                if (!this.pcehrDisclosed.HasValue)
                {
                    this.pcehrDisclosed = false;
                }
                return this.pcehrDisclosed;
            }
            set
            {
                if (this.pcehrDisclosed != value)
                {
                    this.pcehrDisclosed = value;
                    RaisePropertyChanged(() => PcehrDisclosed);
                }
            }
        }

        /// <summary>
        /// Gets a value indicating whether this patient has a PCEHR (Advertised or Disclosed)
        /// </summary>
        /// <value>
        ///   <c>true</c> if [pcehr exists]; otherwise, <c>false</c>.
        /// </value>
        public bool PcehrExists
        {
            get
            {
                if (PcehrAdvertised.HasValue && PcehrAdvertised == true)
                {
                    return true;
                }
                if (PcehrDisclosed.HasValue && PcehrDisclosed == true)
                {
                    return true;
                }
                return false;
            }
        }
        /// <summary>
        /// Gets or sets the admission date.
        /// </summary>
        /// <value>
        /// The admission date.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public DateTime? AdmissionDate
        {
            get
            {
                return this.admissionDate;
            }
            set
            {
                if (this.admissionDate != value)
                {
                    this.admissionDate = value;
                    RaisePropertyChanged(() => AdmissionDate);
                }
            }
        }
        /// <summary>
        /// Gets or sets the discharge date.
        /// </summary>
        /// <value>
        /// The discharge date.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public DateTime? DischargeDate
        {
            get
            {
                return this.dischargeDate;
            }
            set
            {
                if (this.dischargeDate != value)
                {
                    this.dischargeDate = value;
                    RaisePropertyChanged(() => DischargeDate);
                }
            }
        }
        /// <summary>
        /// Gets or sets the ward.
        /// </summary>
        /// <value>
        /// The ward.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public string Ward
        {
            get
            {
                return this.ward;
            }
            set
            {
                if (this.ward != value)
                {
                    this.ward = value;
                    RaisePropertyChanged(() => Ward);
                }
            }
        }
        /// <summary>
        /// Gets or sets the consent withdrawn flag.
        /// </summary>
        /// <value>
        /// The consent withdrawn.
        /// </value>
        [DataBaseInfoAttributes(DatabaseColumnType.ReadOnlyData)]
        public bool? ConsentWithdrawn
        {
            get
            {
                if (!this.consentWithdrawn.HasValue)
                {
                    return false;
                }
                return this.consentWithdrawn;
            }
            set
            {
                if (this.consentWithdrawn != value)
                {
                    this.consentWithdrawn = value;
                    RaisePropertyChanged(() => ConsentWithdrawn);
                }
            }
        }
        /// <summary>
        /// Gets the full name.
        /// </summary>
        /// <value>
        /// The full name.
        /// </value>
        public string FullName
        {
            get
            {
                return string.Format("{0}, {1} {3} ({2})",FamilyName, GivenNames, Mrn, Age);
            }
        }

        /// <summary>
        /// Gets the full name and ward.
        /// </summary>
        /// <value>
        /// The full name and ward.
        /// </value>
        public string FullNameAndWard
        {
            get
            {
                if (string.IsNullOrEmpty(Ward))
                {
                    return FullName;
                }
                else
                {
                    return string.Format("{0}, {1} {4} ({2}) In Ward {3}", FamilyName, GivenNames, Mrn, Ward, Age);
                }
            }
        }
    }
    
}
