﻿using AutoMapper;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;

namespace HIPS.AppServer.HIPSServiceHost.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for HipsResponse.
    /// Every type of business exception should be mapped to a HipsResponse.
    /// This allows newer business logic to be exposed as a V1 service.
    /// </summary>
    internal class HipsResponseProfile : Profile
    {
        #region Properties

        /// <summary>
        /// Gets the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            /*
            Common --> Service
            */

            // System.Exception
            this.CreateMap<System.Exception, HipsResponse>()
                .ForMember(dest => dest.Status, opt => opt.UseValue(HipsResponseIndicator.SystemError))
                .ForMember(dest => dest.ResponseCode, opt => opt.MapFrom(src => src.GetType().Name))
                .ForMember(dest => dest.ResponseCodeDetails, opt => opt.MapFrom(src => src.InnerException.Message))
                .ForMember(dest => dest.HipsErrorMessage, opt => opt.MapFrom(src => src.Message));

            // ItemNotFoundException
            this.CreateMap<ItemNotFoundException, HipsResponse>()
                .ForMember(dest => dest.Status, opt => opt.MapFrom(
                    src => src.Type == ItemNotFoundException.ItemType.Hospital ? HipsResponseIndicator.InvalidHospital
                    : src.Type == ItemNotFoundException.ItemType.Patient ? HipsResponseIndicator.InvalidPatient
                    : src.Type == ItemNotFoundException.ItemType.Episode ? HipsResponseIndicator.InvalidEpisode
                    : src.Type == ItemNotFoundException.ItemType.Document ? HipsResponseIndicator.InvalidDocument
                    : src.Type == ItemNotFoundException.ItemType.Address ? HipsResponseIndicator.InvalidPatient
                    : HipsResponseIndicator.SystemError))
                .ForMember(dest => dest.ResponseCode, opt => opt.MapFrom(src => src.Qualifier))
                .ForMember(dest => dest.ResponseCodeDetails, opt => opt.MapFrom(src => src.Identifier))
                .ForMember(dest => dest.HipsErrorMessage, opt => opt.MapFrom(src => src.Message));

            // InvalidUserException
            this.CreateMap<InvalidUserException, HipsResponse>()
                .ForMember(dest => dest.Status, opt => opt.UseValue(HipsResponseIndicator.InvalidUser))
                .ForMember(dest => dest.ResponseCode, opt => opt.MapFrom(src => src.GetType().Name))
                .ForMember(dest => dest.ResponseCodeDetails, opt => opt.MapFrom(src => src.InnerException.Message))
                .ForMember(dest => dest.HipsErrorMessage, opt => opt.MapFrom(src => src.Message));

            // HiServiceException
            this.CreateMap<HiServiceException, HipsResponse>()
                .ForMember(dest => dest.Status, opt => opt.UseValue(HipsResponseIndicator.HiServiceError))
                .ForMember(dest => dest.ResponseCode, opt => opt.MapFrom(src => src.ResponseCode))
                .ForMember(dest => dest.ResponseCodeDescription, opt => opt.MapFrom(src => src.ResponseCodeDescription))
                .ForMember(dest => dest.ResponseCodeDetails, opt => opt.MapFrom(src => src.ResponseCodeDetails))
                .ForMember(dest => dest.HipsErrorMessage, opt => opt.MapFrom(src => src.Message));

            // PcehrServiceException
            this.CreateMap<PcehrServiceException, HipsResponse>()
                .ForMember(dest => dest.Status, opt => opt.UseValue(HipsResponseIndicator.HiServiceError))
                .ForMember(dest => dest.ResponseCode, opt => opt.MapFrom(src => src.ResponseCode))
                .ForMember(dest => dest.ResponseCodeDescription, opt => opt.MapFrom(src => src.ResponseCodeDescription))
                .ForMember(dest => dest.ResponseCodeDetails, opt => opt.MapFrom(src => src.ResponseCodeDetails))
                .ForMember(dest => dest.HipsErrorMessage, opt => opt.MapFrom(src => src.Message));

            // HipsResponseException
            this.CreateMap<HipsResponseException, HipsResponse>()
                .ConvertUsing(src => src.Detail);
        }

        #endregion Methods
    }
}