﻿using System;
using System.Collections.Generic;
using System.Linq;

using AutoMapper;
using ModelHpii = HIPS.HpiiSchemas;
using NehtaHpii = nehta.mcaR50.ProviderSearchForProviderIndividual;
using ServiceCommon = HIPS.ServiceContracts.Common;
using ServiceCommonDto = HIPS.ServiceContracts.Common.DTO;
using ServiceHpiiDto = HIPS.ServiceContracts.Hpii.DTO;
using ServiceHpiiMessage = HIPS.ServiceContracts.Hpii.Message;

namespace HIPS.AppServer.ServiceHost.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the HPI-I search entities.
    /// </summary>
    internal class HpiiProfile : Profile
    {
        #region Properties

        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        #endregion

        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // DTO --> Model

            this.CreateMap<ServiceHpiiMessage.HpiiBatchRetrieveRequest, ModelHpii.HpiiBatchAsyncRetrieveRequest>()
                .ForMember(dest => dest.BatchIdentifier, opt => opt.MapFrom(src => src.BatchIdentifier));

            this.CreateMap<ServiceHpiiMessage.HpiiBatchSubmitRequest, ModelHpii.HpiiBatchAsyncSubmitRequest>()
                .ForMember(dest => dest.HpiiDemographicQueries, opt => opt.MapFrom(src => src.DemographicQueries))
                .ForMember(dest => dest.HpiiIdentifierQueries, opt => opt.MapFrom(src => src.IdentifierQueries));

            this.CreateMap<ServiceHpiiDto.DemographicQuery, ModelHpii.HpiiDemographicQuery>()
                .ForMember(dest => dest.AustralianAddress, opt => opt.MapFrom(src => src.AustralianAddress))
                .ForMember(dest => dest.InternationalAddress, opt => opt.MapFrom(src => src.InternationalAddress))
                .ForMember(dest => dest.DateofBirth, opt => opt.MapFrom(src => src.DateOfBirth))
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenName, opt => opt.MapFrom(src => src.GivenNames))
                .ForMember(dest => dest.OnlyNameIndicator, opt => opt.MapFrom(src => src.OnlyNameIndicator))
                .ForMember(
                    dest => dest.Sex,
                    opt => opt.MapFrom(src =>
                        src.Sex == null ? (NehtaHpii.SexType?)null :
                        src.Sex == ServiceCommon.Sex.None ? (NehtaHpii.SexType?)null :
                        src.Sex == ServiceCommon.Sex.Female ? NehtaHpii.SexType.F :
                        src.Sex == ServiceCommon.Sex.IntersexOrIndeterminate ? NehtaHpii.SexType.I :
                        src.Sex == ServiceCommon.Sex.Male ? NehtaHpii.SexType.M :
                        NehtaHpii.SexType.N));

            this.CreateMap<ServiceCommonDto.StructuredAustralianAddress, ModelHpii.AustralianAddress>()
                .ForMember(dest => dest.LevelNumber, opt => opt.MapFrom(src => src.LevelNumber))
                .ForMember(dest => dest.LevelType, opt => opt.MapFrom(src => src.LevelType))
                .ForMember(dest => dest.LotNumber, opt => opt.MapFrom(src => src.LotNumber))
                .ForMember(dest => dest.PostalDeliveryNumber, opt => opt.MapFrom(src => src.PostalDeliveryNumber))
                .ForMember(dest => dest.PostalDeliveryType, opt => opt.MapFrom(src => src.PostalDeliveryType))
                .ForMember(dest => dest.PostCode, opt => opt.MapFrom(src => src.Postcode))
                .ForMember(dest => dest.SiteName, opt => opt.MapFrom(src => src.SiteName))
                .ForMember(dest => dest.State, opt => opt.MapFrom(src => src.State == ServiceCommon.AustralianState.None ? (ServiceCommon.AustralianState?)null : src.State))
                .ForMember(dest => dest.StreetName, opt => opt.MapFrom(src => src.StreetName))
                .ForMember(dest => dest.StreetNumber, opt => opt.MapFrom(src => src.StreetNumber))
                .ForMember(dest => dest.StreetSuffixType, opt => opt.MapFrom(src => src.StreetSuffixType))
                .ForMember(dest => dest.StreetType, opt => opt.MapFrom(src => src.StreetType))
                .ForMember(dest => dest.Suburb, opt => opt.MapFrom(src => src.PlaceName))
                .ForMember(dest => dest.UnitNumber, opt => opt.MapFrom(src => src.UnitNumber))
                .ForMember(dest => dest.UnitType, opt => opt.MapFrom(src => src.UnitType));

            this.CreateMap<ServiceCommonDto.InternationalAddress, ModelHpii.InternationalAddress>()
                .ForMember(dest => dest.InternationalPostcode, opt => opt.MapFrom(src => src.Postcode))
                .ForMember(dest => dest.InternationalStateProvince, opt => opt.MapFrom(src => src.StateProvince))
                .ForMember(dest => dest.InternationalAddressLine, opt => opt.MapFrom(src => src.AddressLine))
                .ForMember(
                    dest => dest.Country,
                    opt => opt.ResolveUsing((ServiceCommonDto.InternationalAddress src) =>
                    {
                        if (string.IsNullOrEmpty(src.CountryCode))
                        {
                            // Lookup HIPS Country reference data by country name, ignoring case and surrounding whitespace.
                            var lists = CommonBusinessLogic.Singleton.ListSingleton.Instance;
                            var country = lists.AllCountries.FirstOrDefault(c => string.Equals(
                                c.Description.Trim(),
                                src.CountryName.Trim(),
                                StringComparison.CurrentCultureIgnoreCase));
                            if (country != null)
                            {
                                src.CountryCode = country.Code;
                            }
                            else
                            {
                                throw new Exception(string.Format(
                                    "Country name {0} not found in HIPS Country reference data.",
                                    src.CountryName));
                            }
                        }

                        // The NEHTA enum CountryType has the format "ItemNNNN" where NNNN is the SACC code.
                        // See TECH.SIS.HI.02 - Common Field Processing Reference Document.
                        // See ABS 1269.0 - Standard Australian Classification of Countries (SACC), 2011.
                        NehtaHpii.CountryType result;
                        if (Enum.TryParse<NehtaHpii.CountryType>("Item" + src.CountryCode, out result))
                        {
                            return result;
                        }
                        else
                        {
                            throw new Exception(string.Format(
                                "Country code {0} not found in CountryType enumeration.",
                                src.CountryCode));
                        }
                    }));

            this.CreateMap<ServiceHpiiDto.IdentifierQuery, ModelHpii.HpiiIdentifierQuery>()
                .ForMember(dest => dest.DateofBirth, opt => opt.MapFrom(src => src.DateOfBirth))
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenName, opt => opt.MapFrom(src => src.GivenNames))
                .ForMember(dest => dest.HpiiNumber, opt => opt.MapFrom(src => src.HpiiNumber))
                .ForMember(dest => dest.OnlyNameIndicator, opt => opt.MapFrom(src => src.OnlyNameIndicator))
                .ForMember(dest => dest.PostCode, opt => opt.MapFrom(src => src.AustralianPostcode))
                .ForMember(dest => dest.RegistrationId, opt => opt.MapFrom(src => src.RegistrationId))
                .ForMember(dest => dest.State, opt => opt.MapFrom(src => src.AustralianState == ServiceCommon.AustralianState.None ? null : (ServiceCommon.AustralianState?)src.AustralianState))
                .ForMember(
                    dest => dest.Sex,
                    opt => opt.MapFrom(src =>
                        src.Sex == null ? (NehtaHpii.SexType?)null :
                        src.Sex == ServiceCommon.Sex.None ? (NehtaHpii.SexType?)null :
                        src.Sex == ServiceCommon.Sex.Female ? NehtaHpii.SexType.F :
                        src.Sex == ServiceCommon.Sex.IntersexOrIndeterminate ? NehtaHpii.SexType.I :
                        src.Sex == ServiceCommon.Sex.Male ? NehtaHpii.SexType.M :
                        NehtaHpii.SexType.N));

            // Data --> DTO

            this.CreateMap<ModelHpii.HpiiBatchAsyncSubmitResponse, ServiceHpiiDto.BatchSubmitResult>()
                .ForMember(dest => dest.BatchIdentifier, opt => opt.MapFrom(src => src.BatchIdentifier));

            this.CreateMap<ModelHpii.HpiiBatchAsyncRetrieveResponse, ServiceHpiiDto.BatchRetrieveResult>()
                .ForMember(dest => dest.BatchIdentifier, opt => opt.MapFrom(src => src.BatchIdentifier))
                .ForMember(dest => dest.Results, opt => opt.MapFrom(src => src.HpiiBatchQueryResponses));

            this.CreateMap<ModelHpii.HpiiBatchQueryResponse, ServiceHpiiDto.BatchSearchResult>()
                .ForMember(dest => dest.RequestIdentifier, opt => opt.MapFrom(src => src.RequestIdentifier))
                .ForMember(dest => dest.AustralianAddress, opt => opt.MapFrom(src => src.HpiiQueryResponse.AustralianAddress))
                .ForMember(dest => dest.AustralianPostcode, opt => opt.MapFrom(src => src.HpiiQueryResponse.PostCode))
                .ForMember(dest => dest.AustralianState, opt => opt.MapFrom(src => src.HpiiQueryResponse.State))
                .ForMember(dest => dest.DateOfBirth, opt => opt.MapFrom(src => src.HpiiQueryResponse.DateofBirth))
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.HpiiQueryResponse.FamilyName))
                .ForMember(dest => dest.GivenNames, opt => opt.MapFrom(src => src.HpiiQueryResponse.GivenName))
                .ForMember(dest => dest.HpiiNumber, opt => opt.MapFrom(src => src.HpiiQueryResponse.HpiiNumber))
                .ForMember(dest => dest.HpiiStatus, opt => opt.MapFrom(src => src.HpiiQueryResponse.HpiiStatus))
                .ForMember(dest => dest.RegistrationId, opt => opt.MapFrom(src => src.HpiiQueryResponse.RegistrationId))
                .ForMember(dest => dest.InternationalAddress, opt => opt.MapFrom(src => src.HpiiQueryResponse.InternationalAddress))
                .ForMember(dest => dest.OnlyNameIndicator, opt => opt.MapFrom(src => src.HpiiQueryResponse.OnlyNameIndicator))
                .ForMember(dest => dest.RegistrationId, opt => opt.MapFrom(src => src.HpiiQueryResponse.RegistrationId))
                .ForMember(
                    dest => dest.Sex,
                    opt => opt.MapFrom(src =>
                        src.HpiiQueryResponse.Sex == null ? (ServiceCommon.Sex?)null :
                        src.HpiiQueryResponse.Sex == NehtaHpii.SexType.F ? ServiceCommon.Sex.Female :
                        src.HpiiQueryResponse.Sex == NehtaHpii.SexType.I ? ServiceCommon.Sex.IntersexOrIndeterminate :
                        src.HpiiQueryResponse.Sex == NehtaHpii.SexType.M ? ServiceCommon.Sex.Male :
                        ServiceCommon.Sex.NotStatedOrInadequatelyDescribed));

            this.CreateMap<ModelHpii.HpiiQueryResponse, ServiceHpiiDto.SearchResult>()
                .ForMember(dest => dest.AustralianAddress, opt => opt.MapFrom(src => src.AustralianAddress))
                .ForMember(dest => dest.AustralianPostcode, opt => opt.MapFrom(src => src.PostCode))
                .ForMember(dest => dest.AustralianState, opt => opt.MapFrom(src => src.State))
                .ForMember(dest => dest.DateOfBirth, opt => opt.MapFrom(src => src.DateofBirth))
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenNames, opt => opt.MapFrom(src => src.GivenName))
                .ForMember(dest => dest.HpiiNumber, opt => opt.MapFrom(src => src.HpiiNumber))
                .ForMember(dest => dest.HpiiStatus, opt => opt.MapFrom(src => src.HpiiStatus))
                .ForMember(dest => dest.RegistrationId, opt => opt.MapFrom(src => src.RegistrationId))
                .ForMember(dest => dest.InternationalAddress, opt => opt.MapFrom(src => src.InternationalAddress))
                .ForMember(dest => dest.OnlyNameIndicator, opt => opt.MapFrom(src => src.OnlyNameIndicator))
                .ForMember(dest => dest.RegistrationId, opt => opt.MapFrom(src => src.RegistrationId))
                .ForMember(
                    dest => dest.Sex,
                    opt => opt.MapFrom(src =>
                        src.Sex == null ? (ServiceCommon.Sex?)null :
                        src.Sex == NehtaHpii.SexType.F ? ServiceCommon.Sex.Female :
                        src.Sex == NehtaHpii.SexType.I ? ServiceCommon.Sex.IntersexOrIndeterminate :
                        src.Sex == NehtaHpii.SexType.M ? ServiceCommon.Sex.Male :
                        ServiceCommon.Sex.NotStatedOrInadequatelyDescribed));

            this.CreateMap<ModelHpii.AustralianAddress, ServiceCommonDto.StructuredAustralianAddress>()
                .ForMember(dest => dest.LevelNumber, opt => opt.MapFrom(src => src.LevelNumber))
                .ForMember(dest => dest.LevelType, opt => opt.MapFrom(src => src.LevelType))
                .ForMember(dest => dest.LotNumber, opt => opt.MapFrom(src => src.LotNumber))
                .ForMember(dest => dest.PostalDeliveryNumber, opt => opt.MapFrom(src => src.PostalDeliveryNumber))
                .ForMember(dest => dest.PostalDeliveryType, opt => opt.MapFrom(src => src.PostalDeliveryType))
                .ForMember(dest => dest.Postcode, opt => opt.MapFrom(src => src.PostCode))
                .ForMember(dest => dest.SiteName, opt => opt.MapFrom(src => src.SiteName))
                .ForMember(dest => dest.State, opt => opt.MapFrom(src => src.State))
                .ForMember(dest => dest.StreetName, opt => opt.MapFrom(src => src.StreetName))
                .ForMember(dest => dest.StreetNumber, opt => opt.MapFrom(src => src.StreetNumber))
                .ForMember(dest => dest.StreetSuffixType, opt => opt.MapFrom(src => src.StreetSuffixType))
                .ForMember(dest => dest.StreetType, opt => opt.MapFrom(src => src.StreetType))
                .ForMember(dest => dest.PlaceName, opt => opt.MapFrom(src => src.Suburb))
                .ForMember(dest => dest.UnitNumber, opt => opt.MapFrom(src => src.UnitNumber))
                .ForMember(dest => dest.UnitType, opt => opt.MapFrom(src => src.UnitType));

            this.CreateMap<ModelHpii.InternationalAddress, ServiceCommonDto.InternationalAddress>()
                .ForMember(dest => dest.Postcode, opt => opt.MapFrom(src => src.InternationalPostcode))
                .ForMember(dest => dest.StateProvince, opt => opt.MapFrom(src => src.InternationalStateProvince))
                .ForMember(dest => dest.AddressLine, opt => opt.MapFrom(src => src.InternationalAddressLine))
                .ForMember(dest => dest.CountryCode, opt => opt.MapFrom(src => src.Country.ToString().Substring(4)))
                .ForMember(
                    dest => dest.CountryName,
                    opt => opt.ResolveUsing((ModelHpii.InternationalAddress src) =>
                    {
                        // The NEHTA enum CountryType has the format "ItemNNNN" where NNNN is the SACC code.
                        // See TECH.SIS.HI.02 - Common Field Processing Reference Document.
                        // See ABS 1269.0 - Standard Australian Classification of Countries (SACC), 2011.
                        if (src.Country == null)
                        {
                            return null;
                        }
                        var code = src.Country.ToString().Substring(4);
                        var lists = CommonBusinessLogic.Singleton.ListSingleton.Instance;
                        var country = lists.AllCountries.FirstOrDefault(c => c.Code == code);
                        if (country != null)
                        {
                            return country.Description;
                        }
                        else
                        {
                            throw new Exception(string.Format(
                                "Country code {1} not found in HIPS Country reference data.",
                                code));
                        }
                    }));

            this.CreateMap<NehtaHpii.ServiceMessageType, ServiceCommonDto.Message>()
                .ForMember(dest => dest.Code, opt => opt.MapFrom(src => src.code))
                .ForMember(dest => dest.Description, opt => opt.MapFrom(src => src.reason))
                .ForMember(dest => dest.Origin, opt => opt.UseValue(CommonSchemas.Constants.Origin.HI_SERVICE))
                .ForMember(
                    dest => dest.Level,
                    opt => opt.MapFrom(src =>
                        src.severity == NehtaHpii.SeverityType.Error ? ServiceCommon.MessageLevel.Error :
                        src.severity == NehtaHpii.SeverityType.Fatal ? ServiceCommon.MessageLevel.Error :
                        src.severity == NehtaHpii.SeverityType.Informational ? ServiceCommon.MessageLevel.Information :
                        src.severity == NehtaHpii.SeverityType.Warning ? ServiceCommon.MessageLevel.Warning :
                        ServiceCommon.MessageLevel.None));
        }

        #endregion Methods
    }
}