﻿using System;
using System.Collections.Generic;
using System.ServiceModel;
using HIPS.AppServer.HIPSServiceHost.Services;
using HIPS.AppServer.ServiceHost.Mapping;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonBusinessLogic.Hpii;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.Exceptions;
using HIPS.HpiiSchemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.ServiceContracts.Common;
using HIPS.ServiceContracts.Common.DTO;
using HIPS.ServiceContracts.Common.Fault;
using HIPS.ServiceContracts.Hpii;
using HIPS.ServiceContracts.Hpii.DTO;
using HIPS.ServiceContracts.Hpii.Message;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements web services for searching for a healthcare provider individual using the HI Service.
    /// </summary>
    public partial class HpiiService : BaseService, IHpiiServiceV2
    {
        /// <summary>
        /// This operation retrieves the results of a batch of HPI-I searches from the HI Service.
        /// </summary>
        /// <param name="request">HpiiBatchRetrieveRequest request parameters.</param>
        /// <returns>Response including the results of the batch.</returns>
        public HpiiBatchRetrieveResponse HpiiBatchRetrieve(HpiiBatchRetrieveRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var businessRequest = ObjectMapper.Map<HpiiBatchAsyncRetrieveRequest>(request);
                var facility = ObjectMapper.Map<CommonSchemas.HospitalIdentifier>(request.Facility);

                // Invoke Business Service
                var logic = new HpiiSearch();
                var businessResult = logic.HpiiBatchRetrieve(businessRequest, user, facility);
                if (businessResult.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(businessResult.HipsResponse);
                }

                // Map Common Model to DTO
                var serviceResult = ObjectMapper.Map<BatchRetrieveResult>(businessResult);
                var serviceMessages = this.BuildMessageList(businessResult.ServiceMessagesType);

                return new HpiiBatchRetrieveResponse { Status = ResponseStatus.OK, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(
                    ResponseStrings.ErrorUnableToRetrieveHpiiQueryBatch,
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_176);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// This operation submits a batch of HPI-I searches to the HI Service.
        /// </summary>
        /// <param name="request">HpiiBatchSubmitRequest request parameters.</param>
        /// <returns>Response including the batch identifier.</returns>
        public HpiiBatchSubmitResponse HpiiBatchSubmit(HpiiBatchSubmitRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var businessRequest = ObjectMapper.Map<HpiiBatchAsyncSubmitRequest>(request);
                var facility = ObjectMapper.Map<CommonSchemas.HospitalIdentifier>(request.Facility);

                // Invoke Business Service
                var logic = new HpiiSearch();
                var businessResult = logic.HpiiBatchSubmit(businessRequest, user, facility);
                if (businessResult.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(businessResult.HipsResponse);
                }

                // Map Common Model to DTO
                var serviceResult = ObjectMapper.Map<BatchSubmitResult>(businessResult);
                var serviceMessages = this.BuildMessageList(businessResult.ServiceMessagesType);

                return new HpiiBatchSubmitResponse { Status = ResponseStatus.OK, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(
                    ResponseStrings.ErrorUnableToSubmitHpiiQueryBatch,
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_177);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// This operation submits an HPI-I demographic search to the HI Service.
        /// </summary>
        /// <param name="request">HpiiDemographicSearchRequest request parameters.</param>
        /// <returns>Response including the search results.</returns>
        public HpiiDemographicSearchResponse HpiiDemographicSearch(HpiiDemographicSearchRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var businessRequest = ObjectMapper.Map<HpiiDemographicQuery>(request.Query);
                var facility = ObjectMapper.Map<CommonSchemas.HospitalIdentifier>(request.Facility);

                // Invoke Business Service
                var logic = new HpiiSearch();
                var businessResult = logic.HpiiDemographicSearch(businessRequest, user, facility);
                if (businessResult.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(businessResult.HipsResponse);
                }

                // Map Common Model to DTO
                var serviceResult = ObjectMapper.Map<SearchResult>(businessResult);
                var serviceMessages = this.BuildMessageList(businessResult.ServiceMessagesType);

                return new HpiiDemographicSearchResponse { Status = ResponseStatus.OK, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(
                    ResponseStrings.ErrorUnableToPerformHpiiDemographicSearch,
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_178);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// This operation submits an HPI-I identifier search to the HI Service.
        /// </summary>
        /// <param name="request">HpiiIdentifierSearchRequest request parameters.</param>
        /// <returns>Response including the search results.</returns>
        public HpiiIdentifierSearchResponse HpiiIdentifierSearch(HpiiIdentifierSearchRequest request)
        {
            this.ValidateRequest(request);
            UserDetails user = null;

            try
            {
                // Map DTO to Common Model
                user = ObjectMapper.Map<UserDetails>(request.User);
                var businessRequest = ObjectMapper.Map<HpiiIdentifierQuery>(request.Query);
                var facility = ObjectMapper.Map<CommonSchemas.HospitalIdentifier>(request.Facility);

                // Invoke Business Service
                var logic = new HpiiSearch();
                var businessResult = logic.HpiiValidation(businessRequest, user, facility);
                if (businessResult.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new HipsResponseException(businessResult.HipsResponse);
                }

                // Map Common Model to DTO
                var serviceResult = ObjectMapper.Map<SearchResult>(businessResult);
                var serviceMessages = this.BuildMessageList(businessResult.ServiceMessagesType);

                return new HpiiIdentifierSearchResponse { Status = ResponseStatus.OK, Data = serviceResult, Messages = serviceMessages };
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(
                    ResponseStrings.ErrorUnableToPerformHpiiValidation,
                    ex,
                    user,
                    LogMessage.HIPS_MESSAGE_179);
                throw ObjectMapper.Map<FaultException<ServiceOperationFault>>(ex);
            }
        }

        /// <summary>
        /// Builds a list of HIPS service messages from the NEHTA ServiceMessagesType object.
        /// </summary>
        /// <param name="nehtaMessages">The list of messages returned from the HI Service.</param>
        /// <returns>The HIPS service message list.</returns>
        private List<Message> BuildMessageList(nehta.mcaR50.ProviderSearchForProviderIndividual.ServiceMessagesType nehtaMessages)
        {
            var messages = new List<Message>();
            if (nehtaMessages != null)
            {
                messages.AddRange(ObjectMapper.Map<List<Message>>(nehtaMessages.serviceMessage));
            }
            return messages;
        }
    }
}