﻿using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.CommonBusinessLogic.Consent
{
    /// <summary>
    /// Business logic to record a patient's disclosure of the existence of
    /// their PCEHR record to a healthcare provider organisation.
    /// </summary>
    public class RecordPcehrDisclosure
    {
        /// <summary>
        /// Record whether the patient has disclosed the existence of a PCEHR.
        /// </summary>
        /// <param name="pcehrDisclosed">True if the patient disclosed the existence of a PCEHR, false if the patient has not disclosed the existence of a PCEHR.</param>
        /// <param name="patientIdentifier">The local, state or national patient identifier</param>
        /// <param name="user">Information to identify the person responsible for this action</param>
        /// <param name="auditInformation">Additional audit information to store</param>
        /// <returns>Indicator of any error that occurred</returns>
        public HipsResponse RecordDisclosure(bool pcehrDisclosed, PatientIdentifierBase patientIdentifier, UserDetails user, byte[] auditInformation)
        {
            PatientAccess patientAccess = new PatientAccess(user);
            Hospital hospital;
            HospitalPatient hospitalPatient;
            PatientMaster patientMaster;
            HipsResponse status = patientAccess.GetHospital(patientIdentifier, out hospital);
            if (status.Status != HipsResponseIndicator.OK)
            {
                return status;
            }

            // When disclosing PCEHR existence, it's OK if the patient has an invalid IHI or an unresolved IHI alert.
            status = patientAccess.GetPatient(patientIdentifier, hospital, out hospitalPatient, out patientMaster);
            if (status.Status != HipsResponseIndicator.OK && status.Status != HipsResponseIndicator.InvalidIhi && status.Status != HipsResponseIndicator.UnresolvedIhiAlert)
            {
                return status;
            }
            if (!User.PopulateAndValidateUser(hospital, user))
            {
                return new HipsResponse(HipsResponseIndicator.InvalidUser);
            }
            DisclosureAudit audit = new DisclosureAudit();
            audit.HealthProviderOrganisationId = hospital.HealthProviderOrganisationId;
            audit.PatientMasterId = patientMaster.PatientMasterId.Value;
            audit.PcehrDisclosed = pcehrDisclosed;
            audit.AuditInformation = auditInformation;
            DisclosureAuditDl auditDataAccess = new DisclosureAuditDl(user);
            HealthProviderOrganisationPatient hpioPatient;
            if (patientAccess.HealthProviderOrganisationPatientDataAccess.Get(hospital.HpiO, patientMaster.PatientMasterId.Value, out hpioPatient))
            {
                hpioPatient.PcehrDisclosed = pcehrDisclosed;
                if (auditDataAccess.Insert(audit) && patientAccess.HealthProviderOrganisationPatientDataAccess.Update(hpioPatient))
                {
                    return new HipsResponse(HipsResponseIndicator.OK);
                }
                else
                {
                    return new HipsResponse(HipsResponseIndicator.SystemError);
                }
            }
            else
            {
                hpioPatient.PatientMasterId = patientMaster.PatientMasterId.Value;
                hpioPatient.HealthProviderOrganisationId = hospital.HealthProviderOrganisationId;
                hpioPatient.PcehrDisclosed = pcehrDisclosed;
                if (auditDataAccess.Insert(audit) && patientAccess.HealthProviderOrganisationPatientDataAccess.Insert(hpioPatient))
                {
                    return new HipsResponse(HipsResponseIndicator.OK);
                }
                else
                {
                    return new HipsResponse(HipsResponseIndicator.SystemError);
                }
            }
        }
    }
}