﻿using System;
using System.Configuration;
using System.IO;
using System.Linq;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Transactions;
using System.Xml;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.HpiiSchemas;
using HIPS.HpiiSchemas.Enumerators;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using nehta.mcaR50.ProviderSearchForProviderIndividual;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.HI;

namespace HIPS.CommonBusinessLogic.Hpii
{
    /// <summary>
    /// Helper business logic for HPI-I search.
    /// </summary>
    public static class HpiiHelper
    {
        /// <summary>
        /// Clones the specified source.
        /// </summary>
        /// <typeparam name="T">The type of the source value.</typeparam>
        /// <param name="source">The source value.</param>
        /// <returns>A clone of the source value.</returns>
        public static T Clone<T>(this T source)
        {
            if (!typeof(T).IsSerializable)
            {
                throw new ArgumentException("The type must be serializable.", "source");
            }

            // Don't serialize a null object, simply return the default for that object
            if (object.ReferenceEquals(source, null))
            {
                return default(T);
            }

            IFormatter formatter = new BinaryFormatter();
            Stream stream = new MemoryStream();
            using (stream)
            {
                formatter.Serialize(stream, source);
                stream.Seek(0, SeekOrigin.Begin);
                return (T)formatter.Deserialize(stream);
            }
        }

        /// <summary>
        /// Extracts the HPI-I from the fully qualified URI.
        /// </summary>
        /// <param name="hpii">The fully qualified URI representing the HPI-I.</param>
        /// <returns>The HPI-I number itself.</returns>
        public static string ExtractHpii(string hpii)
        {
            if (string.IsNullOrEmpty(hpii))
            {
                return string.Empty;
            }
            string result = hpii;
            if (hpii != null && hpii.StartsWith(HIQualifiers.HPIIQualifier))
            {
                result = hpii.Substring(HIQualifiers.HPIIQualifier.Length);
            }

            return result;
        }

        /// <summary>
        /// <para>Gets the QualifiedId for the HPI-O element in the HI Service B2B Header.
        /// </para>
        /// <para>If the certificate belongs to a healthcare provider organisation, the HPI-O
        /// must not be provided in the header.
        /// </para>
        /// <para>If the certificate belongs to a contracted service provider, the HPI-O
        /// of the healthcare provider organisation (that the CSP is acting on behalf
        /// of) must be provided in the header.
        /// </para>
        /// </summary>
        /// <param name="userDetails">The user details.</param>
        /// <param name="hpo">The healthcare provider organisation that will perform the search.</param>
        /// <returns>HPI-O element or null</returns>
        /// <exception cref="System.Exception">The HealthProviderOrganisation table has no entry with the given HPI-O number.</exception>
        public static QualifiedId GetHpioQualifiedId(UserDetails userDetails, HealthProviderOrganisation hpo)
        {
            if (hpo.HiCsp)
            {
                QualifiedId hpio = new QualifiedId()
                {
                    id = hpo.Hpio,
                    qualifier = ConfigurationManager.AppSettings["IhiHpioQualifier"].ToString()
                };
                return hpio;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Gets the configured local healthcare provider organisation using the given HPI-O number or the given hospital identifier.
        /// </summary>
        /// <param name="hpioNumber">The HPI-O number for the healthcare provider organisation.</param>
        /// <param name="facility">The hospital identifier that identifies the facility.</param>
        /// <returns>The healthcare provider organisation record.</returns>
        public static HealthProviderOrganisation GetHealthProviderOrganisation(string hpioNumber, HospitalIdentifier facility)
        {
            ListSingleton lists = ListSingleton.Instance;
            HealthProviderOrganisation hpo;
            if (facility != null)
            {
                Hospital hospital = (from h in lists.AllHospitals
                                     from c in h.Codes
                                     where c.CodeSystemCode == facility.HospitalCodeSystem && c.Code == facility.HospitalCode
                                     select h).FirstOrDefault();
                if (hospital == null)
                {
                    throw new Exception("The Hospital table has no entry with the given code and code system.");
                }
                hpo = lists.AllHealthProviderOrganisations.FirstOrDefault(result => result.HealthProviderOrganisationId == hospital.HealthProviderOrganisationId);
            }
            else
            {
                // Hospital identifier not supplied, use first with matching HPI-O number.
                hpo = lists.AllHealthProviderOrganisations.FirstOrDefault(result => result.Hpio == hpioNumber);
                if (hpo == null)
                {
                    throw new Exception("The HealthProviderOrganisation table has no entry with the given HPI-O number.");
                }
            }
            return hpo;
        }

        /// <summary>
        /// Gets the Medicare certificate for connecting to the HI Service.
        /// </summary>
        /// <param name="hpo">The healthcare provider organisation that will perform the search.</param>
        /// <returns>An X509Certificate2 object containing the certificate that will be used to connect to the HI Service.</returns>
        /// <exception cref="System.Exception">The HealthProviderOrganisation table has no entry with the given HPI-O number.</exception>
        /// <exception cref="System.Exception">The serial number for the HI Service connection certificate has not been configured for this organisation.</exception>
        public static X509Certificate2 GetMedicareCertificate(HealthProviderOrganisation hpo)
        {
            string serialNumber = hpo.HiCertSerial;
            if (string.IsNullOrEmpty(serialNumber))
            {
                throw new Exception("The serial number for the HI Service connection certificate has not been configured for this organisation.");
            }

            // Obtain the certificate for use with TLS
            return X509CertificateUtil.GetCertificate(serialNumber, X509FindType.FindBySerialNumber, StoreName.My, StoreLocation.LocalMachine, false);
        }

        /// <summary>
        /// Gets the product type object for the HPI-I search header.
        /// </summary>
        /// <returns>The product type object.</returns>
        public static ProductType GetProduct()
        {
            // Set up client product information (PCIN)
            // Values below should be provided by Medicare
            ProductType product = new ProductType()
            {
                platform = System.Environment.OSVersion.ToString(),
                productName = ConfigurationManager.AppSettings["IhiProductName"],                              // Provided by Medicare
                productVersion = ConfigurationManager.AppSettings["IhiProductVersion"],
                vendor = new QualifiedId()
                {
                    id = ConfigurationManager.AppSettings["IhiVendorId"],                                      // Provided by Medicare
                    qualifier = ConfigurationManager.AppSettings["IhiVendorQualifier"]
                }
            };
            return product;
        }

        /// <summary>
        /// Gets a QualifiedId object representing the user details for the HPI-I search header.
        /// </summary>
        /// <param name="userDetails">The user details.</param>
        /// <returns>The qualified ID for the user.</returns>
        public static QualifiedId GetUserQualifiedId(UserDetails userDetails)
        {
            QualifiedId qualifiedId;
            switch (userDetails.Role)
            {
                case UserRole.ProviderIndividual:
                    qualifiedId = new QualifiedId()
                    {
                        id = userDetails.HpiI,
                        qualifier = ConfigurationManager.AppSettings["IhiUserQualifierProviderIndividual"].ToString()
                    };
                    break;

                case UserRole.InteractiveUser:
                    qualifiedId = new QualifiedId()
                    {
                        id = userDetails.Login,
                        qualifier = string.Format(ConfigurationManager.AppSettings["IhiUserQualifierHiUser"].ToString(), userDetails.Domain)
                    };
                    break;

                case UserRole.AuthorisedEmployee:
                    qualifiedId = new QualifiedId()
                    {
                        id = userDetails.AuthorisedEmployeeUserId,
                        qualifier = ConfigurationManager.AppSettings["IhiUserQualifierAuthorisedEmployee"].ToString()
                    };
                    break;

                default:
                    throw new Exception("Unhandled User Role");
            }
            return qualifiedId;
        }

        /// <summary>
        /// Gets the endpoint URL for the HI Service web services.
        /// </summary>
        /// <returns>The endpoint URL for the HI Service web services.</returns>
        public static Uri HiServiceUrl()
        {
            string url = ConfigurationManager.AppSettings["HiServiceUrl"];
            if (string.IsNullOrEmpty(url))
            {
                throw new Exception("HI Service Url has not been defined in the App.Config");
            }
            return new Uri(url);
        }

        /// <summary>
        /// Inserts an audit record for an invocation of the HPI-I search web services.
        /// There are four operations that can be audited: identifier query, demographic
        /// query, batch submit or batch retrieve.
        /// </summary>
        /// <param name="user">The user who initiated the action.</param>
        /// <param name="soapMessages">Full outgoing and incoming SOAP messages.</param>
        /// <param name="accessingOrganisation">The accessing organisation.</param>
        /// <param name="hpiiIdentifierQuery">The HPI-I identifier query request.</param>
        /// <param name="hpiiDemographicQuery">The HPI-I demographic query request.</param>
        /// <param name="hpiiQueryResponse">The HPI-I query response.</param>
        /// <param name="hpiiBatchAsyncSubmitResponse">The HPI-I batch asynchronous submission response.</param>
        /// <param name="hpiiBatchAsyncRetrieveResponse">The HPI-I batch asynchronous retrieval response.</param>
        /// <returns>Whether the audit record was inserted</returns>
        public static bool InsertAudit(
            UserDetails user,
            HIEndpointProcessor.SoapMessages soapMessages,
            HealthProviderOrganisation accessingOrganisation,
            HpiiIdentifierQuery hpiiIdentifierQuery = null,
            HpiiDemographicQuery hpiiDemographicQuery = null,
            HpiiQueryResponse hpiiQueryResponse = null,
            HpiiBatchAsyncSubmitResponse hpiiBatchAsyncSubmitResponse = null,
            HpiiBatchAsyncRetrieveResponse hpiiBatchAsyncRetrieveResponse = null)
        {
            HpiiLookupAudit audit = new HpiiLookupAudit();
            audit.HpiO = accessingOrganisation.Hpio;

            if (hpiiBatchAsyncSubmitResponse != null)
            {
                audit.FamilyName = "x";

                if (!string.IsNullOrEmpty(hpiiBatchAsyncSubmitResponse.BatchIdentifier))
                {
                    audit.BatchIdentifier = hpiiBatchAsyncSubmitResponse.BatchIdentifier;                
                }                
                
                if (hpiiBatchAsyncSubmitResponse.ServiceMessagesType != null && hpiiBatchAsyncSubmitResponse.ServiceMessagesType.serviceMessage != null)
                {
                    string msg = hpiiBatchAsyncSubmitResponse.ServiceMessagesType.serviceMessage[0].code + ":" + hpiiBatchAsyncSubmitResponse.ServiceMessagesType.serviceMessage[0].reason;
                    audit.Message = msg;
                }
            }

            if (hpiiBatchAsyncRetrieveResponse != null)
            {
                audit.FamilyName = "x";

                if (!string.IsNullOrEmpty(hpiiBatchAsyncRetrieveResponse.BatchIdentifier))
                {
                    audit.BatchIdentifier = hpiiBatchAsyncRetrieveResponse.BatchIdentifier;
                }

                if (hpiiBatchAsyncRetrieveResponse.ServiceMessagesType != null && hpiiBatchAsyncRetrieveResponse.ServiceMessagesType.serviceMessage != null)
                {
                    string msg = hpiiBatchAsyncRetrieveResponse.ServiceMessagesType.serviceMessage[0].code + ":" + hpiiBatchAsyncRetrieveResponse.ServiceMessagesType.serviceMessage[0].reason;
                    audit.Message = msg;
                }
            }

            if (hpiiQueryResponse != null)
            {
                audit.HpiiNumber = hpiiQueryResponse.HpiiNumber;

                if (CheckHpiiStatus(hpiiQueryResponse.HpiiStatus))
                {
                    audit.HpiiStatusId = (int)hpiiQueryResponse.HpiiStatus;
                }
                else
                {
                    audit.HpiiStatusId = (int)HpiiStatus.Undefined;
                }

                audit.RegistrationId = hpiiQueryResponse.RegistrationId;
                if (hpiiQueryResponse.FamilyName == null)
                {
                    if (hpiiIdentifierQuery != null)
                    {
                        if (hpiiIdentifierQuery.FamilyName != null)
                        {
                            audit.FamilyName = hpiiIdentifierQuery.FamilyName;
                        }
                        else
                        {
                            audit.FamilyName = "Unspecified";
                        }
                    }
                    else if (hpiiDemographicQuery != null)
                    {
                        if (hpiiDemographicQuery.FamilyName != null)
                        {
                            audit.FamilyName = hpiiDemographicQuery.FamilyName;
                        }
                        else
                        {
                            audit.FamilyName = "Unspecified";
                        }
                    }
                }
                else
                {
                    audit.FamilyName = hpiiQueryResponse.FamilyName;
                }

                if (hpiiQueryResponse.GivenName != null)
                {
                    for (int i = 0; i < hpiiQueryResponse.GivenName.Count(); i++)
                    {
                        if (!hpiiQueryResponse.GivenName[i].IsNullOrEmptyWhitespace())
                        {
                            if (!audit.GivenNames.IsNullOrEmptyWhitespace())
                            {
                                audit.GivenNames = audit.GivenNames + ", ";
                            }
                            audit.GivenNames = audit.GivenNames + hpiiQueryResponse.GivenName[i].ToString();
                        }
                    }
                }

                audit.DateOfBirth = hpiiQueryResponse.DateofBirth;

                audit.Sex = hpiiQueryResponse.Sex.ToString();

                audit.AustralianState = hpiiQueryResponse.State.ToString();

                if (hpiiQueryResponse.AustralianAddress != null)
                {
                    if (!hpiiQueryResponse.AustralianAddress.PostCode.IsNullOrEmptyWhitespace())
                    {
                        audit.Postcode = hpiiQueryResponse.AustralianAddress.PostCode;
                    }
                    else
                    {
                        audit.Postcode = hpiiQueryResponse.PostCode;
                    }
                }
                else
                {
                    audit.Postcode = hpiiQueryResponse.PostCode;
                }

                if (hpiiQueryResponse.InternationalAddress != null)
                {
                    var e = hpiiQueryResponse.InternationalAddress.Country;
                    
                    // Get the Type of the enum
                    var type =  e.GetType();

                    // Get the FieldInfo for the member field with the enums name
                    var info = type.GetField(e.ToString());

                    // Get the XmlEnumAttribute
                    object[] o = info.GetCustomAttributes(typeof(System.Xml.Serialization.XmlEnumAttribute), false);
                    System.Xml.Serialization.XmlEnumAttribute attribute = (System.Xml.Serialization.XmlEnumAttribute)o[0];

                    audit.CountryCode = attribute.Name;
                }

                if (hpiiQueryResponse.HipsResponse.Status != HipsResponseIndicator.OK || hpiiQueryResponse.HipsResponse.HipsErrorMessage != null)
                {
                    string serviceMessage = string.Format("{0} {1} {2} {3} {4}", hpiiQueryResponse.HipsResponse.Status, hpiiQueryResponse.HipsResponse.HipsErrorMessage, hpiiQueryResponse.HipsResponse.ResponseCode, hpiiQueryResponse.HipsResponse.ResponseCodeDescription, hpiiQueryResponse.HipsResponse.ResponseCodeDetails).Trim();
                    audit.Message = serviceMessage;
                }

                if (!user.HpiI.IsNullOrEmptyWhitespace())
                {
                    audit.OperatorHpiI = user.HpiI;
                }
            }

            if (!string.IsNullOrEmpty(user.Login))
            {
                audit.Operator = user.Domain + @"/" + user.Login + "(" + user.Name + ")";
                audit.UserModified = user.Domain + @"/" + user.Login;
            }
            else
            {
                audit.Operator = user.Name;
                audit.UserModified = user.Name;
            }

            audit.HiWebServiceName = HpiiAuditOperationName.ProviderIndividualSearch;
            audit.HiWebServiceVersion = "1";
            audit.ServiceMessageRequestID = soapMessages.SoapRequestMessageId;
            audit.ServiceMessageResponseID = soapMessages.SoapResponseMessageId;

            audit.Request = soapMessages.SoapRequest;
            audit.Response = soapMessages.SoapResponse;

            HpiiLookupAuditDl dataAccess = new HpiiLookupAuditDl(user);
            using (new TransactionScope(TransactionScopeOption.Suppress))
            {
                return dataAccess.Insert(audit);
            }
        }

        /// <summary>
        /// Inserts the HPI-I qualifier in front of the HPI-I number, if not already present, to form a fully qualified URI.
        /// </summary>
        /// <param name="hpii">The hpii.</param>
        /// <returns>The fully qualified URI that represents the HPI-I.</returns>
        public static string InsertHpiiIdentifier(string hpii)
        {
            string result = hpii;
            if (hpii != null && !hpii.StartsWith(HIQualifiers.HPIIQualifier))
            {
                result = HIQualifiers.HPIIQualifier + hpii;
            }

            return result;
        }

        /// <summary>
        /// Maps the HPI-I search request data to a searchForProviderIndividual from a HpiiDemographicQuery
        /// </summary>
        /// <param name="request">The request.</param>
        /// <returns>A search for provider individual object that can be submitted to an HI Service web service.</returns>
        public static searchForProviderIndividual MapHpiiDemographicRequestData(HpiiDemographicQuery request)
        {
            searchForProviderIndividual svcRequest = new searchForProviderIndividual();

            svcRequest.familyName = request.FamilyName;

            if (request.OnlyNameIndicator)
            {
                svcRequest.onlyNameIndicator = true;
                svcRequest.onlyNameIndicatorSpecified = true;
                svcRequest.givenName = null;
            }
            else
            {
                if (request.GivenName != null && request.GivenName.Count() > 0)
                {
                    svcRequest.givenName = request.GivenName;
                }
            }

            if (request.DateofBirth != null)
            {
                svcRequest.dateOfBirthSpecified = true;
                svcRequest.dateOfBirth = (DateTime)request.DateofBirth;
            }

            if (request.Sex != null)
            {
                svcRequest.sexSpecified = true;
                svcRequest.sex = (SexType)request.Sex;
            }

            if (request.AustralianAddress != null)
            {
                svcRequest.searchAustralianAddress = new SearchAustralianAddressType();
                if (!string.IsNullOrEmpty(request.AustralianAddress.UnitNumber) && request.AustralianAddress.UnitType != null)
                {
                    svcRequest.searchAustralianAddress.unitGroup = new UnitGroupType();
                    svcRequest.searchAustralianAddress.unitGroup.unitNumber = request.AustralianAddress.UnitNumber;
                    svcRequest.searchAustralianAddress.unitGroup.unitType = (UnitType)request.AustralianAddress.UnitType;
                }

                if (!string.IsNullOrEmpty(request.AustralianAddress.PostalDeliveryNumber) && request.AustralianAddress.PostalDeliveryType != null)
                {
                    svcRequest.searchAustralianAddress.postalDeliveryGroup = new PostalDeliveryGroupType();
                    svcRequest.searchAustralianAddress.postalDeliveryGroup.postalDeliveryNumber = request.AustralianAddress.PostalDeliveryNumber;
                    svcRequest.searchAustralianAddress.postalDeliveryGroup.postalDeliveryType = (PostalDeliveryType)request.AustralianAddress.PostalDeliveryType;
                }

                if (!string.IsNullOrEmpty(request.AustralianAddress.LevelNumber) && request.AustralianAddress.LevelType != null)
                {
                    svcRequest.searchAustralianAddress.levelGroup = new LevelGroupType();
                    svcRequest.searchAustralianAddress.levelGroup.levelNumber = request.AustralianAddress.LevelNumber;
                    svcRequest.searchAustralianAddress.levelGroup.levelType = (LevelType)request.AustralianAddress.LevelType;
                }

                svcRequest.searchAustralianAddress.addressSiteName = request.AustralianAddress.SiteName;

                svcRequest.searchAustralianAddress.lotNumber = request.AustralianAddress.LotNumber;

                svcRequest.searchAustralianAddress.streetNumber = request.AustralianAddress.StreetNumber;
                svcRequest.searchAustralianAddress.streetName = request.AustralianAddress.StreetName;

                if (request.AustralianAddress.StreetType != null)
                {
                    svcRequest.searchAustralianAddress.streetTypeSpecified = true;
                    svcRequest.searchAustralianAddress.streetType = (StreetType)request.AustralianAddress.StreetType;
                }

                if (request.AustralianAddress.StreetSuffixType != null)
                {
                    svcRequest.searchAustralianAddress.streetSuffixSpecified = true;
                    svcRequest.searchAustralianAddress.streetSuffix = (StreetSuffixType)request.AustralianAddress.StreetSuffixType;
                }

                svcRequest.searchAustralianAddress.suburb = request.AustralianAddress.Suburb;
                if (request.AustralianAddress.State != null)
                {
                    svcRequest.searchAustralianAddress.state = (StateType)request.AustralianAddress.State;
                }

                svcRequest.searchAustralianAddress.postcode = request.AustralianAddress.PostCode;
            }

            svcRequest.searchInternationalAddress = null;
            if (request.InternationalAddress != null)
            {
                svcRequest.searchInternationalAddress = new SearchInternationalAddressType();
                if (request.InternationalAddress.Country != null)
                {
                    svcRequest.searchInternationalAddress.country = (CountryType)request.InternationalAddress.Country;
                }

                svcRequest.searchInternationalAddress.internationalAddressLine = request.InternationalAddress.InternationalAddressLine;
                svcRequest.searchInternationalAddress.internationalStateProvince = request.InternationalAddress.InternationalStateProvince;
                svcRequest.searchInternationalAddress.internationalPostcode = request.InternationalAddress.InternationalPostcode;
            }

            return svcRequest;
        }

        /// <summary>
        /// Maps the HPI-I search request data to a searchForProviderIndividual from a HpiiIdentifierQuery
        /// </summary>
        /// <param name="request">The request.</param>
        /// <returns>A search for provider individual object that can be submitted to an HI Service web service.</returns>
        public static searchForProviderIndividual MapHpiiIdentifierRequestData(HpiiIdentifierQuery request)
        {
            searchForProviderIndividual svcRequest = new searchForProviderIndividual();

            svcRequest.hpiiNumber = InsertHpiiIdentifier(request.HpiiNumber);
            svcRequest.registrationId = request.RegistrationId;
            svcRequest.familyName = request.FamilyName;

            if (request.OnlyNameIndicator)
            {
                svcRequest.onlyNameIndicator = true;
                svcRequest.onlyNameIndicatorSpecified = true;
                svcRequest.givenName = null;
            }
            else
            {
                svcRequest.givenName = request.GivenName;
            }

            if (request.DateofBirth != null)
            {
                svcRequest.dateOfBirthSpecified = true;
                svcRequest.dateOfBirth = (DateTime)request.DateofBirth;
            }

            if (request.Sex != null)
            {
                svcRequest.sexSpecified = true;
                svcRequest.sex = (SexType)request.Sex;
            }

            if (request.State != null)
            {
                svcRequest.stateSpecified = true;
                svcRequest.state = (StateType)request.State;
            }

            svcRequest.postcode = request.PostCode;

            return svcRequest;
        }

        /// <summary>
        /// Maps the hpii response data to a HpiiQueryResponse from a searchForProviderIndividualResponse.
        /// </summary>
        /// <param name="result">The result of the search for a provider individual.</param>
        /// <param name="accessingOrganisation">The accessing organisation.</param>
        /// <returns>The HIPS schema object for the HPI-I query response.</returns>
        public static HpiiQueryResponse MapHpiiResponseData(searchForProviderIndividualResult result, HealthProviderOrganisation accessingOrganisation)
        {
            HpiiQueryResponse response = new HpiiQueryResponse();

            response.HpioNumber = accessingOrganisation.Hpio;
            response.HpiiNumber = ExtractHpii(result.hpiiNumber);
            response.RegistrationId = result.registrationId;
            response.HpiiStatus = GetHpiiStatusValue(result.status);
            response.FamilyName = result.familyName;
            response.OnlyNameIndicator = result.onlyNameIndicator;
            response.GivenName = result.givenName;
            response.PostCode = result.postcode;
            if (result.dateOfBirthSpecified)
            {
                response.DateofBirth = result.dateOfBirth;
            }
            if (result.sexSpecified)
            {
                response.Sex = result.sex;
            }
            if (result.stateSpecified)
            {
                response.State = result.state;
            }
            if (result.australianAddress != null)
            {
                AustralianAddress austAddress = new AustralianAddress();
                if (result.australianAddress.unitGroup != null)
                {
                    austAddress.UnitType = result.australianAddress.unitGroup.unitType;
                    austAddress.UnitNumber = result.australianAddress.unitGroup.unitNumber;
                }
                austAddress.SiteName = result.australianAddress.addressSiteName;
                if (result.australianAddress.levelGroup != null)
                {
                    austAddress.LevelType = result.australianAddress.levelGroup.levelType;
                    austAddress.LevelNumber = result.australianAddress.levelGroup.levelNumber;
                }
                austAddress.StreetNumber = result.australianAddress.streetNumber;
                austAddress.LotNumber = result.australianAddress.lotNumber;
                austAddress.StreetName = result.australianAddress.streetName;

                if (result.australianAddress.streetSuffixSpecified)
                {
                    austAddress.StreetSuffixType = result.australianAddress.streetSuffix;
                }
                if (result.australianAddress.streetTypeSpecified)
                {
                    austAddress.StreetType = result.australianAddress.streetType;
                }
                if (result.australianAddress.postalDeliveryGroup != null)
                {
                    austAddress.PostalDeliveryType = result.australianAddress.postalDeliveryGroup.postalDeliveryType;
                    austAddress.PostalDeliveryNumber = result.australianAddress.postalDeliveryGroup.postalDeliveryNumber;
                }
                austAddress.Suburb = result.australianAddress.suburb;
                austAddress.State = result.australianAddress.state;
                austAddress.PostCode = result.australianAddress.postcode;

                response.AustralianAddress = austAddress;
            }

            if (result.internationalAddress != null)
            {
                InternationalAddress intAddress = new InternationalAddress();
                intAddress.InternationalAddressLine = result.internationalAddress.internationalAddressLine;
                intAddress.InternationalStateProvince = result.internationalAddress.internationalStateProvince;
                intAddress.InternationalPostcode = result.internationalAddress.internationalPostcode;
                intAddress.Country = result.internationalAddress.country;

                response.InternationalAddress = intAddress;
            }

            response.ServiceMessagesType = result.serviceMessages;

            response.HipsResponse.Status = HipsResponseIndicator.OK;

            // Check errors for correct HI
            if (result.serviceMessages != null)
            {
                if (result.serviceMessages.serviceMessage[0].code == "WSE0035")
                {
                    response.HipsResponse.HipsErrorMessage = "No Data Found";
                    response.HipsResponse.ResponseCode = result.serviceMessages.serviceMessage[0].code;
                    response.HipsResponse.ResponseCodeDetails = result.serviceMessages.serviceMessage[0].reason;
                }
            }

            return response;
        }

        /// <summary>
        /// Formats an XML document with indenting.
        /// </summary>
        /// <param name="xml">The XML document.</param>
        /// <returns>The formatted XML document.</returns>
        public static string XmlViewModel(string xml)
        {
            XmlDocument document = new XmlDocument();
            document.Load(new StringReader(xml));

            StringBuilder builder = new StringBuilder();
            using (XmlTextWriter writer = new XmlTextWriter(new StringWriter(builder)))
            {
                writer.Formatting = Formatting.Indented;
                document.Save(writer);
            }

            return builder.ToString();
        }

        /// <summary>
        /// Checks that the HPI-I status is a valid value.
        /// </summary>
        /// <param name="status">The HPI-I status value.</param>
        /// <returns>True if the status is active, deactivated, retired or undefined, otherwise false.</returns>
        private static bool CheckHpiiStatus(HpiiStatus status)
        {
            if (status == HpiiStatus.Active ||
                status == HpiiStatus.Deactivated ||
                status == HpiiStatus.Retired ||
                status == HpiiStatus.Undefined)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// Gets the HPI-I status value from the string passed in back to an enumeration used in code.
        /// </summary>
        /// <param name="value">The status value.</param>
        /// <returns>The status enumeration member.</returns>
        private static HpiiStatus GetHpiiStatusValue(string value)
        {
            if (value == null)
            {
                return HpiiStatus.Undefined;
            }
            else if (value.Equals("A"))
            {
                // ACTIVE
                return HpiiStatus.Active;
            }
            else if (value.Equals("D"))
            {
                // DEACTIVATED
                return HpiiStatus.Deactivated;
            }
            else if (value.Equals("R"))
            {
                // RETIRED
                return HpiiStatus.Retired;
            }
            else
            {
                // UNDEFINED
                return HpiiStatus.Undefined;
            }
        }
    }
}