﻿using System;
using System.Collections.Generic;
using System.Linq;

using AutoMapper;
using DatabaseSchemas = HIPS.PcehrDataStore.Schemas;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Address entity.
    /// </summary>
    internal class AddressProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Database --> Model

            this.CreateMap<DatabaseSchemas.Schemas.Address, ModelSchemas.Cda.Address>()
                .ForMember(dest => dest.AddressPurpose, opt => opt.ResolveUsing((DatabaseSchemas.Schemas.Address src) =>
                    {
                        switch (src.AddressTypeDescription)
                        {
                            case "Home": return ModelSchemas.Cda.AddressPurpose.Residential;
                            case "Business": return ModelSchemas.Cda.AddressPurpose.Business;
                            case "Temporary": return ModelSchemas.Cda.AddressPurpose.Temporary;
                            case "Mailing": return ModelSchemas.Cda.AddressPurpose.Mailing;
                            case "Financial": return ModelSchemas.Cda.AddressPurpose.Unknown;
                            case "Residential": return ModelSchemas.Cda.AddressPurpose.Residential;
                        }
                        return ModelSchemas.Cda.AddressPurpose.Unknown;
                    }))
                .ForMember(dest => dest.AustralianState, opt => opt.ResolveUsing((DatabaseSchemas.Schemas.Address src) =>
                    {
                        switch (src.AustralianStateCode)
                        {
                            case "ACT": return ModelSchemas.Cda.AustralianState.ACT;
                            case "NSW": return ModelSchemas.Cda.AustralianState.NSW;
                            case "NT": return ModelSchemas.Cda.AustralianState.NT;
                            case "QLD": return ModelSchemas.Cda.AustralianState.QLD;
                            case "SA": return ModelSchemas.Cda.AustralianState.SA;
                            case "TAS": return ModelSchemas.Cda.AustralianState.TAS;
                            case "VIC": return ModelSchemas.Cda.AustralianState.VIC;
                            case "WA": return ModelSchemas.Cda.AustralianState.WA;
                        }
                        return ModelSchemas.Cda.AustralianState.U;
                    }))
                .ForMember(dest => dest.Postcode, opt => opt.MapFrom(src => src.PostCode));

            // Model --> NEHTA

            this.CreateMap<ModelSchemas.Cda.Address, NehtaSchemas.IAddress>()
                .ConstructUsing((ResolutionContext ctx) => NehtaSchemas.Common.BaseCDAModel.CreateAddress())
                .ForMember(dest => dest.AddressPurpose, opt => opt.ResolveUsing((ModelSchemas.Cda.Address src) =>
                    {
                        switch (src.AddressPurpose)
                        {
                            case ModelSchemas.Cda.AddressPurpose.Business:
                                return NehtaSchemas.Common.Enums.AddressPurpose.Business;

                            case ModelSchemas.Cda.AddressPurpose.Mailing:
                                return NehtaSchemas.Common.Enums.AddressPurpose.MailingOrPostal;

                            case ModelSchemas.Cda.AddressPurpose.Residential:
                                return NehtaSchemas.Common.Enums.AddressPurpose.Residential;

                            case ModelSchemas.Cda.AddressPurpose.Temporary:
                                return NehtaSchemas.Common.Enums.AddressPurpose.TemporaryAccommodation;

                            case ModelSchemas.Cda.AddressPurpose.Unknown:
                                return NehtaSchemas.Common.Enums.AddressPurpose.NotStatedUnknownInadequatelyDescribed;

                            default:
                                return NehtaSchemas.Common.Enums.AddressPurpose.Undefined;
                        }
                    }))
                .ForMember(
                    dest => dest.AustralianAddress,
                    opt => opt.MapFrom(src => string.Equals(src.CountryName, HIPS.CommonSchemas.Cda.Constants.CountryName.AUSTRALIA, StringComparison.CurrentCultureIgnoreCase) ? src : null))
                .ForMember(
                    dest => dest.InternationalAddress,
                    opt => opt.MapFrom(src => !string.Equals(src.CountryName, HIPS.CommonSchemas.Cda.Constants.CountryName.AUSTRALIA, StringComparison.CurrentCultureIgnoreCase) ? src : null));

            this.CreateMap<ModelSchemas.Cda.Address, NehtaSchemas.SCSModel.Common.AustralianAddress>()
                .ForMember(dest => dest.SuburbTownLocality, opt => opt.MapFrom(src => src.PlaceName))
                .ForMember(dest => dest.PostCode, opt => opt.MapFrom(src => src.Postcode))
                .ForMember(dest => dest.State, opt => opt.ResolveUsing((ModelSchemas.Cda.Address src) =>
                {
                    switch (src.AustralianState)
                    {
                        case ModelSchemas.Cda.AustralianState.ACT:
                            return NehtaSchemas.Common.Enums.AustralianState.ACT;

                        case ModelSchemas.Cda.AustralianState.NSW:
                            return NehtaSchemas.Common.Enums.AustralianState.NSW;

                        case ModelSchemas.Cda.AustralianState.NT:
                            return NehtaSchemas.Common.Enums.AustralianState.NT;

                        case ModelSchemas.Cda.AustralianState.QLD:
                            return NehtaSchemas.Common.Enums.AustralianState.QLD;

                        case ModelSchemas.Cda.AustralianState.SA:
                            return NehtaSchemas.Common.Enums.AustralianState.SA;

                        case ModelSchemas.Cda.AustralianState.TAS:
                            return NehtaSchemas.Common.Enums.AustralianState.TAS;

                        case ModelSchemas.Cda.AustralianState.VIC:
                            return NehtaSchemas.Common.Enums.AustralianState.VIC;

                        case ModelSchemas.Cda.AustralianState.WA:
                            return NehtaSchemas.Common.Enums.AustralianState.WA;
                    }
                    return NehtaSchemas.Common.Enums.AustralianState.Undefined;
                }))
                .ForMember(
                    dest => dest.UnstructuredAddressLines,
                    opt => opt.MapFrom(src => new List<string>
                    {
                        src.AddressLine1,
                        src.AddressLine2
                    }.Where(a => !string.IsNullOrEmpty(a))));

            this.CreateMap<ModelSchemas.Cda.Address, NehtaSchemas.SCSModel.Common.InternationalAddress>()
                .ForMember(dest => dest.Country, opt => opt.ResolveUsing((ModelSchemas.Cda.Address src) =>
                    {
                        foreach (NehtaSchemas.Common.Enums.Country country in typeof(NehtaSchemas.Common.Enums.Country).GetEnumValues())
                        {
                            string name = Nehta.VendorLibrary.Common.Extensions.GetAttributeValue<NehtaSchemas.Common.NameAttribute, string>(country, m => m.Name);
                            if (string.Equals(name, src.CountryName.Trim(), StringComparison.CurrentCultureIgnoreCase))
                            {
                                return country;
                            }
                        }
                        return NehtaSchemas.Common.Enums.Country.Undefined;
                    }))
                .ForMember(dest => dest.PostCode, opt => opt.MapFrom(src => src.Postcode))
                .ForMember(dest => dest.StateProvince, opt => opt.MapFrom(src => src.InternationalStateCode))
                .ForMember(
                    dest => dest.AddressLine,
                    opt => opt.MapFrom(src => new List<string>
                    {
                        src.AddressLine1,
                        src.AddressLine2
                    }.Where(a => !string.IsNullOrEmpty(a))));

            // Database --> NEHTA  (using ConvertUsing to combine the two maps above)
            this.CreateMap<DatabaseSchemas.Schemas.Address, NehtaSchemas.IAddress>()
                .ConvertUsing(src => ObjectMapper.Map<NehtaSchemas.IAddress>(ObjectMapper.Map<ModelSchemas.Cda.Address>(src)));
        }

        #endregion Methods
    }
}