﻿using System;
using System.Collections.Generic;
using System.Linq;

using AutoMapper;
using DatabaseSchemas = HIPS.PcehrDataStore.Schemas;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the SubjectOfCare (patient) entity in a CDA document.
    /// </summary>
    internal class SubjectOfCareProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Database --> Model
            //
            // The database schema has given names joined by spaces, so they will be split apart.
            //
            // The database schema has a single Suffix and Title which will be an empty string
            // rather than null if not known. This empty string needs to be replaced by an empty
            // list, and must not be a list with one element containing an empty string!
            //
            // The database schema has NameType which differentiates the registered name (legal)
            // from the current name in PAS (preferred) and previous names. The registered name
            // will be mapped to RegisteredName, all others to OtherName.

            this.CreateMap<DatabaseSchemas.PatientMasterName, ModelSchemas.Cda.IndividualName>()
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))

                .ForMember(
                    dest => dest.GivenNames,
                    opt => opt.MapFrom(src => string.IsNullOrWhiteSpace(src.GivenNames)
                        ? new string[0]
                        : src.GivenNames.Split(' ')))
                .ForMember(
                    dest => dest.Suffixes,
                    opt => opt.MapFrom(src => string.IsNullOrWhiteSpace(src.Suffix)
                        ? new string[0]
                        : new string[] { src.Suffix }))
                .ForMember(
                    dest => dest.Titles,
                    opt => opt.MapFrom(src => string.IsNullOrWhiteSpace(src.Title)
                        ? new string[0]
                        : new string[] { src.Title }))
                .ForMember(dest => dest.Usage, opt => opt.ResolveUsing((DatabaseSchemas.PatientMasterName src) =>
                {
                    if (src.NameTypeId == (int)HIPS.Base.Schemas.Enumerators.NameTypes.Legal)
                    {
                        return ModelSchemas.Cda.IndividualNameUsage.RegisteredName;
                    }
                    return ModelSchemas.Cda.IndividualNameUsage.OtherName;
                }));

            // Model --> NEHTA

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationSubjectOfCare>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateSubjectOfCare())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.ISubjectOfCare>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateParticipantForSubjectOfCare())
                .ForMember(dest => dest.Person, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Addresses, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.Address>()
                    {
                        src.Metadata.PatientAddress
                    }))
                .ForMember(
                    dest => dest.ElectronicCommunicationDetails,
                    opt => opt.MapFrom(src => src.Metadata.PatientContactDetails == null ? null : new List<ModelSchemas.Cda.ElectronicCommunicationDetail>()
                    {
                        src.Metadata.PatientContactDetails
                    }))
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.ResolveUsing((ModelSchemas.Cda.CdaDocumentDetails src) => Guid.NewGuid()));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.IPersonSubjectOfCare>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreatePersonForSubjectOfCare())
                .ForMember(dest => dest.DateOfBirth, opt => opt.MapFrom(src => src.PatientMaster.RegisteredDateOfBirth))
                .ForMember(dest => dest.DateOfBirthInludesTime, opt => opt.MapFrom(src => false))
                .ForMember(dest => dest.Gender, opt => opt.MapFrom(src => (DatabaseSchemas.Enumerators.SexEnumerator)src.PatientMaster.CurrentSexId))
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.EntityIdentifier>()
                    {
                        new ModelSchemas.Cda.EntityIdentifier
                        {
                            AssigningAuthorityName = "IHI",
                            AssigningGeographicArea = "National Identifier",
                            Root = string.Format("1.2.36.1.2001.1003.0.{0}", src.PatientMaster.Ihi)
                        },
                        new ModelSchemas.Cda.EntityIdentifier
                        {
                            AssigningAuthorityName = src.Hospital.Name,
                            Root = string.Format("1.2.36.1.2001.1005.49.3.{0}.{1}", src.Hospital.HpiO, src.Hospital.HospitalId),
                            Extension = src.HospitalPatient.Mrn,
                            Code = new ModelSchemas.Cda.CdaCode
                            {
                                Code = "MR",
                                CodeSystem = "2.16.840.1.113883.12.203",
                                CodeSystemName = "Identifier Type (HL7)"
                            }
                        },
                        new ModelSchemas.Cda.EntityIdentifier
                        {
                            AssigningAuthorityName = "Medicare card number",
                            Root = string.IsNullOrEmpty(src.PatientMaster.MedicareIrn)
                                ? "1.2.36.1.5001.1.0.7.1"
                                : "1.2.36.1.5001.1.0.7",
                            Extension = src.PatientMaster.MedicareNumber + src.PatientMaster.MedicareIrn,
                            Code = new ModelSchemas.Cda.CdaCode
                            {
                                Code = "MC",
                                CodeSystem = "2.16.840.1.113883.12.203",
                                CodeSystemName = "Identifier Type (HL7)"
                            }
                        },
                    }.Where(a => a.AssigningAuthorityName == "IHI" || !string.IsNullOrEmpty(a.Extension))))
            .ForMember(dest => dest.PersonNames, opt => opt.MapFrom(src => src.PatientNames));

            this.CreateMap<ModelSchemas.Cda.IndividualName, NehtaSchemas.IPersonName>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreatePersonName())
                .ForMember(dest => dest.FamilyName, opt => opt.MapFrom(src => src.FamilyName))
                .ForMember(dest => dest.GivenNames, opt => opt.MapFrom(src => src.GivenNames))
                .ForMember(dest => dest.NameSuffix, opt => opt.MapFrom(src => src.Suffixes))
                .ForMember(dest => dest.Titles, opt => opt.MapFrom(src => src.Titles))
                .ForMember(dest => dest.NameUsages, opt => opt.ResolveUsing((ModelSchemas.Cda.IndividualName src) =>
                {
                    var list = new List<NehtaSchemas.Common.Enums.NameUsage>();
                    switch (src.Usage)
                    {
                        case ModelSchemas.Cda.IndividualNameUsage.MaidenName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.MaidenName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.NewbornName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.NewbornName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.OtherName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.OtherName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.ProfessionalName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.ProfessionalOrBusinessName);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.RegisteredName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.Legal);
                            break;

                        case ModelSchemas.Cda.IndividualNameUsage.ReportingName:
                            list.Add(NehtaSchemas.Common.Enums.NameUsage.ReportingName);
                            break;
                    }
                    return list;
                }));
        }

        #endregion Methods
    }
}