﻿using System.Linq;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.CommonBusinessLogic.Singleton
{
    /// <summary>
    /// This is the Hospital Singleton class
    /// </summary>
    public class HospitalSingleton
    {
        #region Private Members

        private static HospitalSingleton valueField;

        //private DateTime? lastUpdated;

        /// <summary>
        /// This dictionary enables the fast lookup of a hospital by one of its identifiers.
        /// The key to looking up a hospital by one of its identifiers is to know the codeSystemId (int) and the identifier itself (string).
        /// Therefore the key of this dictionary is a Tuple of int and string.
        /// </summary>
        //private Dictionary<Tuple<string, string>, Hospital> hospitals;

        #endregion Private Members

        #region Private Constructor

        private HospitalSingleton()
        {
        }

        #endregion Private Constructor

        #region Public Properties

        /// <summary>
        /// Retrieves the shared hospital lookup object.
        /// If a configurable period has elapsed, refreshes the contents.
        /// The refresh interval is the same as ListSingleton, and configured using
        /// the LookupRefreshSeconds item in the app.config file.
        /// </summary>
        public static HospitalSingleton Value
        {
            get
            {
                if (valueField == null)
                {
                    valueField = new HospitalSingleton();
                }
                valueField.Refresh();
                return valueField;
            }
        }

        #endregion Public Properties

        #region Public Methods

        /// <summary>
        /// Finds the hospital which has the specified code (e.g. RAH) in the specified code system (e.g. pasFacCd).
        /// </summary>
        /// <param HL7Name="hospitalCode">A code that identifies the hospital.</param>
        /// <param HL7Name="hospitalCodeSystem">A code that identifies the type of hospital code.</param>
        /// <returns>Hospital object or null if not found</returns>
        public Hospital Find(string hospitalCode, string hospitalCodeSystem)
        {
            ListSingleton lists = ListSingleton.Instance;
            int hospitalId = (from codeSystem in lists.AllCodeSystems
                              join hospitalCodes in lists.AllHospitalCodes
                              on codeSystem.CodeSystemId equals hospitalCodes.CodeSystemId
                              where (codeSystem.Code == hospitalCodeSystem) && (hospitalCodes.Code == hospitalCode)
                              select hospitalCodes.HospitalId).FirstOrDefault();

            Hospital result = (from hospital in lists.AllHospitals
                               where hospital.HospitalId == hospitalId
                               select hospital).FirstOrDefault();

            //int codeSystemId = lists.AllCodeSystems.

            //Tuple<string, string> key = new Tuple<string, string>(code, identifier);
            //Hospital hospital;
            //hospitals.TryGetValue(key, out hospital);
            return result;
        }

        /// <summary>
        /// Finds the hospital which has the specified hospital id.
        /// </summary>
        /// <param name="hospitalId">The primary key of the hospital record in the PCEHR Data Store</param>
        /// <returns>Hospital object or null if not found</returns>
        public Hospital Find(int hospitalId)
        {
            return (from hospital in ListSingleton.Instance.AllHospitals where hospital.HospitalId == hospitalId select hospital).FirstOrDefault();
        }

        /// <summary>
        /// Finds the visitor hospital for the hospital which has the specified code (e.g. RAH) in the specified code system (e.g. pasFacCd).
        /// </summary>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <returns></returns>
        public Hospital FindVisitorHospital(string hospitalCode, string hospitalCodeSystem)
        {
            Hospital hospital;
            Hospital result;

            hospital = this.Find(hospitalCode, hospitalCodeSystem);
            result = this.Find(hospital.VisitorHospitalId);

            return result;
        }

        /// <summary>
        /// Finds the visitor hospital for the hospital which has the specified hospital id.
        /// </summary>
        /// <param name="hospitalId">The hospital identifier.</param>
        /// <returns></returns>
        public Hospital FindVisitorHospital(int hospitalId)
        {
            Hospital hospital;
            Hospital result;

            hospital = this.Find(hospitalId);
            result = this.Find(hospital.VisitorHospitalId);

            return result;
        }

        /// <summary>
        /// Checks whether the hospital list is empty. This typically means the database is unavailable.
        /// </summary>
        /// <returns>True if the hospital list is empty.</returns>
        public bool IsHospitalListEmpty()
        {
            return ListSingleton.Instance.AllHospitals.Count == 0;
        }

        #endregion Public Methods

        #region Private Methods

        private void Refresh()
        {
            ListSingleton lists = ListSingleton.Instance;

            //int refreshInterval = lists.RefreshInterval;
            //if (lastUpdated == null || lastUpdated.Value.AddSeconds(refreshInterval) < DateTime.Now)
            //{
            //    hospitals = new Dictionary<Tuple<string, string>, Hospital>();
            //    foreach (Hospital hospital in lists.AllHospitals)
            //    {
            //        foreach (HospitalCode code in hospital.Codes)
            //        {
            //            foreach (HospitalIdentifierNamespace code in Enum.GetValues(typeof(HospitalIdentifierNamespace)))
            //            {
            //                int codeSystemId = (int)code;

            //                if (codeSystemId == code.CodeSystemId)
            //                {
            //                    Tuple<string, string> key = new Tuple<string, string>(codeSystemId, code.Code);
            //                    hospitals[key] = hospital;
            //                }
            //            }
            //        }
            //    }
            // lastUpdated = DateTime.Now;
            //}
        }

        #endregion Private Methods
    }
}