﻿using System;

namespace HIPS.CommonSchemas.Exceptions
{
    /// <summary>
    /// Exception thrown when a required item was not found.
    /// </summary>
    public class ItemNotFoundException : Exception
    {
        /// <summary>
        /// Initialises a new instance of the <see cref="ItemNotFoundException"/> class.
        /// </summary>
        /// <param name="itemType">The type of item that was not found.</param>
        /// <param name="qualifier">The qualifier that was specified by the caller and was not able to locate the item.</param>
        /// <param name="identifier">The identifier value that was specified by the caller and was not able to locate the item.</param>
        public ItemNotFoundException(ItemType itemType, string qualifier, string identifier)
            : base(string.Format("{0} not found.", itemType))
        {
            this.Type = itemType;
            this.Qualifier = qualifier;
            this.Identifier = identifier;
        }

        /// <summary>
        /// Represents a type of item that was not found.
        /// </summary>
        public enum ItemType
        {
            /// <summary>
            /// The hospital was not found.
            /// </summary>
            Hospital,

            /// <summary>
            /// The patient was not found.
            /// </summary>
            Patient,

            /// <summary>
            /// The episode was not found.
            /// </summary>
            Episode,

            /// <summary>
            /// The document was not found.
            /// </summary>
            Document,

            /// <summary>
            /// The address was not found.
            /// </summary>
            Address
        }

        /// <summary>
        /// Gets the identifier value, if any, that was specified by the caller and was not able to locate the item.
        /// </summary>
        public string Identifier { get; internal set; }

        /// <summary>
        /// Gets the qualifier, if any, that was specified by the caller and was not able to locate the item.
        /// </summary>
        public string Qualifier { get; internal set; }

        /// <summary>
        /// Gets a code representing the type of item that was not found.
        /// Known examples are "Hospital", "Patient", "Episode" or "Address".
        /// </summary>
        public ItemType Type { get; internal set; }
    }
}