﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the HealthProviderIndividual table
    /// </summary>
    public class HealthProviderIndividualDl : BaseDl
    {
        #region Constructors

        /// <summary>
        /// Creates a data access object for the health provider individual.
        /// </summary>
        /// <param name="user">The person responsible for the action.</param>
        public HealthProviderIndividualDl(UserDetails user)
        {
            this.User = user;
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Deletes the specified health care provider.
        /// </summary>
        /// <param name="HealthCareProvider">The health care provider.</param>
        /// <returns></returns>
        public bool Delete(HealthProviderIndividual healthProviderIndividual, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.HealthProviderIndividualDelete", transaction))
            {
                return base.Delete<HealthProviderIndividual>(healthProviderIndividual, command);
            }
        }

        /// <summary>
        /// Gets the specified individual health provider by id.
        /// </summary>
        /// <param name="healthProviderIndividualId">The health provider individual id.</param>
        /// <param name="item">The item.</param>
        /// <param name="command">The command.</param>
        /// <returns></returns>
        public bool Get(int healthProviderIndividualId, out HealthProviderIndividual item, SqlCommand command = null)
        {
            bool result = false;
            item = new HealthProviderIndividual();
            try
            {
                if (command == null)
                {
                    using (SqlCommand newCommand = GetSqlCommand("hips.HealthProviderIndividualGet"))
                    {
                        result = GetIndividualProvider(healthProviderIndividualId, item, result, newCommand);
                        newCommand.Connection.Close();
                    }
                }
                else
                {
                    command.CommandText = "hips.HealthProviderIndividualGet";
                    command.Parameters.Clear();
                    result = GetIndividualProvider(healthProviderIndividualId, item, result, command);
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageHealthCareProviderGet, ex, User, LogMessage.HIPS_MESSAGE_045);
            }
            return result;
        }

        /// <summary>
        /// Gets all for a specific hospital.
        /// </summary>
        /// <param name="hospitalId">The hospital id.</param>
        /// <returns></returns>
        public List<HealthProviderIndividual> GetAll(int hospitalId)
        {
            List<HealthProviderIndividual> results = new List<HealthProviderIndividual>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HealthProviderIndividualGet"))
                {
                    results = GetPopulatedBusinessList<HealthProviderIndividual>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageHealthCareProviderGet, ex, User, LogMessage.HIPS_MESSAGE_043);
            }
            return results;
        }

        /// <summary>
        /// Gets a provider with a given CIS identifier - Not currently implemented.
        /// </summary>
        /// <param name="hospitalId">Hospital id</param>
        /// <param name="cisProviderIdentifier">Identifier of this provider allocated by the Clinical Information System at the hospital</param>
        /// <returns>List of health provider individuals with the given CIS identifier at the given hospital</returns>
        //public bool GetByCisIdentifier(int hospitalId, string cisProviderIdentifier, out HealthProviderIndividual item)
        //{
        //    bool result = false;
        //    item = new HealthProviderIndividual();
        //    try
        //    {
        //        using (SqlCommand command = GetSqlCommand("hips.HealthProviderIndividualGet"))
        //        {
        //            AddIntParameter("@HospitalId", hospitalId, command);
        //            AddStringParameter("@CisProviderIdentifier", cisProviderIdentifier, command);

        //            result = PopulateBusinessObject<HealthProviderIndividual>(command.ExecuteReader(), item);
        //            command.Connection.Close();
        //        }
        //    }
        //    catch (Exception ex)
        //    {
        //        EventLogger.WriteLog(ConstantsResource.ErrorMessageHealthCareProviderGet, ex, User);
        //    }
        //    return result;
        //}

        /// <summary>
        /// Gets a provider with a given PAS identifier.
        /// </summary>
        /// <param name="hospitalId">Hospital id</param>
        /// <param name="cisProviderIdentifier">Identifier of this provider allocated by the Patient Administration System at the hospital</param>
        /// <returns>List of health provider individuals with the given PAS identifier at the given hospital</returns>
        public bool GetByPasIdentifier(int hospitalId, string pasProviderIdentifier, out HealthProviderIndividual item)
        {
            bool result = false;
            item = new HealthProviderIndividual();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.HealthProviderIndividualGet"))
                {
                    AddIntParameter("@HospitalId", hospitalId, command);
                    AddStringParameter("@PasProviderIdentifier", pasProviderIdentifier, command);

                    result = PopulateBusinessObject<HealthProviderIndividual>(command.ExecuteReader(), item);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                EventLogger.WriteLog(ConstantsResource.ErrorMessageHealthCareProviderGet, ex, User, LogMessage.HIPS_MESSAGE_044);
            }
            return result;
        }

        /// <summary>
        /// Saves the specified health care provider.
        /// </summary>
        /// <param name="HealthCareProvider">The health care provider.</param>
        /// <returns></returns>
        public bool Save(HealthProviderIndividual healthProviderIndividual, SqlTransaction transaction = null)
        {
            bool result = false;
            HospitalHealthProviderIndividualDl dataAccess = new HospitalHealthProviderIndividualDl(User);
            if (healthProviderIndividual.HealthProviderIndividualId.HasValue)
            {
                result = Update(healthProviderIndividual, transaction);
            }
            else
            {
                result = Insert(healthProviderIndividual, transaction);
            }

            if (result)
            {
                foreach (HospitalHealthProviderIndividual link in healthProviderIndividual.HospitalHealthProviderIndividuals)
                {
                    if (!link.HealthProviderIndividualId.HasValue)
                    {
                        link.HealthProviderIndividualId = healthProviderIndividual.HealthProviderIndividualId;
                        result = dataAccess.Insert(link, transaction);
                    }
                    if (!result) break;
                }
            }

            return result;
        }

        /// <summary>
        /// Gets the individual provider.
        /// </summary>
        /// <param name="healthProviderIndividualId">The health provider individual id.</param>
        /// <param name="item">The item.</param>
        /// <param name="result">if set to <c>true</c> [result].</param>
        /// <param name="command">The command.</param>
        /// <returns></returns>
        private static bool GetIndividualProvider(int healthProviderIndividualId, HealthProviderIndividual item, bool result, SqlCommand command)
        {
            AddIntParameter("healthProviderIndividualId", healthProviderIndividualId, command);
            result = PopulateBusinessObject<HealthProviderIndividual>(command.ExecuteReader(), item);
            return result;
        }

        /// <summary>
        /// Inserts the specified health care provider.
        /// </summary>
        /// <param name="HealthCareProvider">The health care provider.</param>
        /// <returns></returns>
        private bool Insert(HealthProviderIndividual healthProviderIndividual, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.HealthProviderIndividualInsert", transaction))
            {
                return base.Insert<HealthProviderIndividual>(healthProviderIndividual, command);
            }
        }

        /// <summary>
        /// Updates the specified health care provider.
        /// </summary>
        /// <param name="HealthCareProvider">The health care provider.</param>
        /// <returns></returns>
        private bool Update(HealthProviderIndividual healthProviderIndividual, SqlTransaction transaction)
        {
            using (SqlCommand command = GetSqlCommand("hips.HealthProviderIndividualUpdate", transaction))
            {
                return base.Update<HealthProviderIndividual>(healthProviderIndividual, command);
            }
        }

        #endregion Methods
    }
}